<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Cms\Controller;

use Admin\Controller\AdminActionController;
use Admin\Data\Common;
use Cms\Entity\Cms;
use Cms\Entity\CmsClass;
use Cms\Form\CmsClassForm;
use Cms\Service\CmsClassManager;
use Doctrine\ORM\EntityManager;
use Laminas\Mvc\I18n\Translator;

class CmsClassController extends AdminActionController
{
    private $translator;
    private $entityManager;
    private $cmsClassManager;

    public function __construct(
        Translator $translator,
        EntityManager $entityManager,
        CmsClassManager $cmsClassManager
    )
    {
        $this->translator       = $translator;
        $this->entityManager    = $entityManager;
        $this->cmsClassManager  = $cmsClassManager;
    }

    /**
     * 文章分类列表
     * @return array|\Laminas\View\Model\ViewModel
     */
    public function indexAction()
    {
        $cmsClass = $this->entityManager->getRepository(CmsClass::class)->findBy([], ['cmsClassTopId' => 'ASC', 'cmsClassSort' => 'ASC']);
        $classList= Common::classOptions($cmsClass, 0, 'getCmsClassTopId', 'getCmsClassId');

        $cmsNumList = $this->entityManager->getRepository(CmsClass::class)->findClassCmsNum();
        $cmsNumArray= [];
        if ($cmsNumList) foreach ($cmsNumList as $numValue) {
            $cmsNumArray[$numValue['cmsClassId']] = $numValue['cmsNum'];
        }

        return ['classList' => $classList, 'cmsNumArray' => $cmsNumArray];
    }

    /**
     * 添加文章分类
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addAction()
    {
        $form = new CmsClassForm();
        $form->get('cmsClassTopId')->setValueOptions($this->cmsPlugin()->cmsClassTopListOptions());

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if($form->isValid()) {
                $data = $form->getData();
                $this->cmsClassManager->addCmsClass($data);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('文章分类 %s 添加成功!'), $data['cmsClassName']), $this->translator->translate('文章分类管理'));

                return $this->redirect()->toRoute('cms-class');
            }
        }

        return ['form' => $form];
    }

    /**
     * 编辑文章分类
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editAction()
    {
        $cmsClassId = (int) $this->params()->fromRoute('id', -1);

        $cmsClassInfo = $this->entityManager->getRepository(CmsClass::class)->findOneByCmsClassId($cmsClassId);
        if($cmsClassInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该分类不存在!'));
            return $this->redirect()->toRoute('cms-class');
        }

        $form = new CmsClassForm($this->entityManager, $cmsClassInfo);
        $form->get('cmsClassTopId')->setValueOptions($this->cmsPlugin()->cmsClassTopListOptions());

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if($form->isValid()) {
                $data = $form->getData();
                $this->cmsClassManager->editCmsClass($data, $cmsClassInfo);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('文章分类 %s 编辑成功!'), $data['cmsClassName']), $this->translator->translate('文章分类管理'));

                return $this->redirect()->toRoute('cms-class');
            }
        } else {
            $valueArray = $cmsClassInfo->valuesArray();
            $form->setData($valueArray);
        }

        return ['form' => $form, 'cmsClassInfo' => $cmsClassInfo];
    }

    /**
     * 删除文章分类
     * @return mixed
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteAction()
    {
        if(!$this->adminCommon()->validatorCsrf()) return $this->adminCommon()->toReferer();

        $cmsClassId = (int) $this->params()->fromRoute('id', -1);

        $cmsClassInfo = $this->entityManager->getRepository(CmsClass::class)->findOneByCmsClassId($cmsClassId);
        if($cmsClassInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该分类不存在!'));
            return $this->adminCommon()->toReferer();
        }

        $subClassInfo = $this->entityManager->getRepository(CmsClass::class)->findOneBy(['cmsClassTopId' => $cmsClassId]);
        if($subClassInfo) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该分类下存在分类，不能删除!'));
            return $this->adminCommon()->toReferer();
        }

        $cmsInfo = $this->entityManager->getRepository(Cms::class)->findOneBy(['cmsClassId' => $cmsClassId]);
        if($cmsInfo) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该分类下存在文章，不能删除!'));
            return $this->adminCommon()->toReferer();
        }

        $this->cmsClassManager->deleteCmsClass($cmsClassInfo);

        $this->adminCommon()->addOperLog(sprintf($this->translator->translate('文章分类 %s 删除成功!'), $cmsClassInfo->getCmsClassName()), $this->translator->translate('文章分类管理'));

        return $this->adminCommon()->toReferer();
    }
}