<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Extend\Controller;

use Admin\Controller\AdminActionController;
use Doctrine\ORM\EntityManager;
use Extend\Entity\Navigation;
use Extend\Form\NavigationForm;
use Extend\Service\NavigationManager;
use Laminas\Mvc\I18n\Translator;

class NavigationController extends AdminActionController
{
    private $translator;
    private $entityManager;
    private $navigationManager;

    public function __construct(
        Translator $translator,
        EntityManager $entityManager,
        NavigationManager $navigationManager
    )
    {
        $this->translator       = $translator;
        $this->entityManager    = $entityManager;
        $this->navigationManager= $navigationManager;
    }

    /**
     * 导航列表
     * @return array|\Laminas\View\Model\ViewModel
     */
    public function indexAction()
    {
        $navigationList = $this->entityManager->getRepository(Navigation::class)->findBy([], ['navigationSort' => 'ASC']);

        return ['navigationList' => $navigationList];
    }

    /**
     * 添加导航
     * @return NavigationForm[]|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addAction()
    {
        $form = new NavigationForm();
        $form->get('navigationPlace')->setValueOptions([1 => $this->translator->translate('顶部'), 2 => $this->translator->translate('中部'), 3 => $this->translator->translate('底部')]);
        $form->get('goodsClassId')->setValueOptions($this->goodsPlugin()->classListOptions($this->translator->translate('选择商品分类')));

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->navigationManager->addNavigation($data);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('导航 %s 添加成功!'), $data['navigationTitle']), $this->translator->translate('导航设置'));

                return $this->redirect()->toRoute('navigation');
            }
        }

        return ['form' => $form];
    }

    /**
     * 编辑导航
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editAction()
    {
        $navigationId = (int) $this->params()->fromRoute('id', -1);

        $navigation   = $this->entityManager->getRepository(Navigation::class)->findOneBy(['navigationId' => $navigationId]);
        if ($navigation == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该导航不存在!'));
            return $this->redirect()->toRoute('navigation');
        }

        $form = new NavigationForm();
        $form->get('navigationPlace')->setValueOptions([1 => $this->translator->translate('顶部'), 2 => $this->translator->translate('中部'), 3 => $this->translator->translate('底部')]);
        $form->get('goodsClassId')->setValueOptions($this->goodsPlugin()->classListOptions($this->translator->translate('选择商品分类')));

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->navigationManager->editNavigation($data, $navigation);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('导航 %s 编辑成功!'), $data['navigationTitle']), $this->translator->translate('导航设置'));

                return $this->redirect()->toRoute('navigation');
            }
        } else $form->setData($navigation->valuesArray());

        return ['form' => $form, 'navigation' => $navigation];
    }

    /**
     * 删除导航
     * @return \Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteAction()
    {
        if(!$this->adminCommon()->validatorCsrf()) return $this->adminCommon()->toReferer();

        $navigationId = (int) $this->params()->fromRoute('id', -1);
        $navigation   = $this->entityManager->getRepository(Navigation::class)->findOneBy(['navigationId' => $navigationId]);
        if ($navigation == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该导航不存在!'));
            return $this->redirect()->toRoute('navigation');
        }

        $this->navigationManager->deleteNavigation($navigation);

        $this->adminCommon()->addOperLog(sprintf($this->translator->translate('导航 %s 删除成功!'), $navigation->getNavigationTitle()), $this->translator->translate('导航设置'));

        return $this->adminCommon()->toReferer();
    }
}