<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Goods\Controller;

use Admin\Controller\AdminActionController;
use Admin\Data\Common;
use Doctrine\ORM\EntityManager;
use Goods\Entity\GoodsAttribute;
use Goods\Entity\GoodsAttributeValue;
use Goods\Form\GoodsAttributeForm;
use Goods\Form\GoodsAttributeValueFrom;
use Goods\Service\GoodsAttributeManager;
use Goods\Service\GoodsAttributeValueManager;
use Goods\Service\GoodsInAttributeManager;
use Laminas\Mvc\I18n\Translator;

class GoodsAttributeController extends AdminActionController
{
    private $translator;
    private $entityManager;
    private $goodsAttributeManager;
    private $goodsAttributeValueManager;
    private $goodsInAttributeManager;

    public function __construct(
        Translator $translator,
        EntityManager $entityManager,
        GoodsAttributeManager $goodsAttributeManager,
        GoodsAttributeValueManager $goodsAttributeValueManager,
        GoodsInAttributeManager $goodsInAttributeManager
    )
    {
        $this->translator       = $translator;
        $this->entityManager    = $entityManager;
        $this->goodsAttributeManager = $goodsAttributeManager;
        $this->goodsAttributeValueManager = $goodsAttributeValueManager;
        $this->goodsInAttributeManager = $goodsInAttributeManager;
    }

    /**
     * 商品属性列表
     * @return array|\Laminas\View\Model\ViewModel
     */
    public function indexAction()
    {
        $attributeList = $this->entityManager->getRepository(GoodsAttribute::class)->findAllAttribute();

        return ['attributeList' => $attributeList];
    }

    /**
     * 添加商品属性
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addAction()
    {
        $form = new GoodsAttributeForm();
        $form->get('attributeGroupId')->setValueOptions($this->goodsPlugin()->attributeGroupOptions());
        $form->get('attributeType')->setValueOptions(Common::goodsAttributeType($this->translator));

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if($form->isValid()) {
                $data = $form->getData();
                $this->goodsAttributeManager->addAttribute($data);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('属性 %s 添加成功!'), $data['attributeName']), $this->translator->translate('商品属性'));

                return $this->redirect()->toRoute('goods-attribute');
            }
        }

        return ['form' => $form];
    }

    /**
     * 编辑商品属性
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editAction()
    {
        $attributeId = (int) $this->params()->fromRoute('id', -1);

        $attributeInfo = $this->entityManager->getRepository(GoodsAttribute::class)->findOneByAttributeId($attributeId);
        if($attributeInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该属性不存在!'));
            return $this->redirect()->toRoute('goods-attribute');
        }

        $form = new GoodsAttributeForm('edit');
        $form->get('attributeGroupId')->setValueOptions($this->goodsPlugin()->attributeGroupOptions());
        $form->get('attributeType')->setValueOptions(Common::goodsAttributeType($this->translator));

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if($form->isValid()) {
                $data = $form->getData();
                $this->goodsAttributeManager->editAttribute($data, $attributeInfo);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('属性 %s 编辑成功!'), $data['attributeName']), $this->translator->translate('商品属性'));

                return $this->redirect()->toRoute('goods-attribute');
            }
        } else $form->setData($attributeInfo->valuesArray());

        return ['form' => $form, 'attributeInfo' => $attributeInfo];
    }

    /**
     * 删除商品属性
     * @return \Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteAction()
    {
        if(!$this->adminCommon()->validatorCsrf()) return $this->adminCommon()->toReferer();

        $attributeId = (int) $this->params()->fromRoute('id', -1);

        $attributeInfo = $this->entityManager->getRepository(GoodsAttribute::class)->findOneByAttributeId($attributeId);
        if($attributeInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该属性不存在!'));
            return $this->redirect()->toRoute('goods-attribute');
        }

        $this->goodsAttributeManager->deleteAttribute($attributeInfo);
        $this->goodsAttributeValueManager->deleteMoreAttributeValue([$attributeId]);
        $this->goodsInAttributeManager->deleteGoodsInAttributeId([$attributeId]);

        $this->adminCommon()->addOperLog(sprintf($this->translator->translate('属性 %s 删除成功!'), $attributeInfo->getAttributeName()), $this->translator->translate('商品属性'));

        return $this->adminCommon()->toReferer();
    }

    /**
     * 批量修改商品属性排序
     * @return mixed
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     * @throws \Doctrine\Persistence\Mapping\MappingException
     */
    public function updateAllSortAction()
    {
        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            if(!empty($data['attributeSort'])) {
                switch ($data['attributeMoreUpdate']) {
                    case 'sort':
                        $this->goodsAttributeManager->updateAllAttributeSort($data);
                        $this->adminCommon()->addOperLog($this->translator->translate('批量排序处理成功!'), $this->translator->translate('商品属性'));
                        break;
                    case 'delete':
                        $this->goodsAttributeManager->deleteMoreAttribute($data['select_id']);
                        $this->goodsAttributeValueManager->deleteMoreAttributeValue($data['select_id']);
                        $this->goodsInAttributeManager->deleteGoodsInAttributeId($data['select_id']);
                        $this->adminCommon()->addOperLog($this->translator->translate('属性批量删除处理成功!'), $this->translator->translate('商品属性'));
                        break;
                }
            }
        }

        return $this->adminCommon()->toReferer();
    }

    /**
     * 商品属性值列表
     * @return array|\Laminas\Http\Response
     */
    public function attributeValueAction()
    {
        $attributeId = (int) $this->params()->fromRoute('id', -1);

        $attributeInfo = $this->entityManager->getRepository(GoodsAttribute::class)->findOneByAttributeId($attributeId);
        if($attributeInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该属性不存在!'));
            return $this->redirect()->toRoute('goods-attribute');
        }

        $attributeValueList = $this->entityManager->getRepository(GoodsAttributeValue::class)->findBy(['attributeId' => $attributeId], ['valueSort' => 'ASC']);

        return ['attributeValueList' => $attributeValueList, 'attributeInfo' => $attributeInfo];
    }

    /**
     * 添加商品属性值
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\Common\Persistence\Mapping\MappingException
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addAttributeValueAction()
    {
        $attributeId = (int) $this->params()->fromRoute('id', -1);

        $attributeInfo = $this->entityManager->getRepository(GoodsAttribute::class)->findOneByAttributeId($attributeId);
        if($attributeInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该属性不存在!'));
            return $this->redirect()->toRoute('goods-attribute');
        }

        $form = new GoodsAttributeValueFrom();

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if($form->isValid()) {
                $data = $form->getData();
                $this->goodsAttributeValueManager->addAttributeValue($data, $attributeInfo);

                $this->adminCommon()->addOperLog($this->translator->translate('属性值添加成功!'), $this->translator->translate('商品属性'));

                return $this->redirect()->toRoute('goods-attribute', ['action' => 'attributeValue', 'id' => $attributeInfo->getAttributeId()]);
            }
        }

        return ['form' => $form, 'attributeInfo' => $attributeInfo];
    }

    /**
     * 编辑商品属性值
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editAttributeValueAction()
    {
        $attributeValueId   = (int) $this->params()->fromRoute('vid', -1);
        $attributeId        = (int) $this->params()->fromRoute('id', -1);

        $attributeInfo = $this->entityManager->getRepository(GoodsAttribute::class)->findOneByAttributeId($attributeId);
        if($attributeInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('属性值对应的属性不存在!'));
            return $this->redirect()->toRoute('goods-attribute');
        }

        $attributeValueInfo = $this->entityManager->getRepository(GoodsAttributeValue::class)->findOneByValueId($attributeValueId);
        if($attributeValueInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该属性值不存在!'));
            return $this->redirect()->toRoute('goods-attribute', ['action' => 'attributeValue', 'id' => $attributeValueInfo->getAttributeId()]);
        }

        $form = new GoodsAttributeValueFrom('edit');

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if($form->isValid()) {
                $data = $form->getData();
                $this->goodsAttributeValueManager->editAttributeValue($data, $attributeValueInfo);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('属性值 %s 编辑成功!'), $data['valueName']), $this->translator->translate('商品属性'));

                return $this->redirect()->toRoute('goods-attribute', ['action' => 'attributeValue', 'id' => $attributeValueInfo->getAttributeId()]);
            }
        } else $form->setData($attributeValueInfo->valuesArray());

        return ['form' => $form, 'attributeInfo' => $attributeInfo, 'attributeValueInfo' => $attributeValueInfo];

    }

    /**
     * 删除商品属性
     * @return \Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteAttributeValueAction()
    {
        if(!$this->adminCommon()->validatorCsrf()) return $this->adminCommon()->toReferer();

        $attributeValueId = (int) $this->params()->fromRoute('id', -1);

        $attributeValueInfo = $this->entityManager->getRepository(GoodsAttributeValue::class)->findOneByValueId($attributeValueId);
        if($attributeValueInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该属性值不存在!'));
            return $this->redirect()->toRoute('goods-attribute', ['action' => 'attributeValue', 'id' => $attributeValueInfo->getAttributeId()]);
        }

        $this->goodsAttributeValueManager->deleteAttributeValue($attributeValueInfo);

        $this->adminCommon()->addOperLog(sprintf($this->translator->translate('属性值 %s 删除成功!'), $attributeValueInfo->getValueName()), $this->translator->translate('商品属性'));

        return $this->adminCommon()->toReferer();
    }

    /**
     * 商品属性值批量处理
     * @return mixed
     * @throws \Doctrine\Common\Persistence\Mapping\MappingException
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function updateAttributeValueAllAction()
    {
        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            if(!empty($data['select_id']) and !empty($data['editAllState'])) {
                $this->goodsAttributeValueManager->updateAllValue($data);

                $this->adminCommon()->addOperLog($this->translator->translate('属性值批量处理成功!'), $this->translator->translate('商品属性'));
            }
        }
        return $this->adminCommon()->toReferer();
    }
}