<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Goods\Controller;

use Admin\Controller\AdminActionController;
use Doctrine\ORM\EntityManager;
use Goods\Entity\GoodsAttribute;
use Goods\Entity\GoodsAttributeGroup;
use Goods\Form\GoodsAttributeGroupForm;
use Goods\Service\GoodsAttributeGroupManager;
use Laminas\Mvc\I18n\Translator;

class GoodsAttributeGroupController extends AdminActionController
{
    private $translator;
    private $entityManager;
    private $goodsAttributeGroupManager;

    public function __construct(
        Translator $translator,
        EntityManager $entityManager,
        GoodsAttributeGroupManager $goodsAttributeGroupManager
    )
    {
        $this->translator       = $translator;
        $this->entityManager    = $entityManager;
        $this->goodsAttributeGroupManager = $goodsAttributeGroupManager;
    }

    /**
     * 商品属性组列表
     * @return array|\Laminas\View\Model\ViewModel
     */
    public function indexAction()
    {
        $attributeGroupList = $this->entityManager->getRepository(GoodsAttributeGroup::class)->findBy([], ['attributeGroupSort' => 'ASC']);

        return ['attributeGroupList' => $attributeGroupList];
    }

    /**
     * 添加商品属性组
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addAction()
    {
        $form = new GoodsAttributeGroupForm();

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if($form->isValid()) {
                $data = $form->getData();
                $this->goodsAttributeGroupManager->addAttributeGroup($data);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('属性组 %s 添加成功!'), $data['attributeGroupName']), $this->translator->translate('商品属性组'));

                return $this->redirect()->toRoute('goods-attribute-group');
            }
        }

        return ['form' => $form];
    }

    /**
     * 商品属性组编辑
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editAction()
    {
        $attributeGroupId = (int) $this->params()->fromRoute('id', -1);

        $attributeGroupInfo = $this->entityManager->getRepository(GoodsAttributeGroup::class)->findOneByAttributeGroupId($attributeGroupId);
        if($attributeGroupInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该属性组不存在!'));
            return $this->redirect()->toRoute('goods-attribute-group');
        }

        $form = new GoodsAttributeGroupForm();

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->goodsAttributeGroupManager->editAttributeGroup($data, $attributeGroupInfo);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('属性组 %s 编辑成功!'), $data['attributeGroupName']), $this->translator->translate('商品属性组'));

                return $this->redirect()->toRoute('goods-attribute-group');
            }
        } else $form->setData($attributeGroupInfo->valuesArray());

        return['form' => $form, 'attributeGroupInfo' => $attributeGroupInfo];
    }

    /**
     * 删除商品属性组
     * @return \Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteAction()
    {
        if(!$this->adminCommon()->validatorCsrf()) return $this->adminCommon()->toReferer();

        $attributeGroupId = (int) $this->params()->fromRoute('id', -1);

        $attributeGroupInfo = $this->entityManager->getRepository(GoodsAttributeGroup::class)->findOneByAttributeGroupId($attributeGroupId);
        if($attributeGroupInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该属性组不存在!'));
            return $this->redirect()->toRoute('goods-attribute-group');
        }

        $attributeInfo = $this->entityManager->getRepository(GoodsAttribute::class)->findOneBy(['attributeGroupId' => $attributeGroupId]);
        if($attributeInfo) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该属性组下存在商品属性，不能删除!'));
            return $this->adminCommon()->toReferer();
        }

        $this->goodsAttributeGroupManager->deleteAttributeGroup($attributeGroupInfo);

        $this->adminCommon()->addOperLog(sprintf($this->translator->translate('属性组 %s 删除成功!'), $attributeGroupInfo->getAttributeGroupName()), $this->translator->translate('商品属性组'));

        return $this->adminCommon()->toReferer();
    }

    /**
     * 批量修改属性组排序
     * @return mixed
     * @throws \Doctrine\Common\Persistence\Mapping\MappingException
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function updateAllSortAction()
    {
        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            if(!empty($data['attributeGroupSort'])) {
                $this->goodsAttributeGroupManager->updateAllAttributeGroupSort($data);

                $this->adminCommon()->addOperLog($this->translator->translate('批量排序处理成功!'), $this->translator->translate('商品属性组'));
            }
        }

        return $this->adminCommon()->toReferer();
    }
}