<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Goods\Controller;

use Admin\Controller\AdminActionController;
use Admin\Data\Common;
use Doctrine\ORM\EntityManager;
use Goods\Entity\GoodsClass;
use Goods\Entity\GoodsCommonContent;
use Goods\Form\GoodsCommonContentForm;
use Goods\Service\GoodsCommonContentManager;
use Laminas\Mvc\I18n\Translator;

class GoodsCommonContentController extends AdminActionController
{
    private $translator;
    private $entityManager;
    private $goodsCommonContentManager;

    public function __construct(
        Translator $translator,
        EntityManager $entityManager,
        GoodsCommonContentManager $goodsCommonContentManager
    )
    {
        $this->translator       = $translator;
        $this->entityManager    = $entityManager;
        $this->goodsCommonContentManager = $goodsCommonContentManager;
    }

    /**
     * 商品通用内容列表
     * @return array|\Laminas\View\Model\ViewModel
     */
    public function indexAction()
    {
        $contentList    = $this->entityManager->getRepository(GoodsCommonContent::class)->findBy([], ['contentSort' => 'ASC']);
        $goodsClass     = $this->entityManager->getRepository(GoodsClass::class)->findBy([]);
        $goodsClassArray= $this->goodsPlugin()->classValueArray($goodsClass);

        return ['contentList' => $contentList, 'goodsClassArray' => $goodsClassArray];
    }

    /**
     * 添加商品通用内容
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addAction()
    {
        //商品分类
        $goodsClass = $this->entityManager->getRepository(GoodsClass::class)->findBy([], ['classTopId' => 'ASC', 'classSort' => 'ASC']);
        $classList  = Common::classOptions($goodsClass);

        $form = new GoodsCommonContentForm();
        $form->get('classId')->setValueOptions($this->goodsPlugin()->classValueArray($classList));

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->goodsCommonContentManager->addGoodsCommonContent($data);

                $this->adminCommon()->addOperLog($this->translator->translate('商品通用内容添加成功!'), $this->translator->translate('商品通用内容'));

                return $this->redirect()->toRoute('goods-common');
            }
        }

        return ['form' => $form, 'classList' => $classList];
    }

    /**
     * 编辑商品通用内容
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editAction()
    {
        $contentId = (int) $this->params()->fromRoute('id', -1);

        $commonContent = $this->entityManager->getRepository(GoodsCommonContent::class)->findOneBy(['contentId' => $contentId]);
        if($commonContent == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该内容不存在!'));
            return $this->redirect()->toRoute('goods-common');
        }

        //商品分类
        $goodsClass = $this->entityManager->getRepository(GoodsClass::class)->findBy([], ['classTopId' => 'ASC', 'classSort' => 'ASC']);
        $classList  = Common::classOptions($goodsClass);

        $form = new GoodsCommonContentForm();
        $form->get('classId')->setValueOptions($this->goodsPlugin()->classValueArray($classList));

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->goodsCommonContentManager->editGoodsCommonContent($data, $commonContent);

                $this->adminCommon()->addOperLog($this->translator->translate('商品通用内容编辑成功!'), $this->translator->translate('商品通用内容'));

                return $this->redirect()->toRoute('goods-common');
            }
        } else $form->setData(['contentState' => $commonContent->getContentState(), 'contentTitle' => $commonContent->getContentTitle(), 'contentSort' => $commonContent->getContentSort()]);

        return [
            'form' => $form,
            'commonContent' => $commonContent,
            'idArray' => explode(',', trim($commonContent->getClassIdStr(), ',')),
            'classList' => $classList
        ];
    }

    /**
     * 商品通用内容删除
     * @return \Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteAction()
    {
        if(!$this->adminCommon()->validatorCsrf()) return $this->adminCommon()->toReferer();

        $contentId = (int) $this->params()->fromRoute('id', -1);

        $commonContent = $this->entityManager->getRepository(GoodsCommonContent::class)->findOneBy(['contentId' => $contentId]);
        if($commonContent == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该内容不存在!'));
            return $this->redirect()->toRoute('goods-common');
        }

        $this->goodsCommonContentManager->deleteGoodsCommonContent($commonContent);

        $this->adminCommon()->addOperLog($this->translator->translate('商品通用内容删除成功!'), $this->translator->translate('商品通用内容'));

        return $this->adminCommon()->toReferer();
    }
}