<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Goods\Controller;

use Admin\Controller\AdminActionController;
use Doctrine\ORM\EntityManager;
use Goods\Entity\Goods;
use Goods\Entity\GoodsConsultation;
use Goods\Form\GoodsReplyConsultForm;
use Goods\Form\SearchGoodsConsultForm;
use Goods\Service\GoodsConsultationManager;
use Laminas\Mvc\I18n\Translator;
use Laminas\View\Model\JsonModel;
use Laminas\View\Model\ViewModel;

class GoodsConsultationController extends AdminActionController
{
    private $translator;
    private $entityManager;
    private $goodsConsultationManager;

    public function __construct(
        Translator $translator,
        EntityManager $entityManager,
        GoodsConsultationManager $goodsConsultationManager
    )
    {
        $this->translator   = $translator;
        $this->entityManager= $entityManager;
        $this->goodsConsultationManager = $goodsConsultationManager;
    }

    /**
     * 商品咨询列表
     * @return array|\Laminas\View\Model\ViewModel
     */
    public function indexAction()
    {
        $page = (int) $this->params()->fromQuery('page', 1);

        $search = [];
        $searchForm = new SearchGoodsConsultForm();
        if($this->getRequest()->isGet()) {
            $data = $this->params()->fromQuery();
            $searchForm->setData($data);
            if($searchForm->isValid()) $search = $searchForm->getData();
        }

        $query = $this->entityManager->getRepository(GoodsConsultation::class)->findGoodsConsultList($search);
        $paginator  = $this->adminCommon()->shopPaginator($query, $page);

        return ['consultList' => $paginator, 'searchForm' => $searchForm];
    }

    /**
     * 商品咨询回复
     * @return JsonModel|ViewModel
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function replyConsultAction()
    {
        $view = new ViewModel();
        $view->setTerminal(true);

        $consultId = (int) $this->params()->fromRoute('id', -1);
        $consultInfo = $this->entityManager->getRepository(GoodsConsultation::class)->findOneByConsultId($consultId);
        if($consultInfo == null) return $this->getResponse()->setContent('');

        $form = new GoodsReplyConsultForm();
        $form->get('consultShowStatus')->setValueOptions([1 => $this->translator->translate('前台显示'), 2 => $this->translator->translate('前台不显示')]);

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if($form->isValid()) {
                $data = $form->getData();
                $this->goodsConsultationManager->editGoodsConsultation([
                    'replyContent'      => $data['replyContent'],
                    'consultShowStatus' => $data['consultShowStatus'],
                    'replyTime'         => time(),
                    'replyUser'         => $this->translator->translate('店主')
                ], $consultInfo);

                $this->adminCommon()->addOperLog($this->translator->translate('商品咨询回复成功!'), $this->translator->translate('商品咨询'));

                return new JsonModel(['status' => 'true']);
            } else return new JsonModel(['status' => 'false', 'message' => $this->translator->translate('咨询回复失败!')]);
        }

        $goodsInfo = $this->entityManager->getRepository(Goods::class)->findOneByGoodsId($consultInfo->getGoodsId());

        $form->setData(['consultShowStatus' => $consultInfo->getConsultShowStatus(), 'replyContent' => $consultInfo->getReplyContent()]);

        return $view->setVariables(['form' => $form, 'consultInfo' => $consultInfo, 'goodsInfo' => $goodsInfo]);
    }

    /**
     * 商品咨询删除
     * @return mixed
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteAction()
    {
        if(!$this->adminCommon()->validatorCsrf()) return $this->adminCommon()->toReferer();

        $consultId = (int) $this->params()->fromRoute('id', -1);
        $consultInfo = $this->entityManager->getRepository(GoodsConsultation::class)->findOneByConsultId($consultId);
        if($consultInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该商品咨询不存在!'));
            return $this->adminCommon()->toReferer();
        }

        $this->goodsConsultationManager->deleteGoodsConsultation($consultInfo);

        $this->adminCommon()->addOperLog($this->translator->translate('商品咨询删除成功!'), $this->translator->translate('商品咨询'));

        return $this->adminCommon()->toReferer();
    }

    /**
     * 批量删除商品咨询
     * @return mixed
     */
    public function deleteMoreConsultationAction()
    {
        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            if (!empty($data['select_id'])) {
                $this->goodsConsultationManager->deleteMoreGoodsConsultation($data['select_id']);
                $this->adminCommon()->addOperLog($this->translator->translate('商品咨询批量删除成功!'), $this->translator->translate('商品咨询'));
            }
        }
        return $this->adminCommon()->toReferer();
    }
}