<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Goods\Controller;

use Admin\Controller\AdminActionController;
use Doctrine\ORM\EntityManager;
use Goods\Entity\GoodsTag;
use Goods\Entity\GoodsTagGroup;
use Goods\Form\GoodsTagGroupForm;
use Goods\Service\GoodsTagGroupManager;
use Laminas\Mvc\I18n\Translator;

class GoodsTagGroupController extends AdminActionController
{
    private $translator;
    private $entityManager;
    private $goodsTagGroupManager;

    public function __construct(
        Translator $translator,
        EntityManager $entityManager,
        GoodsTagGroupManager $goodsTagGroupManager
    )
    {
        $this->translator       = $translator;
        $this->entityManager    = $entityManager;
        $this->goodsTagGroupManager = $goodsTagGroupManager;
    }

    /**
     * 标签组列表
     * @return array|\Laminas\View\Model\ViewModel
     */
    public function indexAction()
    {
        $tagGroupList = $this->entityManager->getRepository(GoodsTagGroup::class)->findBy([], ['tagGroupSort' => 'ASC']);

        return ['tagGroupList' => $tagGroupList];
    }

    /**
     * 添加标签组信息
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addAction()
    {
        $form = new GoodsTagGroupForm();

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if($form->isValid()) {
                $data = $form->getData();
                $this->goodsTagGroupManager->addGoodsTagGroup($data);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('标签组 %s 添加成功!'), $data['tagGroupName']), $this->translator->translate('商品标签组'));

                return $this->redirect()->toRoute('goods-tag-group');
            }
        }

        return ['form' => $form];
    }

    /**
     * 编辑标签组信息
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editAction()
    {
        $tagGroupId = (int) $this->params()->fromRoute('id', -1);

        $tagGroupInfo = $this->entityManager->getRepository(GoodsTagGroup::class)->findOneByTagGroupId($tagGroupId);
        if($tagGroupInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该标签组不存在!'));
            return $this->redirect()->toRoute('goods-tag-group');
        }

        $form = new GoodsTagGroupForm();

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->goodsTagGroupManager->editGoodsTagGroup($data, $tagGroupInfo);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('标签组 %s 编辑成功!'), $data['tagGroupName']), $this->translator->translate('商品标签组'));

                return $this->redirect()->toRoute('goods-tag-group');
            }
        } else $form->setData($tagGroupInfo->valuesArray());

        return['form' => $form, 'goodsTagGroup' => $tagGroupInfo];
    }

    /**
     * 删除标签组
     * @return mixed
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteAction()
    {
        if(!$this->adminCommon()->validatorCsrf()) return $this->adminCommon()->toReferer();

        $tagGroupId = (int) $this->params()->fromRoute('id', -1);

        $tagGroupInfo = $this->entityManager->getRepository(GoodsTagGroup::class)->findOneByTagGroupId($tagGroupId);
        if($tagGroupInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该标签组不存在!'));
            return $this->adminCommon()->toReferer();
        }

        $tagInfo = $this->entityManager->getRepository(GoodsTag::class)->findOneBy(['tagGroupId' => $tagGroupId]);
        if($tagInfo) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该标签组下存在商品标签，不能删除!'));
            return $this->adminCommon()->toReferer();
        }

        $this->goodsTagGroupManager->deleteGoodsTagGroup($tagGroupInfo);

        $this->adminCommon()->addOperLog(sprintf($this->translator->translate('标签组 %s 删除成功!'), $tagGroupInfo->getTagGroupName()), $this->translator->translate('商品标签组'));

        return $this->adminCommon()->toReferer();
    }

    /**
     * 批量处理排序
     * @return mixed
     * @throws \Doctrine\Common\Persistence\Mapping\MappingException
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function updateAllSortAction()
    {
        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            if(!empty($data['tagGroupSort'])) {
                $this->goodsTagGroupManager->updateAllTagGroupSort($data);

                $this->adminCommon()->addOperLog($this->translator->translate('批量排序处理成功!'), $this->translator->translate('商品标签组'));
            }
        }

        return $this->adminCommon()->toReferer();
    }
}