<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Goods\Event;

use Admin\Data\Common;
use Admin\Data\Config;
use Doctrine\ORM\EntityManager;
use Goods\Service\GoodsConsultationManager;
use Goods\Service\GoodsInSpecImageManager;
use Goods\Service\GoodsInTagManager;
use Goods\Service\GoodsSpecTagGoodsManager;
use Laminas\EventManager\Event;
use Laminas\EventManager\EventManagerInterface;
use Laminas\EventManager\ListenerAggregateInterface;

class GoodsListener implements ListenerAggregateInterface
{
    protected $listeners = [];

    private $entityManager;
    private $goodsInTagManager;
    private $goodsInSpecImageManager;
    private $goodsConsultationManager;
    private $goodsSpecTagGoodsManager;

    public function __construct(
        EntityManager   $entityManager,
        GoodsInTagManager $goodsInTagManager,
        GoodsInSpecImageManager $goodsInSpecImageManager,
        GoodsConsultationManager $goodsConsultationManager,
        GoodsSpecTagGoodsManager $goodsSpecTagGoodsManager
    )
    {
        $this->entityManager = $entityManager;
        $this->goodsInTagManager = $goodsInTagManager;
        $this->goodsInSpecImageManager = $goodsInSpecImageManager;
        $this->goodsConsultationManager= $goodsConsultationManager;
        $this->goodsSpecTagGoodsManager = $goodsSpecTagGoodsManager;
    }

    public function attach(EventManagerInterface $events, $priority = 1)
    {
        $shareEvents = $events->getSharedManager();

        //商品编辑时的处理
        $this->listeners[] = $shareEvents->attach(
            'Goods\Controller\IndexController', 'goods.edit.post', [$this, 'onEditGoodsOther']
        );

        //标签删除时，同时删除商品中绑定的该标签
        $this->listeners[] = $shareEvents->attach(
            'Goods\Controller\GoodsTagController', 'tag.delete.post', [$this, 'onDeleteGoodsInTag']
        );
        //商品图片删除时，同时删除规格里面的图片
        $this->listeners[] = $shareEvents->attach(
            'Goods\Controller\IndexController', 'goodsImage.delete.post', [$this, 'onDeleteGoodsSpecImage']
        );
        //商品删除时，删除处理
        $this->listeners[] = $shareEvents->attach(
            'Goods\Controller\IndexController', 'goods.delete.post', [$this, 'onDeleteGoodsOther']
        );
    }

    public function detach(EventManagerInterface $events)
    {
        foreach ($this->listeners as $index => $listener) {
            $events->detach($listener);
            unset($this->listeners[$index]);
        }
    }

    /**
     * 商品编辑时的处理
     * @param Event $event
     */
    public function onEditGoodsOther(Event $event)
    {
        $goodsId = (int) $event->getParam('goodsId');
        if($goodsId > 0) {
            //分享海报删除
            Common::deleteDirAndFile(Config::GOODS_QRCODE_IMAGE_PATH . $goodsId, false);
        }
    }

    /**
     * 标签删除时，同时删除商品中绑定的该标签
     * @param Event $event
     */
    public function onDeleteGoodsInTag(Event $event)
    {
        $tagIdArray = $event->getParams();
        if(!empty($tagIdArray)) foreach ($tagIdArray as $tagId) {
            $this->goodsInTagManager->deleteTagIdGoods(intval($tagId));
        }
    }

    /**
     * 商品图片删除时，同时删除规格里面的图片
     * @param Event $event
     */
    public function onDeleteGoodsSpecImage(Event $event)
    {
        $goodsImageId = (int) $event->getParam('goodsImageId');
        if($goodsImageId > 0) $this->goodsInSpecImageManager->deleteMoreSpecImage($goodsImageId);
    }

    /**
     * 删除商品下的内容
     * 1)商品咨询
     * 2)海报删除
     * @param Event $event
     */
    public function onDeleteGoodsOther(Event $event)
    {
        $goodsId = (int) $event->getParam('goodsId');
        if($goodsId > 0) {
            //特定标签商品
            $this->goodsSpecTagGoodsManager->deleteMoreSpecTagGoods($goodsId);
            //商品咨询
            $this->goodsConsultationManager->deleteOneGoodsMoreConsult($goodsId);
            //分享海报
            Common::deleteDirAndFile(Config::GOODS_QRCODE_IMAGE_PATH.$goodsId);
        }
    }
}