<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Goods\Plugin;

use Admin\Data\Common;
use Doctrine\ORM\EntityManager;
use Goods\Entity\Goods;
use Goods\Entity\GoodsAttributeGroup;
use Goods\Entity\GoodsBrand;
use Goods\Entity\GoodsBrandInClass;
use Goods\Entity\GoodsClass;
use Goods\Entity\GoodsInClass;
use Goods\Entity\GoodsTagGroup;
use Laminas\Mvc\Controller\Plugin\AbstractPlugin;
use Laminas\Mvc\I18n\Translator;

class GoodsPlugin extends AbstractPlugin
{
    private $entityManager;
    private $translator;

    public function __construct(
        EntityManager   $entityManager,
        Translator      $translator
    )
    {
        $this->entityManager    = $entityManager;
        $this->translator       = $translator;
    }

    /**
     * 品牌列表
     * @param string $topName
     * @param bool $haveState
     * @return array|string[]
     */
    public function goodsBrandOptions($topName = '', $haveState = true)
    {
        $brandList = [];
        if($haveState) $brandList = ['0' => empty($topName) ? $this->translator->translate('===选择品牌===') : $topName];
        $goodsBrand= $this->entityManager->getRepository(GoodsBrand::class)->findBy([], ['brandSort' => 'ASC']);
        if($goodsBrand) {
            foreach ($goodsBrand as $value) {
                $brandList[$value->getBrandId()] = $value->getBrandName();
            }
        }
        return $brandList;
    }

    /**
     * 虚拟商品类型获取
     * @param $type
     * @return mixed
     */
    public function getVirtualGoodsType($type)
    {
        $data = Common::virtualGoodsType($this->translator);
        return $data[$type];
    }

    /**
     * 虚拟商品状态
     * @param $state
     * @return mixed
     */
    public function getVirtualGoodsState($state)
    {
        $data = Common::virtualGoodsState($this->translator);
        return $data[$state];
    }

    /**
     * 生成商品货号
     * @return string
     */
    public function createGoodsItem()
    {
        $goodsId    = $this->entityManager->getRepository(Goods::class)->getMaxGoodsId();
        $nextGoodsId= $goodsId == null ? 1 : $goodsId + 1;
        $goodsItem  = Common::configValue('goods', 'config')['goods_sn_prefix'] . str_repeat('0', 6 - strlen($nextGoodsId)) . $nextGoodsId;

        return $goodsItem;
    }

    /**
     * 获取品牌所在的分类
     * @param $brandId
     * @return array
     */
    public function selectedGoodsBrandInClassArray($brandId)
    {
        $inGoodsClass = $this->entityManager->getRepository(GoodsBrandInClass::class)->findBy(['brandId' => $brandId]);
        $selectedClass= [];
        if($inGoodsClass) foreach ($inGoodsClass as $classValue) {
            $selectedClass[] = $classValue->getClassId();
        }
        return $selectedClass;
    }

    /**
     * 获取分类的值数组
     * @param $classList
     * @return array
     */
    public function classValueArray($classList)
    {
        $class = [];
        if(!empty($classList)) foreach ($classList as $classValue) {
            $class[$classValue->getClassId()] = $classValue->getClassName();
        }
        return $class;
    }

    /**
     * 商品选中的分类数组信息
     * @param $goodsId
     * @return array
     */
    public function selectedGoodsClassArray($goodsId)
    {
        $inGoodsClass = $this->entityManager->getRepository(GoodsInClass::class)->findBy(['goodsId' => $goodsId]);
        $selectedClass= [];
        if($inGoodsClass) foreach ($inGoodsClass as $classValue) {
            $selectedClass[] = $classValue->getClassId();
        }

        return $selectedClass;
    }

    /**
     * 获取分类列表
     * @param string $topName
     * @param bool $haveState
     * @return array|string[]
     */
    public function classListOptions($topName = '', $haveState = true)
    {
        $goodsClassList = [];

        if($haveState) $goodsClassList = ['0' => empty($topName) ? $this->translator->translate('===顶级分类===') : $topName];
        $goodsClass     = $this->entityManager->getRepository(GoodsClass::class)->findBy([], ['classTopId' => 'ASC', 'classSort' => 'ASC']);
        if($goodsClass) {
            $nbspStr = html_entity_decode('&nbsp;', ENT_COMPAT, 'UTF-8');
            $classArray = Common::classOptions($goodsClass);
            foreach ($classArray as $value) {
                if($value->getClassTopId() > 0) {
                    $goodsClassList[$value->getClassId()] =  str_repeat($nbspStr, substr_count($value->getClassPath(), ',') * 3) . '|----' . $value->getClassName();
                } else {
                    $goodsClassList[$value->getClassId()] = $value->getClassName();
                }
            }
        }
        return $goodsClassList;
    }

    /**
     * 标签组
     * @param string $topName
     * @param bool $showTop 是否显示头信息
     * @return array
     */
    public function tagGroupOptions($topName = '', $showTop = true)
    {
        $tagGroupList = [];
        if($showTop)    $tagGroupList = ['' => empty($topName) ? $this->translator->translate('===选择标签分组===') : $topName];
        $tagGroup       = $this->entityManager->getRepository(GoodsTagGroup::class)->findBy([], ['tagGroupSort' => 'ASC']);
        if($tagGroup) {
            foreach ($tagGroup as $value) {
                $tagGroupList[$value->getTagGroupId()] = $value->getTagGroupName() . (empty($value->getTagGroupMark()) ? '' : '['.$value->getTagGroupMark().']');
            }
        }
        return $tagGroupList;
    }

    /**
     * 商品属性组select
     * @param string $topName
     * @return array
     */
    public function attributeGroupOptions($topName = '')
    {
        $attributeGroupList = [0 => empty($topName) ? $this->translator->translate('===选择属性分组===') : $topName];
        $attributeGroup     = $this->entityManager->getRepository(GoodsAttributeGroup::class)->findBy([], ['attributeGroupSort' => 'ASC']);
        if($attributeGroup) {
            foreach ($attributeGroup as $value) {
                $attributeGroupList[$value->getAttributeGroupId()] = $value->getAttributeGroupName();
            }
        }
        return $attributeGroupList;
    }
}