<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Goods\Service;

use Doctrine\ORM\EntityManager;
use Goods\Entity\GoodsInTag;
use Goods\Entity\GoodsTag;

class GoodsInTagManager
{
    private $entityManager;

    public function __construct(
        EntityManager $entityManager
    )
    {
        $this->entityManager = $entityManager;
    }

    /**
     * 添加单个标签商品
     * @param array $data
     * @return GoodsInTag
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addOneGoodsInTag(array $data)
    {
        $goodsInTag = new GoodsInTag();
        $goodsInTag->valuesSet($data);

        $this->entityManager->persist($goodsInTag);
        $this->entityManager->flush();

        return $goodsInTag;
    }

    /**
     * 添加商品对应的标签
     * @param array $data
     * @param $goodsId
     * @return string
     * @throws \Doctrine\Common\Persistence\Mapping\MappingException
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addGoodsInTag(array $data, $goodsId)
    {
        $tagIdStr = '';
        if(isset($data['tagId']) && !empty($data['tagId'])) foreach ($data['tagId'] as $tagId)
        {
            $tagInfo = $this->entityManager->getRepository(GoodsTag::class)->findOneByTagId(intval($tagId));
            if($tagInfo) {
                $goodsInTag = new GoodsInTag();
                $goodsInTag->valuesSet(['tagId' => $tagId, 'goodsId' => $goodsId, 'tagGoodsSort' => 255]);

                $tagIdStr .= $tagId . ',';

                $this->entityManager->persist($goodsInTag);
                $this->entityManager->flush();
                $this->entityManager->clear(GoodsInTag::class);
            }
        }
        return empty($tagIdStr) ? '' : ','.$tagIdStr;
    }

    /**
     * 编辑商品对应的标签
     * @param array $data
     * @param $goodsId
     * @return string
     * @throws \Doctrine\Common\Persistence\Mapping\MappingException
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editGoodsInTag(array $data, $goodsId)
    {
        $tagIdStr = '';
        if(isset($data['tagId']) && !empty($data['tagId'])) {
            $inTagList = $this->entityManager->getRepository(GoodsInTag::class)->findBy(['goodsId' => $goodsId]);
            if($inTagList) foreach ($inTagList as $tagValue) {
                if(!in_array($tagValue->getTagId(), $data['tagId'])) $this->entityManager->remove($tagValue);
                else {
                    $tagIdStr .= $tagValue->getTagId() . ',';
                    unset($data['tagId'][array_search($tagValue->getTagId(), $data['tagId'])]);
                }
            }

            if(!empty($data['tagId'])) foreach ($data['tagId'] as $tagId) {
                $goodsInTag = new GoodsInTag();
                $goodsInTag->valuesSet(['tagId' => $tagId, 'goodsId' => $goodsId, 'tagGoodsSort' => 255]);

                $tagIdStr .= $tagId . ',';

                $this->entityManager->persist($goodsInTag);
                $this->entityManager->flush();
                $this->entityManager->clear(GoodsInTag::class);
            }
        } else {
            $this->deleteTagGoodsId($goodsId);
        }
        return empty($tagIdStr) ? '' : ','.$tagIdStr;
    }

    /**
     * 删除某个标签商品
     * @param GoodsInTag $goodsInTag
     * @return bool
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteGoodsInTag(GoodsInTag $goodsInTag)
    {
        $this->entityManager->remove($goodsInTag);
        $this->entityManager->flush();

        return true;
    }

    /**
     * 删除标签对应的所有商品
     * @param $tagId
     */
    public function deleteTagIdGoods($tagId)
    {
        $qb = $this->entityManager->createQueryBuilder();
        $qb->delete(GoodsInTag::class, 'i')->where('i.tagId = :tagId')->setParameter('tagId', $tagId);

        $qb->getQuery()->execute();
    }

    /**
     * 根据商品id删除商品里的标签
     * @param $goodsId
     */
    public function deleteTagGoodsId($goodsId)
    {
        $qb = $this->entityManager->createQueryBuilder();
        $qb->delete(GoodsInTag::class, 'i')->where('i.goodsId = :goodsId')->setParameter('goodsId', $goodsId);

        $qb->getQuery()->execute();
    }
}