<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Shop\Controller;

use Admin\Data\Common;
use Doctrine\ORM\EntityManager;
use Laminas\Mvc\I18n\Translator;
use Shop\Form\ChangeUserEmailForm;
use Shop\Form\ChangeUserLoginPasswordForm;
use Shop\Form\ChangeUserPayPasswordForm;
use Shop\Form\ChangeUserPhoneForm;
use User\Entity\User;
use User\Service\UserManager;

class HomeSecurityController extends HomeActionController
{
    private $translator;
    private $writeEntityManager;
    private $userManager;

    public function __construct(
        Translator      $translator,
        EntityManager   $writeEntityManager,
        UserManager     $userManager
    )
    {
        $this->translator           = $translator;
        $this->writeEntityManager   = $writeEntityManager;
        $this->userManager          = $userManager;
    }

    /**
     * 账户安全页面
     * @return array|\Laminas\View\Model\ViewModel
     */
    public function indexAction()
    {
        $this->layout()->setVariable('titleName', $this->translator->translate('账户安全'));

        return [];
    }

    /**
     * 修改登录密码
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function changeUserLoginPasswordAction()
    {
        $this->layout()->setVariable('titleName', $this->translator->translate('登录密码'));

        $userInfo = $this->writeEntityManager->getRepository(User::class)->findOneBy(['userId' => parent::$userId]);

        $form = new ChangeUserLoginPasswordForm($userInfo);
        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->userManager->changeUserPassword($data['userPassword'], $userInfo);

                $this->getEventManager()->trigger('shop.user.login-password-edit.post', $this, ['userInfo' => $userInfo, 'loginPassword' => $data['userPassword']]);

                $this->flashMessenger()->addSuccessMessage($this->translator->translate('登录密码修改成功!'));
                return $this->redirect()->toRoute('home-security', ['action' => 'changeUserLoginPassword']);
            }
        }

        return ['form' => $form, 'userInfo' => $userInfo];
    }

    /**
     * 修改支付密码
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function changeUserPayPasswordAction()
    {
        $this->layout()->setVariable('titleName', $this->translator->translate('支付密码'));

        $userInfo = $this->writeEntityManager->getRepository(User::class)->findOneBy(['userId' => parent::$userId]);

        $form = new ChangeUserPayPasswordForm($userInfo);
        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->userManager->changePayPassword($data['payPassword'], $userInfo);

                $this->getEventManager()->trigger('shop.user.pay-password-edit.post', $this, ['userInfo' => $userInfo]);

                $this->flashMessenger()->addSuccessMessage($this->translator->translate('支付密码修改成功!'));
                return $this->redirect()->toRoute('home-security', ['action' => 'changeUserPayPassword']);
            }
        }

        return ['form' => $form, 'userInfo' => $userInfo];
    }

    /**
     * 修改电子邮箱
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function changeUserEmailAction()
    {
        $this->layout()->setVariable('titleName', $this->translator->translate('电子邮件'));

        $userInfo = $this->writeEntityManager->getRepository(User::class)->findOneBy(['userId' => parent::$userId]);

        $form = new ChangeUserEmailForm($this->writeEntityManager, $userInfo);
        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->userManager->changeUserEmail($data['userEmail'], $userInfo);

                $this->getEventManager()->trigger('shop.user.email-edit.post', $this, ['userInfo' => $userInfo]);

                $this->flashMessenger()->addSuccessMessage($this->translator->translate('电子邮箱修改成功!'));
                return $this->redirect()->toRoute('home-security', ['action' => 'changeUserEmail']);
            }
        }

        return ['form' => $form, 'userInfo' => $userInfo];
    }

    /**
     * 修改手机号码
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function changeUserPhoneAction()
    {
        $this->layout()->setVariable('titleName', $this->translator->translate('手机号码'));

        $phoneArea  = Common::getPhoneArea();
        $userInfo   = $this->writeEntityManager->getRepository(User::class)->findOneBy(['userId' => parent::$userId]);
        $form       = new ChangeUserPhoneForm($phoneArea, $this->writeEntityManager, $userInfo);
        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->userManager->changeUserPhone($data, $userInfo);

                $this->getEventManager()->trigger('shop.user.phone-edit.post', $this, ['userInfo' => $userInfo]);

                $this->flashMessenger()->addSuccessMessage($this->translator->translate('手机号码修改成功!'));
                return $this->redirect()->toRoute('home-security', ['action' => 'changeUserPhone']);
            }
        }

        return ['form' => $form, 'phoneArea' => $phoneArea, 'userInfo' => $userInfo];
    }
}