<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Shop\Form;

use Admin\Data\Common;
use Admin\Data\Config;
use Laminas\Form\Form;
use Shop\Common\ShopCommon;
use Shop\Validator\CheckoutAddressValidator;
use Shop\Validator\SelfLocationValidator;

class CheckoutForm extends Form
{
    public $inputFilter;

    private $entityManager;
    private $userId;
    private $deliveryType;
    private $allGoodsVirtual;
    private $haveGoodsVirtual;

    public function __construct($userId, $entityManager, $deliveryType, $allGoodsVirtual, $haveGoodsVirtual, $name = 'checkout-form', $options = [])
    {
        parent::__construct($name, $options);

        $this->setAttribute('method', 'post');

        $this->userId           = $userId;
        $this->entityManager    = $entityManager;
        $this->deliveryType     = $deliveryType;
        $this->allGoodsVirtual  = $allGoodsVirtual;
        $this->haveGoodsVirtual = $haveGoodsVirtual;

        $this->addElements();
        $this->inputFilter = $this->addInputFilter();
    }

    protected function addElements()
    {
        $this->add([
            'type'  => 'hidden',
            'name'  => 'deliveryType',
            'attributes' => [
                'id'    => 'deliveryType'
            ]
        ]);

        if($this->deliveryType == 'self-fetch') {
            $this->add([
                'type'  => 'hidden',
                'name'  => 'locationId',
                'attributes' => [
                    'id'    => 'locationId'
                ]
            ]);
        } elseif(!$this->allGoodsVirtual) {
            $this->add([
                'type'  => 'hidden',
                'name'  => 'addressId',
                'attributes' => [
                    'id'    => 'addressId'
                ]
            ]);

            $this->add([
                'type'  => 'hidden',
                'name'  => 'deliveryTimeInfo',
                'attributes' => [
                    'id'    => 'deliveryTimeInfo'
                ]
            ]);
        }

        $this->add([
            'type'  => 'hidden',
            'name'  => 'paymentType',
            'attributes' => [
                'id'    => 'paymentType'
            ]
        ]);

        $this->add([
            'type'  => 'textarea',
            'name'  => 'orderMessage',
            'attributes' => [
                'id'    => 'orderMessage'
            ]
        ]);

        $this->add([
            'type'  => 'csrf',
            'name'  => 'shopCsrf',
            'options' => [
                'csrf_options' => [
                    'timeout'  => Config::SHOP_POST_TOKEN_TIMEOUT
                ]
            ]
        ]);
    }

    protected function addInputFilter()
    {
        $inputFilter = $this->getInputFilter();

        $inputFilter->add([
            'name'      => 'deliveryType',
            'required'  => true,
            'validators'=> [
                [
                    'name'      => 'InArray',
                    'options'   => [
                        'haystack'  => Common::deliveryType($this->allGoodsVirtual, $this->haveGoodsVirtual)
                    ]
                ]
            ]
        ]);

        if($this->deliveryType == 'self-fetch') {
            $inputFilter->add([
                'name'      => 'locationId',
                'required'  => true,
                'filters'   => [
                    ['name' => 'ToInt']
                ],
                'validators'=> [
                    [
                        'name'      => SelfLocationValidator::class,
                        'options'   => [
                            'entityManager' => $this->entityManager
                        ]
                    ]
                ]
            ]);
        } elseif(!$this->allGoodsVirtual) {
            $inputFilter->add([
                'name'      => 'addressId',
                'required'  => true,
                'filters'   => [
                    ['name' => 'ToInt']
                ],
                'validators'=> [
                    [
                        'name'      => CheckoutAddressValidator::class,
                        'options'   => [
                            'entityManager' => $this->entityManager,
                            'userId'        => $this->userId
                        ]
                    ]
                ]
            ]);

            $inputFilter->add([
                'name'      => 'deliveryTimeInfo',
                'required'  => false,
                'validators'=> [
                    [
                        'name'      => 'InArray',
                        'options'   => [
                            'haystack'  => ShopCommon::shopShippingDate()
                        ]
                    ]
                ]
            ]);
        }

        $inputFilter->add([
            'name'      => 'paymentType',
            'required'  => true,
            'validators'=> [
                [
                    'name'      => 'InArray',
                    'options'   => [
                        'haystack'  => Common::paymentType($this->allGoodsVirtual, $this->haveGoodsVirtual, $this->deliveryType)
                    ]
                ]
            ]
        ]);

        $inputFilter->add([
            'name'      => 'orderMessage',
            'required'  => false,
            'filters'   => [
                ['name' => 'StringTrim'],
                ['name' => 'StripTags'],
                ['name' => 'HtmlEntities']
            ],
            'validators'=> [
                [
                    'name'      => 'StringLength',
                    'options'   => [
                        'min'   => 1,
                        'max'   => 500
                    ]
                ]
            ]
        ]);

        return $inputFilter;
    }

    public function addElementsAndFilter($formArray)
    {
        foreach ($formArray as $formInput) {
            $this->add($formInput['input']);
            $this->inputFilter->add($formInput['filter']);
        }
    }
}