<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace User\Controller;

use Admin\Controller\AdminActionController;
use Doctrine\ORM\EntityManager;
use User\Entity\User;
use User\Entity\UserGroup;
use User\Form\UserGroupForm;
use User\Service\UserGroupManager;
use Laminas\Mvc\I18n\Translator;

class UserGroupController extends AdminActionController
{
    private $translator;
    private $entityManager;
    private $userGroupManager;

    public function __construct(
        Translator $translator,
        EntityManager $entityManager,
        UserGroupManager $userGroupManager
    )
    {
        $this->translator       = $translator;
        $this->entityManager    = $entityManager;
        $this->userGroupManager = $userGroupManager;
    }

    /**
     * 客户组列表
     * @return array|\Laminas\View\Model\ViewModel
     */
    public function indexAction()
    {
        $userGroup = $this->entityManager->getRepository(UserGroup::class)->findAll();

        return ['userGroup' => $userGroup];
    }

    /**
     * 添加客户组
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addAction()
    {
        $form = new UserGroupForm();

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->userGroupManager->addUserGroup($data);
                $this->createConfig()->createUserGroup();

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('客户组 %s 添加成功!'), $data['userGroupName']), $this->translator->translate('客户组管理'));

                return $this->redirect()->toRoute('user-group');
            }
        }

        return ['form' => $form];
    }

    /**
     * 编辑客户组
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editAction()
    {
        $userGroupId = (int) $this->params()->fromRoute('id', -1);

        $userGroupInfo = $this->entityManager->getRepository(UserGroup::class)->findOneByUserGroupId($userGroupId);
        if($userGroupInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该客户组不存在!'));
            return $this->redirect()->toRoute('user-group');
        }

        $form = new UserGroupForm();

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $this->userGroupManager->editUserGroup($data, $userGroupInfo);
                $this->createConfig()->createUserGroup();

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('客户组 %s 编辑成功!'), $data['userGroupName']), $this->translator->translate('客户组管理'));

                return $this->redirect()->toRoute('user-group');
            }
        } else $form->setData($userGroupInfo->valuesArray());

        return ['form' => $form, 'userGroup' => $userGroupInfo];
    }

    /**
     * 删除客户组
     * @return mixed
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteAction()
    {
        if(!$this->adminCommon()->validatorCsrf()) return $this->adminCommon()->toReferer();

        $userGroupId = (int) $this->params()->fromRoute('id', -1);

        if($userGroupId == 1) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该客户组不能删除!'));
            return $this->adminCommon()->toReferer();
        }

        $userGroupInfo = $this->entityManager->getRepository(UserGroup::class)->findOneByUserGroupId($userGroupId);
        if($userGroupInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该客户组不存在!'));
            return $this->adminCommon()->toReferer();
        }

        $userInfo = $this->entityManager->getRepository(User::class)->findOneBy(['userGroupId' => $userGroupId]);
        if($userInfo) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该客户组下有客户存在，不能删除!'));
            return $this->adminCommon()->toReferer();
        }

        $this->userGroupManager->deleteUserGroup($userGroupInfo);
        $this->createConfig()->createUserGroup();

        $this->adminCommon()->addOperLog(sprintf($this->translator->translate('客户组 %s 删除成功!'), $userGroupInfo->getUserGroupName()), $this->translator->translate('客户组管理'));

        return $this->adminCommon()->toReferer();
    }
}