<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace User\Form;

use Admin\Data\Common;
use Admin\Data\Config;
use Admin\Filter\ImageThumb;
use Admin\Filter\OldImage;
use User\Validator\UserEmailExistsValidator;
use User\Validator\UserExistsValidator;
use Laminas\Form\Form;
use User\Validator\UserPhoneExistsValidator;

class UserForm extends Form
{
    private $phoneArea;
    private $entityManager;
    private $userInfo;

    public function __construct(array $phoneArea, $entityManager = null, $userInfo = null, $name = 'user-form', $options = [])
    {
        parent::__construct($name, $options);
        $this->setAttribute('method', 'post');
        $this->setAttribute('enctype', 'multipart/form-data');
        $this->setAttribute('class', 'form-horizontal');

        $this->phoneArea    = $phoneArea;
        $this->entityManager= $entityManager;
        $this->userInfo     = $userInfo;

        $this->addElements();
        $this->addInputFilter();
    }

    protected function addElements()
    {
        $this->add([
            'type'  => 'select',
            'name'  => 'userGroupId',
            'attributes'    => [
                'id'            => 'userGroupId',
                'class'         => 'form-control'
            ]
        ]);

        $this->add([
            'type'  => 'select',
            'name'  => 'userSex',
            'attributes'    => [
                'id'            => 'userSex',
                'class'         => 'form-control'
            ]
        ]);

        $this->add([
            'type'  => 'text',
            'name'  => 'userName',
            'attributes'    => [
                'id'            => 'userName',
                'class'         => 'form-control'
            ]
        ]);

        if($this->userInfo == null) {
            $this->add([
                'type'  => 'password',
                'name'  => 'userPassword',
                'attributes'    => [
                    'id'            => 'userPassword',
                    'class'         => 'form-control'
                ]
            ]);

            $this->add([
                'type'  => 'password',
                'name'  => 'payPassword',
                'attributes'    => [
                    'id'            => 'payPassword',
                    'class'         => 'form-control'
                ]
            ]);
        }

        $this->add([
            'type'  => 'file',
            'name'  => 'userAvatar',
            'attributes'    => [
                'id'            => 'userAvatar',
                'class'         => 'form-control'
            ]
        ]);

        $this->add([
            'type'  => 'email',
            'name'  => 'userEmail',
            'attributes'    => [
                'id'            => 'userEmail',
                'class'         => 'form-control'
            ]
        ]);

        $this->add([
            'type'  => 'text',
            'name'  => 'userBirthday',
            'attributes'    => [
                'id'            => 'userBirthday',
                'class'         => 'form-control'
            ]
        ]);

        if(count($this->phoneArea) == 1) {
            $this->add([
                'type'  => 'hidden',
                'name'  => 'areaCode',
                'attributes'=> [
                    'id'    => 'areaCode',
                    'value' => $this->phoneArea[0]['areaCode']
                ]
            ]);
        } else {
            $this->add([
                'type'  => 'select',
                'name'  => 'areaCode',
                'attributes'=> [
                    'id'    => 'areaCode',
                    'class' => 'form-control',
                    'value' => $this->phoneArea['default']
                ],
                'options' => [
                    'value_options' => $this->phoneArea['list']
                ]
            ]);
        }

        $this->add([
            'type'  => 'text',
            'name'  => 'userPhone',
            'attributes'    => [
                'id'            => 'userPhone',
                'class'         => 'form-control'
            ]
        ]);

        $this->add([
            'type'  => 'select',
            'name'  => 'userStatus',
            'attributes'    => [
                'id'            => 'userStatus',
                'class'         => 'form-control'
            ]
        ]);
    }

    protected function addInputFilter()
    {
        $inputFilter = $this->getInputFilter();

        $inputFilter->add([
            'name'      => 'userGroupId',
            'required'  => true,
            'filters'   => [
                ['name' => 'ToInt']
            ]
        ]);

        $inputFilter->add([
            'name'      => 'userSex',
            'required'  => true,
            'validators'=> [
                [
                    'name'      => 'InArray',
                    'options'   => [
                        'haystack'  => [1, 2, 3]
                    ]
                ]
            ]
        ]);

        if($this->userInfo == null) {
            $inputFilter->add([
                'name'      => 'userName',
                'required'  => true,
                'filters'   => [
                    ['name' => 'StringTrim'],
                    ['name' => 'StripTags'],
                    ['name' => 'HtmlEntities']
                ],
                'validators'=> [
                    [
                        'name'      => 'StringLength',
                        'options'   => [
                            'min'   => 1,
                            'max'   => 100
                        ]
                    ],
                    [
                        'name'      => UserExistsValidator::class,
                        'options'    => [
                            'entityManager' => $this->entityManager,
                            'userInfo'      => $this->userInfo
                        ]
                    ]
                ]
            ]);

            $inputFilter->add([
                'name'      => 'userPassword',
                'required'  => true,
                'filters'   => [
                    ['name' => 'StringTrim']
                ],
                'validators'=> [
                    [
                        'name'      => 'StringLength',
                        'options'   => [
                            'min'   => 6
                        ]
                    ]
                ]
            ]);

            $inputFilter->add([
                'name'      => 'payPassword',
                'required'  => true,
                'filters'   => [
                    ['name' => 'StringTrim']
                ],
                'validators'=> [
                    [
                        'name'      => 'StringLength',
                        'options'   => [
                            'min'   => 6
                        ]
                    ]
                ]
            ]);
        }

        $inputFilter->add([
            'type'      => 'Laminas\InputFilter\FileInput',
            'name'      => 'userAvatar',
            'required'  => false,
            'validators'=> [
                ['name' => 'FileUploadFile'],
                [
                    'name' => 'FileMimeType',
                    'options' => [
                        'mimeType' => Common::configValue('image_type')
                    ]
                ],
                ['name' => 'FileIsImage'],
                [
                    'name' => 'FileSize',
                    'options' => Common::configValue('upload_image_max')
                ]
            ],
            'filters'  => [
                [
                    'name' => 'FileRenameUpload',
                    'options' => [
                        'target' => Config::USER_AVATAR_IMAGE_PATH,
                        'useUploadName'=>true,
                        'useUploadExtension'=>true,
                        'overwrite'=>true
                    ]
                ],
                [
                    'name' => ImageThumb::class,
                    'options' => [
                        'width' => 100,
                        'height'=> 100,
                        'mode'  => 'outbound',
                        'rename'    => true,
                        'newName'   => $this->userInfo ? $this->userInfo->getUserId().'_'.time() : time()
                    ]
                ],
                [
                    'name' => OldImage::class,
                    'options' => [
                        'oldImage' => $this->userInfo ? $this->userInfo->getUserAvatar() : ''
                    ]
                ]
            ]
        ]);

        $inputFilter->add([
            'name'      => 'userEmail',
            'required'  => false,
            'filters'   => [
                ['name' => 'StringTrim']
            ],
            'validators'=> [
                [
                    'name'      => 'StringLength',
                    'options'   => [
                        'min'   => 1,
                        'max'   => 100
                    ]
                ],
                /*[
                    'name'      => 'EmailAddress',
                    'options'   => [
                        'allow'         => Hostname::ALLOW_DNS,
                        'useMxCheck'    => false
                    ]
                ],*/
                [
                    'name'      => UserEmailExistsValidator::class,
                    'options'    => [
                        'entityManager' => $this->entityManager,
                        'userInfo'      => $this->userInfo
                    ]
                ]
            ]
        ]);

        $inputFilter->add([
            'name'      => 'userBirthday',
            'required'  => false,
            'filters'   => [
                ['name' => 'StringTrim'],
                ['name' => 'StripTags'],
                ['name' => 'HtmlEntities']
            ]
        ]);

        $inputFilter->add([
            'name'      => 'userPhone',
            'required'  => false,
            'filters'   => [
                ['name' => 'StringTrim'],
                ['name' => 'StripTags'],
                ['name' => 'HtmlEntities']
            ],
            'validators'=> [
                [
                    'name' => UserPhoneExistsValidator::class,
                    'options'   => [
                        'entityManager' => $this->entityManager,
                        'userInfo'      => $this->userInfo
                    ]
                ]
            ]
        ]);

        $inputFilter->add([
            'name'      => 'userStatus',
            'required'  => true,
            'validators'=> [
                [
                    'name'      => 'InArray',
                    'options'   => [
                        'haystack'  => [1, 2, 3]
                    ]
                ]
            ]
        ]);
    }
}