<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace User\Service;

use Admin\Data\Common;
use Admin\Data\Config;
use Doctrine\ORM\EntityManager;
use User\Entity\Cart;
use User\Entity\User;
use User\Entity\UserAddress;
use User\Entity\UserGoodsFavorites;
use User\Entity\UserGroup;
use User\Entity\UserOtherLogin;

class UserManager
{
    private $entityManager;

    public function __construct(
        EntityManager $entityManager
    )
    {
        $this->entityManager = $entityManager;
    }

    /**
     * 添加客户
     * @param array $data
     * @return User
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addUser(array $data)
    {
        $userGroup = $this->entityManager->getRepository(UserGroup::class)->findOneByUserGroupId($data['userGroupId']);

        if(!empty($data['userAvatar']['tmp_name'])) $data['userAvatar'] = str_replace(Config::PUBLIC_PATH, '', $data['userAvatar']['tmp_name']);
        else unset($data['userAvatar']);

        $user = new User();
        $user->valuesSet($data);
        $user->setUserMoney(0.00);
        $user->setIntegralType1Num(0);
        $user->setIntegralType2Num(0);
        $user->setUserAddTime(time());
        $user->setOneUserGroup($userGroup);

        $this->entityManager->persist($user);
        $this->entityManager->flush();

        return $user;
    }

    /**
     * 编辑客户
     * @param array $data
     * @param User $user
     * @return User
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editUser(array $data, User $user)
    {
        unset($data['userName']);

        if(!empty($data['userAvatar']['thumbImage'])) $data['userAvatar'] = str_replace(Config::PUBLIC_PATH, '', $data['userAvatar']['thumbImage']);
        else unset($data['userAvatar']);

        $user->valuesSet($data);
        $this->entityManager->flush();
        return $user;
    }

    /**
     * 编辑会员余额
     * @param $userMoney
     * @param User $user
     * @return User
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function updateUserMoney($userMoney, User $user)
    {
        $user->setUserMoney($userMoney);
        $this->entityManager->flush();

        return $user;
    }

    /**
     * 编辑会员积分
     * @param array $data
     * @param User $user
     * @return User
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function updateUserIntegral(array $data, User $user)
    {
        if(isset($data['integralType1Num'])) {
            $user->setIntegralType1Num($data['integralType1Num']);
        }
        if(isset($data['integralType2Num'])) {
            $user->setIntegralType2Num($data['integralType2Num']);
        }
        $this->entityManager->flush();

        return $user;
    }

    /**
     * 修改电子邮件
     * @param $email
     * @param User $user
     * @return bool
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function changeUserEmail($email, User $user)
    {
        if($email != $user->getUserEmail()) {
            $user->setUserEmail($email);
            $this->entityManager->flush();
        }
        return true;
    }

    /**
     * 修改会员组id
     * @param $userGroupId
     * @param User $user
     * @return bool
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function changeUserGroupId($userGroupId, User $user)
    {
        if ($userGroupId != $user->getUserGroupId()) {
            $user->setUserGroupId($userGroupId);
            $this->entityManager->flush();
        }
        return true;
    }

    /**
     * 修改手机号码
     * @param array $data
     * @param User $user
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function changeUserPhone(array $data, User $user)
    {
        $user->setAreaCode($data['areaCode']);
        $user->setUserPhone($data['userPhone']);

        $this->entityManager->flush();
    }

    /**
     * 修改登录密码
     * @param $password
     * @param User $user
     * @return bool
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function changeUserPassword($password, User $user)
    {
        if(!Common::verifyUserPassword($password, $user->getUserPassword())) {
            $user->setUserPassword($password);
            $this->entityManager->flush();
        }
        return true;
    }

    /**
     * 修改余额支付密码
     * @param $password
     * @param User $user
     * @return bool
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function changePayPassword($password, User $user)
    {
        if(!Common::verifyPayPassword($password, $user->getPayPassword())) {
            $user->setPayPassword($password);
            $this->entityManager->flush();
        }
        return true;
    }

    /**
     * 删除客户
     * @param User $user
     * @return bool
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteUser(User $user)
    {
        $userId = $user->getUserId();
        $this->entityManager->remove($user);
        $this->entityManager->flush();

        $qb = $this->entityManager->createQueryBuilder();
        $qb->delete(UserAddress::class, 'a')->where('a.userId = :userId')->setParameter('userId', $userId)->getQuery()->execute();
        $qb->delete(Cart::class, 'd')->where('d.userId = :userId')->setParameter('userId', $userId)->getQuery()->execute();
        $qb->delete(UserGoodsFavorites::class, 'f')->where('f.userId = :userId')->setParameter('userId', $userId)->getQuery()->execute();
        $qb->delete(UserOtherLogin::class, 'o')->where('o.userId = :userId')->setParameter('userId', $userId)->getQuery()->execute();

        return true;
    }
}