<?php

class Http
{
    /**
     * 请求方法
     */
    const GET = 'GET';
    const POST = 'POST';

    /**
     * 请求的目标路径
     *
     * @var string
     */
    private $target;

    /**
     * 被请求资源的主机名
     *
     * @var string
     */
    private $host;

    /**
     * 被请求资源的主机端口
     *
     * @var integer
     */
    private $port = 0;

    /**
     * 封装协议
     *
     * @var string
     */
    private $schema = 'http';

    /**
     * 请求的方法
     *
     * @var string
     */
    private $method = self::GET;

    /**
     * 请求的参数
     *
     * @var array
     */
    private $params = array();

    /**
     * 请求的cookie参数
     *
     * @var array
     */
    private $cookies = array();

    /**
     * 超时时间
     *
     * @var integer
     */
    private $timeout = 30;

    /**
     * 请求的源资源
     *
     * @var string
     */
    private $referrer;

    /**
     * 请求的软件类型
     *
     * @var string
     */
    private $userAgent = 'Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.2; .NET CLR 1.1.4322; .NET CLR 2.0.50727; NBCHINA 2.0)';

    /**
     * 是否启用gzip压缩
     *
     * @var boolean
     */
    private $gzip = false;

    /**
     * 客户端可以处理的媒体类型，按优先级排序
     *
     * @var string
     */
     private $accept = '*/*';
    //private $accept = 'text/xml,application/xml,application/xhtml+xml,text/html,text/plain,image/png,image/jpeg,image/gif,*/*';

    /**
     * 客户端可以理解的编码机制
     *
     * @var string
     */
    private $acceptEncoding = 'gzip, deflate';

    /**
     * 客户端乐于接受的自然语言列表 en,de
     *
     * @var string
     */
    private $acceptLanguage = 'zh-cn';

    /**
     * 用户名
     *
     * @var string
     */
    private $username;

    /**
     * 口令
     *
     * @var string
     */
    private $password;

    /**
     * 请求结果
     *
     * @var string
     */
    private $result;

    /**
     * 头部信息
     *
     * @var array
     */
    private $headers = array();

    /**
     * 状态码
     *
     * @var string
     */
    private $status = 0;

    /**
     * 是否重定向
     *
     * @var boolean
     */
    private $redirect = TRUE;

    /**
     * 最大重定向次数
     *
     * @var integer
     */
    private $maxRedirect = 3;

    /**
     * 当前重定向次数
     *
     * @var integer
     */
    private $selfRedirect = 0;

    /**
     * 请求参数的字符串
     *
     * @var string
     */
    private $queryString;

    /**
     * 请求的cookie字符串
     *
     * @var string
     */
    private $cookieString;

    /**
     * 构造函数
     *
     * @access public
     * @return void
     */
    public function __construct($target = '')
    {
        if ($target != '')
        {
            $this->target = $target;
        }
    }

    /**
     * 析构函数
     *
     * @access public
     * @return void
     */
    public function __destruct()
    {
    }

    public static function quickGet($target, $referrer = '', $data = array())
    {
        return Http::quickRequest($target, $referrer, Http::GET, $data);
    }
    public static function quickPost($target, $referrer = '', $data = array())
    {
        return Http::quickRequest($target, $referrer, Http::POST, $data);
    }

    public static function quickRequest($target, $referrer, $method, $data = array())
    {
        $http = new Http();
        $http->setMethod($method);
        $http->setTarget($target);
        $http->setReferrer($referrer);
        if (is_array($data) && count($data) >0)
        {
            foreach ($data as $key => $value)
            {
                $http->addParam($key, $value);
            }
        }
        $http->sendRequest();
        return $http->getResult();
    }

    public function sendRequest()
    {
        $this->initRequest();
        $requestString = $this->buildRequestString();
//dump($requestString);
        $fp = fsockopen($this->host, $this->port, $errorNumber, $errorString, $this->timeout);
        if (!$fp)
        {
            $this->throwException('试图打开socket连接时发生异常：' . $errorString . ' (' . $errorNumber . ')');
        }

        fwrite($fp, $requestString);

        // HTTP的头部信息是以\r\n\r\n结束的
        do
        {
            $responseHeaderString .= fread($fp, 1);
        }
        while (!preg_match('/\\r\\n\\r\\n$/', $responseHeaderString));

        //dump($responseHeaderString, '$responseHeaderString');

        $this->parseHeaders($responseHeaderString);

        // 解析返回的cookie
        if (isset($this->headers['set-cookie']))
        {
            $cookies = $this->headers['set-cookie'];
            if (!is_array($cookies))
            {
                $cookies = array($cookies);
            }

            foreach ($cookies as $cookie)
            {
                if (preg_match('/([^=]+)=([^;]+);/', $cookie, $m))
                {
                    $this->cookies[$m[1]] = $m[2];
                }
            }
        }

        // 重定向
        if ($this->status == '302' && $this->redirect)
        {
            if ($this->selfRedirect >= $this->maxRedirect) { $this->throwException('重定向次数太多!'); }

            $parsed = parse_url($this->headers['location']);

            if ($parsed['host'])
            {
                $target = $this->headers['location'];
            }
            else
            {
                $target = $this->schema . '://' . $this->host . '/' . $this->headers['location'];
            }

            $this->port     = 0;
            $this->status   = 0;
            $this->params   = array();
            $this->method   = self::GET;
            $this->referrer = $this->target;
            $this->target   = $target;
            $this->selfRedirect++;
            $this->sendRequest();
        }

        $responseHeaderString = '';
        $responseContentString = '';

        if ($this->headers['transfer-encoding'] != 'chunked')
        {
            while (!feof($fp))
            {
                $responseContentString .= fgets($fp, 128);
            }
        }
        else
        {
            while ($chunkLength = hexdec(fgets($fp)))
            {
                $responseContentChunk = '';
                $readLength = 0;
                while ($readLength < $chunkLength)
                {
                    $responseContentChunk .= fread($fp, $chunkLength - $readLength);
                    $readLength = strlen($responseContentChunk);
                }

                $responseContentString .= $responseContentChunk;
                fgets($fp);
            }
        }

        $this->result = chop($responseContentString);

        // gzip解码
        if (isset($this->headers['content-encoding']) && $this->headers['content-encoding'] == 'gzip')
        {
            $this->result = substr($this->result, 10);
            $this->result = gzinflate($this->result);
        }

    }

    /**
     * 解析返回的头信息
     *
     * @access private
     * @param string $headerString
     * @return void
     */
    private function parseHeaders($headerString)
    {
        $headers = explode("\r\n", $headerString);
        $this->headers = array();

        // 状态
        if ($this->status == 0)
        {
            if (!eregi($match = "(^http/[0-9]+\\.[0-9])+[ \t]+([0-9]+)[ \t]*(.*)\$", $headers[0], $matches))
            {
                $this->throwException('Unexpected HTTP response status');
            }

            $this->status = $matches[2];
            array_shift($headers);
        }

        foreach ($headers as $key => $value)
        {
            if (preg_match('/([^:]+):\\s*(.*)/', $value, $m))
            {
                $k = strtolower(trim($m[1]));
                $v = trim($m[2]);
                if (isset($this->headers[$k]))
                {
                    if (is_array($this->headers[$k]))
                    {
                        $this->headers[$k][] = $v;
                    }
                    else
                    {
                        $this->headers[$k] = array($this->headers[$k], $v);
                    }
                }
                else
                {
                    $this->headers[$k] = $v;
                }
            }
        }
    }

    /**
     * 初始化请求
     *
     * @access private
     * @return void
     */
    private function initRequest()
    {
        // 请求参数
        $this->queryString = $this->buildQueryString($this->params);

        // 如果请求方式为GET,则将参数附加至请求路径
        if (($this->method == self::GET) && $this->queryString != '')
        {
            $this->target = $this->target . '?' . $this->queryString;
        }

        $parsed = parse_url($this->target);

        // 处理ssl连接
        if ($parsed['scheme'] == 'https')
        {
            $this->host = 'ssl://' . $parsed['host'];
            $this->port = ($this->port != 0) ? $this->port : 443;
        }
        else
        {
            $this->host = $parsed['host'];
            $this->port = ($this->port != 0) ? $this->port : 80;
        }

        // 最终的请求路径
        $this->path = (isset($parsed['path'])
                        ? $parsed['path']
                        : '/') . (isset($parsed['query'])
                                    ? '?' . $parsed['query']
                                    : '');
        $this->schema = $parsed['scheme'];

        //cookie参数
        $this->cookieString = $this->buildQueryString($this->cookies);
    }

    /**
     * 构造请求字符串
     *
     * @access private
     * @return string
     */
    private function buildRequestString()
    {
        $requestString  = '';
        $requestString .= $this->method . ' ' . $this->path . " HTTP/1.1\r\n";
        $requestString .= 'Host: ' . $this->host . "\r\n";
        $requestString .= 'User-Agent: ' . $this->userAgent . "\r\n";
        $requestString .= 'Accept: ' . $this->accept . "\r\n";
        if ($this->useGzip)
        {
            if (function_exists('gzinflate'))
            {
                $requestString .= "Accept-Encoding: gzip, deflate\r\n";
            }
            else
            {
                $requestString .= "Accept-Encoding: identity\r\n";
            }
        }
        $requestString .= 'Accept-Language: ' . $this->acceptLanguage . "\r\n";
        $requestString .= "Content-Type: application/x-www-form-urlencoded\r\n";

        if ($this->cookieString != '')
        {
            $requestString .= 'Cookie: ' . $this->cookieString . "\r\n";
        }

        if ($this->method == self::POST)
        {
            $requestString .= 'Content-Length: ' . strlen($this->queryString) . "\r\n";
        }

        if ($this->referrer != '')
        {
            $requestString .= 'Referer: ' . $this->referrer . "\r\n";
        }

        if ($this->username && $this->password)
        {
            $requestString .= 'Authorization: Basic '
                            . base64_encode($this->username . ':' . $this->password) . "\r\n";
        }

        $requestString .= "Connection: close\r\n\r\n";

        if ($this->method == self::POST)
        {
            $requestString .= $this->queryString;
        }

        return $requestString;
    }

    /**
     * 将数组参数转化成字符串
     *
     * @access private
     * @param array $date   参数数组
     * @return string
     */
    private function buildQueryString($data)
    {
        $queryString = '';

        if (is_array($data) && count($data) > 0)
        {
            $tempQueryString = array();

            foreach ($data as $key => $value)
            {
                if (strlen(trim($value)) > 0)
                {
                    $tempQueryString[] = $key . '=' . urlencode($value);
                }
            }

            $queryString = join('&', $tempQueryString);
        }

        return $queryString;
    }

    /**
     * 设置请求的目标
     *
     * @access public
     * @param string $target
     * @return void
     */
    public function setTarget($target)
    {
        $this->target = (string) $target;
    }

    /**
     * 返回请求的目标
     *
     * @access public
     * @return string
     */
    public function getTarget()
    {
        return (string) $this->target;
    }

    /**
     * 设置schema
     *
     * @access public
     * @param string $schema
     * @return void
     */
    public function setSchema($schema)
    {
        $this->schema = (string) $schema;
    }

    /**
     * 设置请求方法
     *
     * @access public
     * @param string $method  Http::GET/Http::POST
     * @return void
     */
    public function setMethod($method = self::GET)
    {
        $this->method = ($method == self::POST) ? self::POST : self::GET;
    }

    /**
     * 设置请求来源
     *
     * @access public
     * @param string $referrer
     * @return void
     */
    public function setReferrer($referrer)
    {
        $this->referrer = (string) $referrer;
    }

    /**
     * 设置超时时间
     *
     * @access public
     * @param integer $timeout
     * @return void
     */
    public function setTimeout($timeout)
    {
        $this->timeout = (integer) $timeout;
    }

    /**
     * 设置请求的软件类型
     *
     * @access public
     * @param string $timeout
     * @return void
     */
    public function setUserAgent($userAgent)
    {
        $this->userAgent = (string) $userAgent;
    }

    /**
     * 设置是否启用gzip压缩
     *
     * @access public
     * @param boolean $gzip true/false
     * @return void
     */
    public function setGzip($gzip)
    {
        $this->gzip = (boolean) $gzip;
    }

    /**
     * 设置用户名
     *
     * @access public
     * @param string $username
     * @return void
     */
    public function setUsername($username)
    {
        $this->username = (string) $username;
    }

    /**
     * 设置密码
     *
     * @access public
     * @param string $password
     * @return void
     */
    public function setPassword($password)
    {
        $this->password = (string) $password;
    }

    /**
     * 设置是否启用重定向
     *
     * @access public
     * @param boolean $redirect true/false
     * @return void
     */
    public function setRedirect($redirect)
    {
        $this->redirect = (boolean) $redirect;
    }

    /**
     * 设置最大重定向次数
     *
     * @access public
     * @param integer $maxRedirect
     * @return void
     */
    public function setMaxRedirect($maxRedirect)
    {
        $this->maxRedirect = (integer) $maxRedirect;
    }

    /**
     * 返回当前重定向次数
     *
     * @access public
     * @return integer
     */
    public function getSelfRedirect()
    {
        return (integer) $this->selfRedirect;
    }

    /**
     * 添加请求参数
     *
     * @access public
     * @param string $name   参数的键
     * @param string $value  参数的值
     * @return void
     */
    public function addParam($name, $value)
    {
        if (!empty($name) && !empty($value))
        {
            $this->params[$name] = (string) $value;
        }
    }

    /**
     * 增加一个cookie到请求参数中
     *
     * @access public
     * @param string $name   参数名称
     * @param string $value  参数的值
     * @return void
     */
    public function addCookie($name, $value)
    {
        if (!empty($name) && !empty($value))
        {
            $this->cookies[$name] = (string) $value;
        }
    }

    /**
     * 返回头部信息
     *
     * @access public
     * @return array
     */
    public function getHeaders()
    {
        return (array) $this->headers;
    }

    /**
     * 返回指定的头部信息
     *
     * @access public
     * @param string $header
     * @return string
     */
    public function getHeader($header)
    {
        $header = strtolower($header);
        if (isset($this->headers[$header]))
        {
            return $this->headers[$header];
        }
        else
        {
            return false;
        }
    }

    /**
     * 返回应答状态
     *
     * @access public
     * @return integer
     */
    public function getStatus()
    {
        return (integer) $this->status;
    }

    /**
     * 返回结果
     *
     * @access public
     * @return string
     */
    public function getResult()
    {
        return (string) $this->result;
    }

    /**
     * 返回cookies
     *
     * @access public
     * @return array
     */
    public function getCookies()
    {
        return (array) $this->cookies;
    }

    /**
     * 抛出异常
     *
     * @access private
     * @param string $message 错误信息
     * @return string
     */
    private function throwException($message)
    {
        self_ob_clean();
        @header('Content-Type: text/html; charset=utf-8');
        //throw new Exception($message);
        echo "\r\n<br />";
        echo $message;
        echo "\r\n<br />";
        exit;
    }

}

?>