<?php
/**
 * MySQL数据库操作类
 */

if (!defined('__IN_SY__')) { exit('Access Denied'); }

if (!defined('DB_PARAM_BOOL')) { define('DB_PARAM_BOOL', PDO::PARAM_BOOL); }
if (!defined('DB_PARAM_NULL')) { define('DB_PARAM_NULL', PDO::PARAM_NULL); }
if (!defined('DB_PARAM_INT')) { define('DB_PARAM_INT', PDO::PARAM_INT); }
if (!defined('DB_PARAM_STR')) { define('DB_PARAM_STR', PDO::PARAM_STR); }
if (!defined('DB_PARAM_LOB')) { define('DB_PARAM_LOB', PDO::PARAM_LOB); }
if (!defined('DB_PARAM_STMT')) { define('DB_PARAM_STMT', PDO::PARAM_STMT); }
if (!defined('DB_PARAM_INPUT_OUTPUT')) { define('DB_PARAM_INPUT_OUTPUT', PDO::PARAM_INPUT_OUTPUT); }

include_once('SqlParameter.class.php');

class MySQL
{
    /**
     * 版本
     */
    const VERSION = '1.0.0';

    /**
     * pdo对象
     *
     * @var PDO
     */
    private $_pdo;

    /**
     * 查询对象
     *
     * @var PDOStatement
     */
    private $_stmt;

    /**
     * 数据库编码
     *
     * @var string
     */
    private $_charset = 'utf8';

    /**
     * 数据库查询次数
     *
     * @var integer
     */
    private $_num_queries = 0;

    /**
     * 最后一次查询语句
     *
     * @var string
     */
     private $_last_query;

    /**
     * 开始执行时间
     *
     * @var float
     */
     private $_timer_start;

    /**
     * 停止执行时间
     *
     * @var float
     */
     private $_timer_stop;

    /**
     * 构造函数
     *
     * @access public
     * @return void
     */
    public function __construct($db_host = '', $db_user = '', $db_password = '', $db_name = '', $db_pconnect = false)
    {
        try
        {
            if (!extension_loaded('pdo')) { throw new Exception('PDO extension not loaded!'); }

            $db_host     = ($db_host)     ? $db_host     : __DB_HOST__;
            $db_user     = ($db_user)     ? $db_user     : __DB_USER__;
            $db_password = ($db_password) ? $db_password : __DB_PASSWORD__;
            $db_name     = ($db_name)     ? $db_name     : __DB_NAME__;
            $db_pconnect = ($db_pconnect) && __DB_PCONNECT__ ? true : false;

            $dsn = 'mysql:host=' . $db_host . ';dbname=' . $db_name;

            $this->_pdo = new PDO($dsn, $db_user, $db_password);
        }
        catch (PDOException $e)
        {
            $this->_throwException($e->getMessage());
        }
    }

    /**
     * 析构函数
     *
     * @access public
     * @return void
     */
    public function __destruct()
    {
        @$this->close();
    }

    /**
     * 设置编码
     *
     * @access public
     * @param string $charset 编码
     * @return void
     */
    public function setCharset($charset)
    {
        //$this->_charset = $charset;
        $this->_pdo->query("SET NAMES $charset;");
        $this->_pdo->query("SET sql_mode='';");
    }

    /**
     * 设置超时时间
     *
     * @access public
     * @param string $timeout 超时时间
     * @return void
     */
    public function setTimeout($timeout)
    {
        $this->_pdo->setAttribute(PDO::ATTR_TIMEOUT, $timeout);
    }

    /**
     * 返回查询次数
     *
     * @access public
     * @return integer
     */
    public function getQueries()
    {
        return (integer) $this->_num_queries;
    }

    /**
     * 返回最后一次查询SQL语句
     *
     * @access public
     * @return string
     */
    public function getLastQuery()
    {
        return (string) $this->_last_query;
    }

    /**
     * 返回mysql版本
     *
     * @access public
     * @return string
     */
    public function getVersion($b = FALSE)
    {
        $version = $this->_pdo->getAttribute(PDO::ATTR_SERVER_VERSION);
        return $b ? $version : str_replace('-community-nt', '', trim($version));
    }

    /**
     * 开始事务处理
     *
     * @access public
     * @return string
     */
    public function beginTransaction()
    {
        $this->_pdo->beginTransaction();
    }

    /**
     * 提交事务处理
     *
     * @access public
     * @return string
     */
    public function commit()
    {
        $this->_pdo->commit();
    }

    /**
     * 回滚事务处理
     *
     * @access public
     * @return string
     */
    public function rollBack()
    {
        $this->_pdo->rollBack();
    }

    /**
     * 返回受影响的行数
     *
     * @access public
     * @param string $sql sql语句
     * @param ArrayObject $params 查询参数SqlParameter的集合
     * @param boolean $insertid
     * @return void
     */
    public function executeNonQuery($sql, $params = NULL, $insertid = TRUE)
    {
        $this->_executeQuery($sql, $params);
        $result = $this->_stmt->rowCount();

        if ($insertid)
        {
            $result = $this->_pdo->lastInsertId() > 0 ? $this->_pdo->lastInsertId() : $result;
        }

        return $result;
    }

    /**
     * 返回查询的数据集
     *
     * @access public
     * @param string $sql         sql语句
     * @param ArrayObject $params 查询参数SqlParameter的集合
     * @param boolean $singleRow  是否只获取一条符合条件的记录
     * @return ResultSet
     */
    public function executeDateSet($sql, $params = NULL, $singleRow = FALSE)
    {
        if ($singleRow)
        {
            if (!eregi("limit", $sql)) { $sql = eregi_replace("[,;]$", "", trim($sql)) . " LIMIT 0,1;"; }
        }

        // 设置属性
        $this->_setAttribute();
        $this->_executeQuery($sql, $params);
        // 关联数组形式返回结果集
        $this->_stmt->setFetchMode(PDO::FETCH_ASSOC);
        $result = $this->_stmt->fetchAll();

        if ($result)
        {
            require_once('ResultSet.class.php');
            return new ResultSet($result);
        }

        return false;
    }

    /**
     * 返回查询的数据集
     *
     * @access public
     * @param string $sql         sql语句
     * @param ArrayObject $params 查询参数SqlParameter的集合
     * @param boolean $singleRow  是否只获取一条符合条件的记录
     * @return array
     */
    public function executeReader($sql, $params = NULL, $singleRow = FALSE)
    {
        if ($singleRow)
        {
            if (!eregi("limit", $sql)) { $sql = eregi_replace("[,;]$", "", trim($sql)) . " LIMIT 0,1;"; }
        }

        // 设置属性
        $this->_setAttribute();
        $this->_executeQuery($sql, $params);
        // 关联数组形式返回结果集
        $this->_stmt->setFetchMode(PDO::FETCH_ASSOC);
        return $singleRow ? $this->_stmt->fetch() : $this->_stmt->fetchAll();
    }

    /**
     * 返回第一行第一列的值
     *
     * @access public
     * @param string $sql         sql语句
     * @param ArrayObject $params 查询参数SqlParameter的集合
     * @return mixed
     */
    public function executeScalar($sql, $params = NULL)
    {
        $this->_executeQuery($sql, $params);
        $result = $this->_stmt->fetchColumn();

        return $result;
    }

    /**
     * 向表中添加数据
     *
     * @access public
     * @param string $tableName  表名
     * @param array $data        数据
     * @param boolean $isReplace 是否删除旧有记录
     * @return mixed
     */
    public function insert($tableName, $data, $isReplace = FALSE)
    {
        if (!is_array($data)) { $this->_throwException('$data参数必须为数组！'); }

        $i = 1; $str_fields = $str_values = $comma = '';
        $params = new ArrayObject();

        foreach ($data as $field => $value)
        {
            $str_fields .= $comma . $this->antiQuote($field);
            $str_values .= $comma . '?';
            $params->append($this->getParam($tableName, $field, $i++, $value));
            $comma = ', ';
        }

        $method = $isReplace ? 'REPLACE' : 'INSERT';
        $sql = $method . ' INTO ' . $tableName . ' (' . $str_fields . ') VALUES (' . $str_values . ')';
        return $this->executeNonQuery($sql, $params);
    }

    /**
     * 选择数据
     *
     * @access public
     * @param string $tableName  表名
     * @param mixed  $condition  条件
     * @param mixed  $columns    字段
     * @param boolean $singleRow 是否只取一条记录
     * @return array
     */
    public function select($tableName, $condition = array(), $columns = '', $singleRow = FALSE)
    {
        $i = 1; $str_fields = $str_where = $comma = $expr = '';
        $params = new ArrayObject();

        if (empty($columns))
        {
            $str_fields = '*';
        }
        elseif (is_array($columns))
        {
            foreach ($columns as $key => $value)
            {
                $str_fields .= $comma . $this->antiQuote($value);
                $comma = ', ';
            }
        }
        elseif (is_string($columns))
        {
            $str_fields = $columns;
        }
        else
        {
            $str_fields = '*';
        }

        if (empty($condition))
        {
            $str_where = '1';
        }
        elseif (is_array($condition))
        {
            $comma = '';
            if (array_key_exists('expr', $condition))
            {
                $expr = strtoupper(trim($condition['expr']));
                unset($condition['expr']);
            }
            foreach ($condition as $field => $value)
            {
                $str_where .= $comma . $this->antiQuote($field) . ' = ?';
                $params->append($this->getParam($tableName, $field, $i++, $value));
                $comma = ($expr == 'OR') ? ' OR ' : ' AND ';
            }
        }
        elseif (is_string($condition))
        {
            $str_where = $condition;
        }
        else
        {
            $str_where = '1';
        }

        $sql = 'SELECT ' . $str_fields . ' FROM ' . $tableName . ' WHERE ' . $str_where;
        //dump($sql);
        return $this->executeReader($sql, $params, $singleRow);
    }

    /**
     * 更新表中数据
     *
     * @access public
     * @param string $tableName  表名
     * @param array $data        数据
     * @param boolean $condition 条件
     * @return mixed
     */
    public function update($tableName, $data, $condition)
    {
        if (empty($tableName) || !is_array($data)) { $this->_throwException('请检查参数'); }

        $i = 1; $str_fields = $str_where = $comma = '';
        $params = new ArrayObject();

        foreach ($data as $field => $value)
        {
            $str_fields .= $comma . $this->antiQuote($field) . ' = ?';
            $params->append($this->getParam($tableName, $field, $i++, $value));
            $comma = ', ';
        }

        if (is_array($condition))
        {
            $comma = '';
            foreach ($condition as $field => $value)
            {
                $str_where .= $comma . $this->antiQuote($field) . ' = ?';
                $params->append($this->getParam($tableName, $field, $i++, $value));
                $comma = ' AND ';
            }
        }
        else
        {
            $str_where = $condition;
        }

        $sql = 'UPDATE ' . $tableName . ' SET ' . $str_fields . ' WHERE ' . $str_where;
        return $this->executeNonQuery($sql, $params);
    }

    /**
     * 删除表中数据
     * $expr = array(
     *             '字段1' => 值1,
     *             '字段2' => 值2,
     *             'expr' => 'or' // 默认为 and
     * );
     * $this->delete('表名', $expr);
     * @access public
     * @param string $tableName  表名
     * @param boolean $condition 条件
     * @return mixed
     */
    public function delete($tableName, $condition)
    {
        $i = 1; $str_where = $comma = $expr = '';
        $params = new ArrayObject();

        if (empty($tableName) || empty($condition)) { $this->_throwException('请检查参数。'); }

        if (is_array($condition))
        {
            $comma = '';
            if (array_key_exists('expr', $condition))
            {
                $expr = strtoupper(trim($condition['expr']));
                unset($condition['expr']);
            }
            foreach ($condition as $field => $value)
            {
                $str_where .= $comma . $this->antiQuote($field) . ' = ?';
                $params->append($this->getParam($tableName, $field, $i++, $value));
                $comma = ($expr == 'OR') ? ' OR ' : ' AND ';
            }
        }
        else
        {
            $str_where = $condition;
        }

        $sql = 'DELETE FROM ' . $tableName . ' WHERE ' . $str_where;
        return $this->executeNonQuery($sql, $params);
    }

    /**
     * 返回数量
     *
     * @access public
     * @param string $tableName 表名
     * @param mixed  $condition 条件
     * @param string $columns 字段
     * @return integer
     */
    public function getScalar($tableName, $condition, $columns = 'COUNT(*)')
    {
        $columns = $columns . ' AS `scalar`';
        $result = $this->select($tableName, $condition, $columns, TRUE);
        return $result['scalar'];
    }

    /**
     * 更新表字段的值
     *
     * @access public
     * @param string $tableName  表名
     * @param array/string $data 数据
     * @param boolean $condition 条件
     * @return mixed
     */
    public function auto($tableName, $data, $condition)
    {
        if (empty($tableName) || !is_array($data)) { $this->_throwException('请检查参数'); }

        $i = 1; $str_fields = $str_where = $comma = '';
        $params = new ArrayObject();

        foreach ($data as $field => $value)
        {
            $_hyphen = substr($value, 0, 1) == '-' ? ' - ' : ' + ';
            $str_fields .= $comma . $this->antiQuote($field)
                        . ' = ' . $this->antiQuote($field) . $_hyphen . '?';
            $params->append($this->getParam($tableName, $field, $i++, abs($value)));
            $comma = ', ';
        }

        if (is_array($condition))
        {
            $comma = '';
            foreach ($condition as $field => $value)
            {
                $str_where .= $comma . $this->antiQuote($field) . ' = ?';
                $params->append($this->getParam($tableName, $field, $i++, $value));
                $comma = ' AND ';
            }
        }
        else
        {
            $str_where = $condition;
        }

        $sql = 'UPDATE ' . $tableName . ' SET ' . $str_fields . ' WHERE ' . $str_where;
        return $this->executeNonQuery($sql, $params);
    }

    /**
     * 返回表信息文件
     *
     * @access public
     * @param string $tableName 表名
     * @return string
     */
    public function getColumns($tableName)
    {
        $file = __DATA__ . '/table/' . $tableName . '.php';
        if (!file_exists($file))
        {
            $arr_temp = array();
            $result = $this->executeReader('show columns from ' . $tableName . ';');
            foreach ($result as $key => $column)
            {
                $retval = array('datatype' => DB_PARAM_STR, 'length' => NULL);
                $type = strtolower($column['type']);
                switch ($type)
                {
                    case preg_match("/^(tinyint|smallint|mediumint|int|integer|bigint)/i", $type) == 1 :
                        $retval['datatype'] = DB_PARAM_INT;
                        $retval['length']   = NULL;
                        break;
                    case preg_match_all("/^(char|varchar)\((\d+)\)/i", $type, $m) > 0 :
                        $retval['datatype'] = DB_PARAM_STR;
                        $retval['length']   = (integer) $m[2][0];
                        break;
                    case preg_match("/^(char|varchar|text|longtext|tinytext|mediumtext)/i", $type, $m) == 1 :
                        $retval['datatype'] = DB_PARAM_STR;
                        $retval['length']   = NULL;
                        break;
                    case preg_match_all("/^(date|time|year|datetime|timestamp)/i", $type) == 1 :
                        $retval['datatype'] = DB_PARAM_STR;
                        $retval['length']   = NULL;
                        break;
                    case preg_match("/^(tinyblob|blob|mediumblob|longblob|enum|set)/i", $type) == 1 :
                        $retval['datatype'] = DB_PARAM_LOB;
                        $retval['length']   = NULL;
                        break;
                    default :
                        $retval['datatype'] = DB_PARAM_STR;
                        $retval['length']   = NULL;
                        break;
                }
                $arr_temp[$column['field']] = $retval;
            }
            $content = "<?php\r\nif (!defined('__IN_SY__')) { exit('Access Denied'); }\r\n\$table_$tableName = "
                        . var_export($arr_temp, TRUE) . ";\n?>";
            file_put_contents($file, $content);
            @chmod($file, 0777);
        }
        return format_dir($file);
    }

    /**
     * 返回SqlParameter
     *
     * @access public
     * @param string $tableName 表名
     * @param string $columnName 字段名
     * @param string $idx 参数
     * @param string $value 参数值
     * @return array
     */
    public function getParam($tableName, $columnName, $idx, $value)
    {
        global ${'table_' . $tableName};
        include_once($this->getColumns($tableName));
        $arr_temp = ${'table_' . $tableName}[$columnName];
        if ($arr_temp['datatype'] == DB_PARAM_STR) { $value = self_stripslashes($value); }
        return new SqlParameter($idx, $value, $arr_temp['datatype'], $arr_temp['length']);
    }

    /**
     * 为字符串添加引号
     *
     * @access public
     * @param mixed $string
     * @param int $parameter_type
     * @return mixed
     */
    public function quote($string, $parameter_type = PDO::PARAM_STR)
    {
        if (is_instance_of($this->_pdo, 'PDO'))
        {
            return $this->_pdo->quote($string, $parameter_type);
        }
        else
        {
            return $string;
        }
    }

    /**
     * 为字符串添加反引号
     *
     * @access public
     * @param mixed $string 需要添加反引号的字符串
     * @return string
     */
    public function antiQuote($string)
    {
        if (is_string($string))
        {
            return '`' . $string . '`';
        }
        else
        {
            return $string;
        }
    }

    /**
     * 开始执行时间，调试用
     *
     * @access public
     * @return true
     */
    public function timerStart()
    {
        $this->_timer_start = microtime(TRUE);
        return TRUE;
    }

    /**
     * 执行结束时间，调试用
     *
     * @access public
     * @return true
     */
    public function timerStop()
    {
        $this->_timer_stop = microtime(TRUE);
        return TRUE;
    }

    /**
     * 执行时间，调试用
     *
     * @access public
     * @return float
     */
    public function timerTotal()
    {
        return $this->_timer_stop - $this->_timer_start;;
    }

    /**
     * 关闭连接
     *
     * @access public
     * @return void
     */
    public function close()
    {
        $this->_pdo = NULL;
    }

    /**
     * 执行SQL
     *
     * @access private
     * @param string $sql         sql语句
     * @param ArrayObject $params 参数
     * @return void
     */
    private function _executeQuery($sql, $params = NULL)
    {
        if (empty($sql)) { $this->_throwException('SQL can not be empty!'); }

        if (substr(trim($sql), -1) != ';') { $sql .= ';'; }
//dump($sql);
        $this->_last_query = $sql;
        // 生成一个查询对象
        $this->_stmt = $this->_pdo->prepare($sql);
        $this->_bindParam($params);
        $this->_stmt->execute();
        $this->_num_queries++;
    }

    /**
     * 设置属性
     *
     * @access private
     * @return void
     */
    private function _setAttribute()
    {
        // 设置错误处理方式为抛出异常
        $this->_pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        // 设置返回的字段名称为小写
        $this->_pdo->setAttribute(PDO::ATTR_CASE, PDO::CASE_LOWER);
        // 强制将数据库返回的空字符转换成null
        $this->_pdo->setAttribute(PDO::ATTR_ORACLE_NULLS, PDO::NULL_EMPTY_STRING);
    }

    /**
     * 绑定预处理占位符的值
     *
     * @access private
     * @param ArrayObject $params 参数
     * @return void
     */
    private function _bindParam($params = null)
    {
        if (!is_instance_of($this->_stmt, 'PDOStatement')) { return false; }
        //dump($params);
        if (is_instance_of($params, 'ArrayObject') && $params->count() > 0)
        {
            $iterator = $params->getIterator();
            for ($iterator->rewind(); $iterator->valid(); $iterator->next())
            {
                $param = $iterator->current();
                if (!is_null($param->length))
                {
                    $this->_stmt->bindParam($param->id, $param->variable, $param->datatype, $param->length);
                }
                else
                {
                    $this->_stmt->bindParam($param->id, $param->variable, $param->datatype);
                }
            }
        }
    }

    /**
     * 返回上一个MySQL操作中的错误信息
     *
     * @access public
     * @return array
     */
    public function errorInfo()
    {
        return (array) ($this->_pdo) ? $this->_pdo->errorInfo() : array();
    }

    /**
     * 返回上一个MySQL操作中的错误代码
     *
     * @access public
     * @return integer
     */
    public function errorCode()
    {
        return (integer) ($this->_pdo) ? $this->_pdo->errorCode() : -99999;
    }

    /**
     * 抛出异常
     *
     * @access private
     * @param string $message 错误信息
     * @return string
     */
    private function _throwException($message)
    {
        $errorinfo = $this->errorInfo();
        $str = '<html><head><meta http-equiv="content-type" content="text/html; charset=utf-8" />';
        $str .= '<title>错误消息</title></head><body>';
        $str .= '<div style="position:absolute;font-size:11px;font-family:verdana,arial;background:#FFFFFF;padding:0.5em;">';
        if (!DEBUG)
        {
        $str .= "<strong>数据库错误(MySQL Error)：</strong><br />
                <strong>错误消息:</strong> $message<br />
                <strong>错误SQL:</strong> $this->_last_query<br />
                <strong>错误说明:</strong> " . $errorinfo[2] . "<br />
                <strong>错误代码:</strong> " . $this->errorCode();
        }
        else
        {
            $str .= '站点发生特大异常，系统自动关闭，请等待管理员修复。';
        }
        $str .= '</div></body></html>';
        self_ob_clean();
        echo $str;
        @$this->close();
        exit();
    }
}

?>