<?php

if (!defined('__IN_SY__')) { exit('Access Denied'); }

class Upload
{

    /**
     * 上传文件的最大值
     *
     * @var integer
     */
    private $_max_size;

    /**
     * 是否支持多文件上传
     *
     * @var boolean
     */
    private $_support_multi;

    /**
     * 允许上传的文件后缀
     *
     * @var array
     */
    private $_allow_exts;

    /**
     * 允许上传的文件类型
     *
     * @var array
     */
    private $_allow_types;

    /**
     * 上传文件保存路径
     *
     * @var string
     */
    private $_save_path;

    /**
     * 成功保存后的文件名
     *
     * @var string
     */
    private $_file_name;

    /**
     * 命名上传文件的函数名称
     *
     * @var string
     */
    private $_save_rule;

    /**
     * 存在同名是否覆盖
     *
     * @var boolean
     */
    private $_is_replace;

    /**
     * 是否发生错误
     *
     * @var boolean
     */
    private $_error = FALSE;

    /**
     * 错误信息
     *
     * @var boolean
     */
    private $_error_message = NULL;

    /**
     * 上传成功的文件信息
     *
     * @var array
     */
    private $_file_info;

    /**
     * 构造函数
     *
     * @access public
     * @param
     * @return void
     */
    public function __construct()
    {
        // 设置默认值
        $this->_max_size = 1024000; //1M
        $this->_allow_exts = array('jpg', 'gif', 'png', 'rar', 'zip');
        $this->_is_replace = FALSE;
        $this->_error = FALSE;
        $this->_error_message = '';
        $this->_save_rule = 'time';
        $this->_file_name = '';
    }

    /**
     * 析构函数
     *
     * @access public
     * @param
     * @return void
     */
    public function __destruct()
    {
    }

    /**
     * 设置上传的最大值
     *
     * @access public
     * @param string $size
     * @return void
     */
    public function setMaxSize($size)
    {
        if (is_numeric($size) && intval($size)) { $this->_max_size = (integer) $size; }
    }

    /**
     * 设置允许的后缀
     *
     * @access public
     * @param array $exts
     * @return void
     */
    public function setAllowExts($exts)
    {
        if (is_array($exts)) { $this->_allow_exts = (array) $exts; }
    }

    /**
     * 设置允许的类型
     *
     * @access public
     * @param array $exts
     * @return void
     */
    public function setAllowTypes($types)
    {
        if (is_array($types)) { $this->_allow_types = (array) $types; }
    }

    /**
     * 存在同名文件是否覆盖
     *
     * @access public
     * @param boolean $is_replace
     * @return void
     */
    public function setReplace($is_replace)
    {
        $this->_is_replace = (boolean) $is_replace;
    }

    /**
     * 是否支持多文件上传
     *
     * @access public
     * @param boolean $support_multi
     * @return void
     */
    public function setSupportMulti($support_multi)
    {
        $this->_support_multi = (boolean) $support_multi;
    }

    /**
     * 设置上传路径
     *
     * @access public
     * @param string $save_path
     * @return void
     */
    public function setSavePath($save_path)
    {
        $this->_save_path = (string) $save_path;
    }

    /**
     * 命名上传文件的函数名称
     *
     * @access public
     * @param string $save_rule
     * @return void
     */
    public function setSaveRule($save_rule)
    {
        $this->_save_rule = (string) $_save_rule;
    }

    /**
     * 上传文件的信息
     *
     * @access public
     * @param
     * @return ArrayObject
     */
    public function getFileInfo()
    {
        return $this->_file_info;
    }

    /**
     * 上传成功后的文件名
     *
     * @access public
     * @param
     * @return string
     */
    public function getFileName()
    {
        return (string) $this->_file_name;
    }

    /**
     * 是否发生错误
     *
     * @access public
     * @param
     * @return boolean
     */
    public function getError()
    {
        return (boolean) $this->_error;
    }

    /**
     * 错误信息
     *
     * @access public
     * @param
     * @return string
     */
    public function getErrorMessage()
    {
        return (string) $this->_error_message;
    }

    /**
     * 保存文件
     *
     * @access public
     * @param $savepath 保存的路径
     * @return boolean
     */
    public function save($savepath = '')
    {
        // 设置上传路径
        if (empty($savepath)) { $savepath = $this->_save_path; }
        if (substr($savepath, -1) != '/') { $savepath .= '/'; }

        // 检查上传目录是否存在
        if (!is_dir($savepath))
        {
            $this->_setErrorMessage('上传目录' . $savepath . '不存在！');
            return FALSE;
        }

        // 检查上传目录是否具有可写的权限
        if (!is_writeable($savepath))
        {
            $this->_setErrorMessage('上传目录' . $savepath . '不可写！');
            return FALSE;
        }

        $arrayObject = new ArrayObject();
        $is_upload = FALSE;

        $files = &$_FILES;

        if (FALSE === $this->_support_multi && count($files) > 1)
        {
            $this->_setErrorMessage('不允许上传多个文件！');
            return FALSE;
        }

        foreach ($files as $key => $file)
        {
            if (!empty($file['name']))
            {
                $pathinfo = pathinfo($file['name']);
                $file['extension'] = $pathinfo['extension'];
                $file['savepath'] = $savepath;
                $file['savename'] = $this->_getSaveName($file);

                if ($file['error'] !== 0)
                {
                    // 捕获文件上传失败的原因
                    $this->_setErrorMessage($this->_getErrorMessage($file['error']));
                    return FALSE;
                }

                // 检查上传文件后缀
                if (is_array($this->_allow_exts))
                {
                    if (!in_array(strtolower($file['extension']), $this->_allow_exts))
                    {
                        $this->_setErrorMessage('上传文件的后缀类型(' . $file['extension'] . ')不允许！');
                        return FALSE;
                    }
                }

                // 检查上传文件大小
                if (!(!($file['size'] > $this->_max_size) || (-1 == $this->_max_size)))
                {
                    $this->_setErrorMessage('上传文件的大小超过' . format_size($this->_max_size) . '！');
                    return FALSE;
                }

                // 检查上传文件MIME类型
                if (is_array($this->_allow_types))
                {
                    if (!in_array(strtolower($file['type']), $this->_allow_types))
                    {
                        $this->_setErrorMessage('上传文件MIME类型(' . $file['type'] . ')不允许！');
                        return FALSE;
                    }
                }

                // 检查是否非法上传文件
                if (!is_uploaded_file($file['tmp_name']))
                {
                    $this->_setErrorMessage('非法上传文件！');
                    return FALSE;
                }

                $filename = $file['savepath'] . $file['savename'];
                if (!$this->_is_replace && file_exists($filename))
                {
                    $this->_setErrorMessage('文件(' . $filename . ')已经存在！');
                    return FALSE;
                }

                if (!move_uploaded_file($file['tmp_name'], $filename))
                {
                    $this->_setErrorMessage('上传文件保存时发生错误！');
                    return FALSE;
                }

                if (function_exists(md5_file))
                {
                    $file['hash'] =  md5_file($filename);
                }
                unset($file['tmp_name'], $file['error']);

                $arrayObject->append($file);
                $is_upload = TRUE;
            }
        }

        if (TRUE === $is_upload)
        {
            $this->_file_info = $arrayObject;
            return TRUE;
        }
        $this->_setErrorMessage('没有选择上传文件！');
        return FALSE;
    }

    private function _getSaveName($filename)
    {
        $rule = $this->_save_rule;
        if (empty($rule))
        {
            //没有定义命名规则，则保持文件名不变
            $savename = $filename['name'];
        }
        else
        {
            if (function_exists($rule))
            {
                //使用函数生成一个唯一文件标识号
                $savename = $rule() . '.' . $filename['extension'];
            }
            else
            {
                //使用给定的文件名作为标识号
                $savename = $rule . '.' . $filename['extension'];
            }
        }
        $this->_file_name = $savename;
        return $savename;
    }

    /**
     * 根据错误代码返回错误信息
     *
     * @access private
     * @param $no 代码
     * @return string
     */
    private function _getErrorMessage($no)
    {
        $message = '';
        switch($no)
        {
            case 1:
                //$message = '上传的文件超过了 php.ini 中 upload_max_filesize 选项限制的值！';
                $message = '上传的文件超过了 php.ini 中限制的值！';
                break;
            case 2:
                //$message = '上传文件的大小超过了 HTML 表单中 MAX_FILE_SIZE 选项指定的值！';
                $message = '上传的文件超过了 HTML 表单中指定的值！';
                break;
            case 3:
                $message = '文件只有部分被上传！';
                break;
            case 4:
                $message = '没有文件被上传！';
                break;
            case 6:
                $message = '找不到临时文件夹！';
                break;
            case 7:
                $message = '文件写入失败！';
                break;
            default:
                $message = '未知上传错误！';
        }
        return $message;
    }

    /**
     * 设置错误信息
     *
     * @access private
     * @param $message 错误信息
     * @return void
     */
    private function _setErrorMessage($message)
    {
        if ($message)
        {
            $this->_error = TRUE;
            $this->_error_message = (string) $message;
        }
    }

}

?>