/**
 **	File ......... ICMPTwistSocket.cpp
 **	Published ....  2004-07-03
 **	Author ....... grymse@alhem.net
**/
/*
Copyright (C) 2004  Anders Hedstrom
Parts Copyright (c) Vergil of Ragestorm.net, 2003

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
#include <ISocketHandler.h>
#include <stdint.h>
#include <linux/types.h>
#include <linux/icmp.h>
#include <errno.h>
#include <Utility.h>
#include "ICMPTwistSocket.h"




ICMPTwistSocket::ICMPTwistSocket(ISocketHandler& h) : Socket(h)
,m_seq_no(0)
{
	SOCKET s = CreateSocket(AF_INET, SOCK_RAW, "icmp");
	if (s != INVALID_SOCKET)
	{
		// socket to filter out irrelevant ICMP, and make IO non-blocking
		static uint32_t flags = ~(1 << ICMP_INFO_REQUEST);
		if (setsockopt(s, SOL_RAW, ICMP_FILTER, &flags, sizeof(uint32_t)) == -1)
		{
			Handler().LogError(this, "setsockopt", errno, strerror(errno));
		}
		Attach(s);
		Set(true, false, false);
	}
	memset(&m_info, 0, sizeof(m_info));
	m_info.type = ICMP_INFO_REQUEST;
}


ICMPTwistSocket::~ICMPTwistSocket()
{
}


void ICMPTwistSocket::OnRead()
{
	int len;
	int hsize;
	uint8_t incoming[1000];

	if (recv(GetSocket(), m_info.buffer, 4, MSG_PEEK) != -1)
	{
		len = ntohs(*(uint16_t *)(m_info.buffer + 2));
		if (len < 1000)
		{
			if (recv(GetSocket(), incoming, len, 0) == -1)
			{
				Handler().LogError(this, "recv", errno, strerror(errno), LOG_LEVEL_WARNING);
			}
			else
			{
				// IP Header length + ICMP header length
				hsize = (m_info.buffer[0] & 0xf) << 2;
				uint16_t crc;
				memcpy(&crc, incoming + hsize + 2, 2);
				if (calc_crc(incoming + hsize, len - hsize) != crc) //*(uint16_t *)(incoming + hsize + 2))
				{
					printf("(Bad crc)\n");
				}
				else
				{
					printf("Received: %s\n", incoming + hsize + 8);
				}
			}
		}
	}
}


void ICMPTwistSocket::Connect(const std::string& host)
{
	struct sockaddr_in dest;
	ipaddr_t a;

	if (Utility::u2ip(host, a))
	{
		memset(&dest, 0, sizeof(dest));
		dest.sin_port = htons(IPPROTO_ICMP);
		dest.sin_family = AF_INET;
		memcpy(&dest.sin_addr, &a, 4);
		if (connect(GetSocket(), (struct sockaddr *)&dest, sizeof(struct sockaddr)) == -1)
		{
			Handler().LogError(this, "Connect", errno, strerror(errno), LOG_LEVEL_FATAL);
			SetCloseAndDelete();
		}
	}
	else
	{
		printf("u2ip failed\n");
	}
}


void ICMPTwistSocket::Send(const std::string& line)
{
	strcpy(m_info.buffer, line.c_str());
	int len = line.size() + 9;
	m_info.checksum = calc_crc((uint8_t *)&m_info, len);
	int n;
	if ((n = send(GetSocket(), (void *)&m_info, len, 0)) == -1)
	{
		Handler().LogError(this, "Send", errno, strerror(errno));
	}
	m_info.seq_no = htons(m_seq_no++);
}


uint16_t ICMPTwistSocket::calc_crc(uint8_t * packet, size_t len)
{
	uint16_t r;
	uint16_t i;
	uint32_t s;

	s = (packet[0] << 8) + packet[1];
	for (i = 4; i < len - 1; i += 2)
	{
		s += (packet[i] << 8) + packet[i + 1];
	}
	if (len % 2)	// an extra member
	{
		s += packet[len - 1] << 8;
	}
	s = (s & 0xffff) + (s >> 16);
	r = s + (s >> 16);
	return htons(~r);
}


