/*
 * The MIT License
 *
 * Copyright 2015 Jesse Glick.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package jenkins.model.queue;

import hudson.model.Computer;
import hudson.model.Executor;
import hudson.model.ExecutorListener;
import hudson.model.OneOffExecutor;
import hudson.model.Queue.Executable;
import hudson.model.Queue.FlyweightTask;
import hudson.model.Resource;
import hudson.model.ResourceActivity;
import hudson.model.ResourceController;
import hudson.model.ResourceList;
import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import net.jcip.annotations.GuardedBy;
import jenkins.model.Jenkins;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.NoExternalUse;

/**
 * Special means of indicating that an executable will proceed in the background without consuming a native thread ({@link Executor}).
 * May be thrown from {@link Executable#run} after doing any preparatory work synchronously.
 * <p>{@link Executor#isActive} will remain true (even though {@link Executor#isAlive} is not) until {@link #completed} is called.
 * The thrower will need to hold on to a reference to this instance as a handle to call {@link #completed}.
 * <p>The execution may not extend into another Jenkins session; if you wish to model a long-running execution, you must schedule a new task after restart.
 * This class is not serializable anyway.
 * <p>Mainly intended for use with {@link OneOffExecutor} (from a {@link FlyweightTask}), of which there could be many,
 * but could also be used with a heavyweight executor even though the number of executors is bounded by node configuration.
 * <p>{@link ResourceController}/{@link ResourceActivity}/{@link ResourceList}/{@link Resource} are not currently supported.
 * Nor are {@link hudson.model.Queue.Task#getSubTasks} other than the primary task.
 * @since 1.607
 */
public abstract class AsynchronousExecution extends RuntimeException {

    @GuardedBy("this")
    private Executor executor;

    /**
     * Initially null, and usually stays null.
     * If {@link #completed} is called before {@link #setExecutorWithoutCompleting}, then either {@link #NULL} for success, or the error.
     */
    @GuardedBy("this")
    private @CheckForNull Throwable result;

    /** Constructor for subclasses. */
    protected AsynchronousExecution() {}

    /**
     * Called in lieu of {@link Thread#interrupt} by {@link Executor#interrupt()} and its overloads.
     * As with the standard Java method, you are requested to cease work as soon as possible, but there is no enforcement of this.
     * You might also want to call {@link Executor#recordCauseOfInterruption} on {@link #getExecutor}.
     * @param forShutdown if true, this interruption is because Jenkins is shutting down (and thus {@link Computer#interrupt} was called from {@link Jenkins#cleanUp}); otherwise, a normal interrupt such as by {@link Executor#doStop()}
     */
    public abstract void interrupt(boolean forShutdown);

    /**
     * Allows an executable to indicate whether it is currently doing something which should prevent Jenkins from being shut down safely.
     * You may return false if it is fine for an administrator to exit/restart Jenkins despite this executable still being running.
     * (If so, {@link #interrupt} will be passed {@code forShutdown=true}.)
     * @return traditionally always true
     * @see hudson.model.RestartListener.Default#isReadyToRestart
     */
    public abstract boolean blocksRestart();

    /**
     * Allows an executable to control whether or not to display {@code executorCell.jelly}.
     *
     * <p>
     * If this method returns false, the asynchronous execution becomes invisible from UI.
     *
     * @return traditionally always true
     */
    public abstract boolean displayCell();

    /**
     * Obtains the associated executor.
     * @return Associated Executor. May be {@code null} if {@link #setExecutorWithoutCompleting(hudson.model.Executor)} 
     * has not been called yet.
     */
    @CheckForNull
    public final synchronized Executor getExecutor() {
        return executor;
    }

    /**
     * Set the executor without notifying it about task completion.
     * The caller <b>must</b> also call {@link #maybeComplete()}
     * after releasing any problematic locks.
     */
    @Restricted(NoExternalUse.class)
    public final synchronized void setExecutorWithoutCompleting(@NonNull Executor executor) {
        assert this.executor == null;
        this.executor = executor;
    }

    /**
     * If there is a pending completion notification, deliver it to the executor.
     * Must be called after {@link #setExecutorWithoutCompleting(Executor)}.
     */
    @Restricted(NoExternalUse.class)
    public final synchronized void maybeComplete() {
        assert this.executor != null;
        if (result != null) {
            executor.completedAsynchronous(result != NULL ? result : null);
            result = null;
        }
    }

    /**
     * To be called when the task is actually complete.
     * @param error normally null (preferable to handle errors yourself), but may be specified to simulate an exception from {@link Executable#run}, as per {@link ExecutorListener#taskCompletedWithProblems}
     */
    public final synchronized void completed(@CheckForNull Throwable error) {
        if (executor!=null) {
            executor.completedAsynchronous(error);
        } else {
            result = error == null ? NULL : error;
        }
    }

    /** @see #result */
    private static final Throwable NULL = new Throwable("NULL");

}
