// +build !scanner

package oval

import (
	"reflect"
	"testing"

	"github.com/future-architect/vuls/config"
	"github.com/future-architect/vuls/models"
	"github.com/future-architect/vuls/util"
	ovalmodels "github.com/kotakanbe/goval-dictionary/models"
)

func TestParseCvss2(t *testing.T) {
	type out struct {
		score  float64
		vector string
	}
	var tests = []struct {
		in  string
		out out
	}{
		{
			in: "5/AV:N/AC:L/Au:N/C:N/I:N/A:P",
			out: out{
				score:  5.0,
				vector: "AV:N/AC:L/Au:N/C:N/I:N/A:P",
			},
		},
		{
			in: "",
			out: out{
				score:  0,
				vector: "",
			},
		},
	}
	for _, tt := range tests {
		s, v := RedHatBase{}.parseCvss2(tt.in)
		if s != tt.out.score || v != tt.out.vector {
			t.Errorf("\nexpected: %f, %s\n  actual: %f, %s",
				tt.out.score, tt.out.vector, s, v)
		}
	}
}

func TestParseCvss3(t *testing.T) {
	type out struct {
		score  float64
		vector string
	}
	var tests = []struct {
		in  string
		out out
	}{
		{
			in: "5.6/CVSS:3.0/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L",
			out: out{
				score:  5.6,
				vector: "CVSS:3.0/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L",
			},
		},
		{
			in: "6.1/CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L",
			out: out{
				score:  6.1,
				vector: "CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L",
			},
		},
		{
			in: "",
			out: out{
				score:  0,
				vector: "",
			},
		},
	}
	for _, tt := range tests {
		s, v := RedHatBase{}.parseCvss3(tt.in)
		if s != tt.out.score || v != tt.out.vector {
			t.Errorf("\nexpected: %f, %s\n  actual: %f, %s",
				tt.out.score, tt.out.vector, s, v)
		}
	}
}

func TestPackNamesOfUpdate(t *testing.T) {
	var tests = []struct {
		in       models.ScanResult
		defPacks defPacks
		out      models.ScanResult
	}{
		{
			in: models.ScanResult{
				ScannedCves: models.VulnInfos{
					"CVE-2000-1000": models.VulnInfo{
						AffectedPackages: models.PackageFixStatuses{
							{Name: "packA"},
							{Name: "packB", NotFixedYet: false},
						},
					},
				},
			},
			defPacks: defPacks{
				def: ovalmodels.Definition{
					Advisory: ovalmodels.Advisory{
						Cves: []ovalmodels.Cve{
							{
								CveID: "CVE-2000-1000",
							},
						},
					},
				},
				binpkgFixstat: map[string]fixStat{
					"packB": {
						notFixedYet: true,
						fixedIn:     "1.0.0",
					},
				},
			},
			out: models.ScanResult{
				ScannedCves: models.VulnInfos{
					"CVE-2000-1000": models.VulnInfo{
						AffectedPackages: models.PackageFixStatuses{
							{Name: "packA"},
							{Name: "packB", NotFixedYet: true},
						},
					},
				},
			},
		},
	}

	util.Log = util.NewCustomLogger(config.ServerInfo{})
	for i, tt := range tests {
		RedHat{}.update(&tt.in, tt.defPacks)
		e := tt.out.ScannedCves["CVE-2000-1000"].AffectedPackages
		a := tt.in.ScannedCves["CVE-2000-1000"].AffectedPackages
		if !reflect.DeepEqual(a, e) {
			t.Errorf("[%d] expected: %v\n  actual: %v\n", i, e, a)
		}
	}
}
