/**
 * module deps.
 * @type {exports|module.exports}
 */
var Url = require('url');
var util = require('util');
var EventEmitter = require('events').EventEmitter;
var httpStatus = require('./http-status');
var contentTypes = require('./contentTypes');
var httpMethods = require('./methods');
var querystring = require('querystring');

/**
 * http module
 * http.request & http.response
 * use http.createServer to runing start.
 * @type {Function}
 */
var http = module.exports = function(){
    var queryString = process.env.QUERY_STRING, remote;

    if ( /^404\;/.test(queryString) ){
        remote = queryString.substr(4).trim();
    }else{
        remote = process.env.HTTPS === 'off' ? 'http://': 'https://';
        remote += process.env.SERVER_NAME;
        remote += process.env.URL;
        if ( process.env.QUERY_STRING.length ) remote += '?' + queryString;
    }

    var env = Url.parse(remote, true);
    this.protocol = env.protocol;
    this.port = env.port || 80;
    this.query = env.query;
    this.pathname = env.pathname;
    this.path = env.path;
    this.host = env.host;
    this.hostname = env.hostname;
    this.search = env.search;
    this.href = env.href;
    this.slashes = env.slashes;
    this.auth = env.auth;
    this.method = process.env.REQUEST_METHOD.toUpperCase();
    this.res = new http.ServerResponse(this);

    var Forms =  new Enumerator(Request.Form);
    var bodys = {};
    for ( ; !Forms.atEnd(); Forms.moveNext() ){
        bodys[Forms.item()] = Request.Form(Forms.item())();
    }
    this.body = bodys;
}

/**
 * extend prototype.
 */
util.inherits(http, EventEmitter);

http.Url = Url;
http.EventEmitter = EventEmitter;
http.Server = http;
http.httpStatus = httpStatus;
http.contentTypes = contentTypes;
http.httpMethods = httpMethods;

http.prototype.createServer = function(foo){
    typeof foo === 'function' &&
        foo(this, this.res);
}

http.ServerResponse = function(req){
    this.req = req;
    this.statusCode = 100;
    this.statusMessage = null;
    // Boolean value that indicates whether the response has completed. Starts as false. After response.end() executes, the value will be true.
    this.finished = false;
}

util.inherits(http.ServerResponse, EventEmitter);

// http://www.w3school.com.cn/asp/asp_ref_response.asp
http.ServerResponse.prototype.writeHead = function(statusCode, headers){
    var i;
    if ( !headers ){
        if ( util.isObject(statusCode) ){
            for ( i in statusCode ){
                this.setHeader(i, statusCode[i]);
            }
        }else{
            this.statusCode = statusCode;
        }
    }else{
        this.statusCode = statusCode;
        for ( i in headers ){
            this.setHeader(i, headers[i]);
        }
    }
}

// 将字符集的名称追加到 Response 对象中的 content-type 报头。
http.ServerResponse.prototype.charset = function(encoding){
    Response.Charset = encoding;
}

// 设置 Response 对象的 HTTP 内容类型。
http.ServerResponse.prototype.contentType = function(str){
    Response.ContentType = str;
}

// 设置页面在失效前的浏览器缓存时间（分钟）。
http.ServerResponse.prototype.expires = function(time){
    Response.Expires = time;
}

// 设置浏览器上页面缓存失效的日期和时间。
http.ServerResponse.prototype.expiresAbsolute = function(time){
    Response.ExpiresAbsolute = time;
}

// 向 response 报头的 PICS 标志追加值。
http.ServerResponse.prototype.pics = function(value){
    Response.Pics = value;
}

// 设置头部信息
http.ServerResponse.prototype.setHeader = function(key, value){
    // value is string or array
    if ( util.isArray(value) ) value = value.join('; ');
    Response.AddHeader(key, value);
}

// 获取头部信息
http.ServerResponse.prototype.getHeader = function(key){
    return process.env[key];
}

// 删除头部某信息
http.ServerResponse.prototype.removeHeader = function(key){
    this.setHeader(key, '');
}

// 向页面输出信息
http.ServerResponse.prototype.write = function(chunk){
    Response.Write(chunk);
}

// 指示页面完成了操作并输出内容
http.ServerResponse.prototype.end = function(data){
    var message;
    if ( data ) this.write(data);
    if ( httpStatus[this.statusCode] ){
        if ( this.statusMessage != null ){
            message = this.statusMessage;
        }else{
            message = httpStatus[this.statusCode];
        }
    }else{
        message = this.statusMessage || '';
    }
    Response.Status = this.statusCode + " " + message;
    this.finished = true;
    Response.End();
}

// 向服务器记录项目（server log entry）的末端添加字符串。
http.ServerResponse.prototype.appendToLog = function(text){
    Response.AppendToLog(text);
}

// 在没有任何字符转换的情况下直接向输出写数据。
http.ServerResponse.prototype.binaryWrite = function(binary){
    Response.BinaryWrite(binary);
}

// 清除已缓存的 HTML 输出。
http.ServerResponse.prototype.clear = function(){
    Response.Clear();
}

// 立即发送已缓存的 HTML 输出。
http.ServerResponse.prototype.flush = function(){
    Response.Flush();
}

// 把用户重定向到另一个 URL。
http.ServerResponse.prototype.redirect = function(url){
    Response.Redirect(url);
}

// 输出JSON数据串并结束
http.ServerResponse.prototype.json = function(value){
    this.end(JSON.stringify(value));
}

// 输出JSON数据串并结束
http.ServerResponse.prototype.jsonp = function(value, jsonp){
    this.end(jsonp + '(' + JSON.stringify(value) + ')');
}

/**
 * var http = require('nodeasp-http');
 * var req = new http.request({
 *    host: 'http://api.webkits.cn',
 *    data: {
 *        a:1,
 *        b:2
 *    },
 *    method: 'GET'
 * });
 * req.on('success', function(xhr){
 *    var binary = xhr.responseBody;
 *    Response.BinaryWrite(binary);
 * });
 * req.send();
 * @param options
 */
http.request = function(options){
    this.xhr = new ActiveXObject("Microsoft.XMLHTTP");
    this.configs = util._extend(http.request.defaults, options || {});
    this.configs.method = this.configs.method.toUpperCase();
}

util.inherits(http.request, EventEmitter);

http.request.prototype.send = function(){
    var that = this,
        xhr = this.xhr,
        parser = Url.parse(this.configs.host, true);

    if ( that.configs.method != 'POST' ){
        parser.query = util._extend(parser.query, this.configs.data || {});
        parser.search = null;
    }

    that.emit('connect');
    xhr.open( that.configs.method, parser.format(), this.configs.async );
    xhr.onreadystatechange = function() {
        that.emit('change', xhr);
        if ( xhr.readyState === 4 ) {
            that.emit('accept', xhr);
            if ( xhr.status === 200 ){
                that.emit('success', xhr);
            }else{
                that.emit('failed', xhr);
            }
        }
    };
    if ( that.configs.method === 'POST' ){
        xhr.send(querystring.stringify(this.configs.data || {}));
    }else{
        xhr.send(null);
    };
    that.emit('close', xhr);
}

http.request.defaults = {
    host: '',
    data: {},
    method: 'GET',
    async: false
}

http.fetch = function(url, data, method, charset){
    var result;
    var req = new http.request({
        host: url,
        data: data,
        method: method
    });
    req.on('success', function(xhr){
        var binary = xhr.responseBody;
        result = BinaryToString(binary, charset || Response.Charset);
    });
    req.send();
    return result;
}

http.get = function(url, data){
    return http.fetch(url, data, 'GET');
}

http.post = function(url, data){
    return http.fetch(url, data, 'POST');
}

http.createServer = function(foo) {
    var HTTP = new http();
    return HTTP.createServer(foo);
}

function BinaryToString(text, charset){
    var obj = new ActiveXObject("Adodb.Stream"),
        ret;
    obj.Type = 1;
    obj.Mode = 3;
    obj.Open;
    obj.Write(text);
    obj.Position = 0;
    obj.Type = 2;
    obj.Charset = charset;
    ret = obj.ReadText;
    obj.Close;

    return ret;
}
