package com.zebra.common.utils.file;

import java.io.*;

import org.apache.commons.io.FilenameUtils;
import org.springframework.web.multipart.MultipartFile;

import com.zebra.common.config.ConfigServerApplication;
import com.zebra.common.exception.file.FileNameLengthLimitExceededException;
import com.zebra.common.exception.file.FileSizeLimitExceededException;
import com.zebra.common.exception.file.InvalidExtensionException;
import com.zebra.common.utils.DateUtils;
import com.zebra.common.utils.StringUtils;
import com.zebra.common.utils.security.Md5Utils;
import com.zebra.common.utils.spring.SpringUtils;

/**
 * 文件上传工具类
 *
 * @author ruoyi
 */
public class FileUploadUtils {
	/**
	 * 默认大小 50M
	 */
	public static final long DEFAULT_MAX_SIZE = 50 * 1024 * 1024;

	/**
	 * 默认的文件名最大长度 100
	 */
	public static final int DEFAULT_FILE_NAME_LENGTH = 100;

	/**
	 * 默认上传的地址
	 */
	private static String defaultBaseDir = null;

	private static int counter = 0;

	public static void setDefaultBaseDir(String defaultBaseDir) {
		ConfigServerApplication configServerApplication = SpringUtils.getBean(ConfigServerApplication.class);
		FileUploadUtils.defaultBaseDir = configServerApplication.getProfile();
	}

	public static String getDefaultBaseDir() {
		return defaultBaseDir;
	}

	/**
	 * 以默认配置进行文件上传
	 *
	 * @param file
	 *            上传的文件
	 * @return 文件名称
	 * @throws Exception
	 */
	public static final String upload(MultipartFile file) throws IOException {
		try {
			return upload(getDefaultBaseDir(), file, MimeTypeUtils.DEFAULT_ALLOWED_EXTENSION);
		} catch (Exception e) {
			throw new IOException(e.getMessage(), e);
		}
	}

	/**
	 * 根据文件路径上传
	 *
	 * @param baseDir
	 *            相对应用的基目录
	 * @param file
	 *            上传的文件
	 * @return 文件名称
	 * @throws IOException
	 */
	public static final String upload(String baseDir, MultipartFile file) throws IOException {
		try {
			return upload(baseDir, file, MimeTypeUtils.DEFAULT_ALLOWED_EXTENSION);
		} catch (Exception e) {
			throw new IOException(e.getMessage(), e);
		}
	}

	/**
	 * 文件上传
	 *
	 * @param baseDir
	 *            相对应用的基目录
	 * @param file
	 *            上传的文件
	 * @param extension
	 *            上传文件类型
	 * @return 返回上传成功的文件名
	 * @throws FileSizeLimitExceededException
	 *             如果超出最大大小
	 * @throws FileNameLengthLimitExceededException
	 *             文件名太长
	 * @throws IOException
	 *             比如读写文件出错时
	 * @throws InvalidExtensionException
	 *             文件校验异常
	 */
	public static final String upload(String baseDir, MultipartFile file, String[] allowedExtension)
			throws FileSizeLimitExceededException, IOException, FileNameLengthLimitExceededException,
			InvalidExtensionException {
		int fileNamelength = file.getOriginalFilename().length();
		if (fileNamelength > FileUploadUtils.DEFAULT_FILE_NAME_LENGTH) {
			throw new FileNameLengthLimitExceededException(FileUploadUtils.DEFAULT_FILE_NAME_LENGTH);
		}

		assertAllowed(file, allowedExtension);

		String fileName = extractFilename(file);

		File desc = getAbsoluteFile(baseDir, fileName);
		file.transferTo(desc);
		return fileName;
	}

	/**
	 * 编码文件名
	 */
	public static final String extractFilename(MultipartFile file) {
		String filename = file.getOriginalFilename();
		String extension = getExtension(file);
		filename = DateUtils.datePath() + "/" + encodingFilename(filename) + "." + extension;
		return filename;
	}

	private static final File getAbsoluteFile(String uploadDir, String filename) throws IOException {
		File desc = new File(uploadDir + File.separator + filename);

		if (!desc.getParentFile().exists()) {
			desc.getParentFile().mkdirs();
		}
		if (!desc.exists()) {
			desc.createNewFile();
		}
		return desc;
	}

	/**
	 * 编码文件名
	 */
	private static final String encodingFilename(String filename) {
		filename = filename.replace("_", " ");
		filename = Md5Utils.hash(filename + System.nanoTime() + counter++);
		return filename;
	}

	/**
	 * 文件大小校验
	 *
	 * @param file
	 *            上传的文件
	 * @return
	 * @throws FileSizeLimitExceededException
	 *             如果超出最大大小
	 * @throws InvalidExtensionException
	 */
	public static final void assertAllowed(MultipartFile file, String[] allowedExtension)
			throws FileSizeLimitExceededException, InvalidExtensionException {
		long size = file.getSize();
		if (DEFAULT_MAX_SIZE != -1 && size > DEFAULT_MAX_SIZE) {
			throw new FileSizeLimitExceededException(DEFAULT_MAX_SIZE / 1024 / 1024);
		}

		String filename = file.getOriginalFilename();
		String extension = getExtension(file);
		if (allowedExtension != null && !isAllowedExtension(extension, allowedExtension)) {
			if (allowedExtension == MimeTypeUtils.IMAGE_EXTENSION) {
				throw new InvalidExtensionException.InvalidImageExtensionException(allowedExtension, extension,
						filename);
			} else if (allowedExtension == MimeTypeUtils.FLASH_EXTENSION) {
				throw new InvalidExtensionException.InvalidFlashExtensionException(allowedExtension, extension,
						filename);
			} else if (allowedExtension == MimeTypeUtils.MEDIA_EXTENSION) {
				throw new InvalidExtensionException.InvalidMediaExtensionException(allowedExtension, extension,
						filename);
			} else {
				throw new InvalidExtensionException(allowedExtension, extension, filename);
			}
		}

	}

	/**
	 * 判断MIME类型是否是允许的MIME类型
	 *
	 * @param extension
	 * @param allowedExtension
	 * @return
	 */
	public static final boolean isAllowedExtension(String extension, String[] allowedExtension) {
		for (String str : allowedExtension) {
			if (str.equalsIgnoreCase(extension)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 获取文件名的后缀
	 *
	 * @param file
	 *            表单文件
	 * @return 后缀名
	 */
	public static final String getExtension(MultipartFile file) {
		String extension = FilenameUtils.getExtension(file.getOriginalFilename());
		if (StringUtils.isEmpty(extension)) {
			extension = MimeTypeUtils.getExtension(file.getContentType());
		}
		return extension;
	}

	/**
	 * 保存图片
	 * @param filepath
	 * @param filename
	 * @param filebytes
	 * @return
	 * @throws Exception
	 */
	public static String savefile(String filepath, String filename, byte[] filebytes) throws Exception {
		// 文件夹是否存在
		File file = new File(filepath);
		if (!file.exists()) {
			file.mkdirs();
		}
		// 文件是否存在
		file = new File(filepath + filename);
		if (file.exists()) {
			file.delete();
		}
		FileOutputStream outputStream = null;
		InputStream iStream = null;
		try {
			// 生成文件;
			byte[] Buffer = new byte[1024 * 5];
			outputStream = new FileOutputStream(file);
			iStream = new ByteArrayInputStream(filebytes);
			int size = 0;
			while ((size = iStream.read(Buffer)) != -1) {
				outputStream.write(Buffer, 0, size);
			}
		} catch (Exception e) {
			throw e;
		} finally {
			try {
				if (iStream != null) {
					iStream.close();
				}
				if (outputStream != null) {
					outputStream.close();
				}
			} catch (IOException e) {
				throw e;
			}
		}
		return filepath + filename;
	}
}
