package com.zebra.framework.web.exception;

import javax.servlet.http.HttpServletRequest;

import org.apache.shiro.authz.AuthorizationException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.servlet.ModelAndView;

import com.zebra.common.core.domain.AjaxResult;
import com.zebra.common.exception.BusinessException;
import com.zebra.common.exception.CacheException;
import com.zebra.common.exception.DemoModeException;
import com.zebra.common.exception.LimitIpException;
import com.zebra.common.utils.ServletUtils;
import com.zebra.common.utils.security.PermissionUtils;

import lombok.extern.slf4j.Slf4j;

/**
 * 全局异常处理器
 *
 * @author ruoyi
 */
@RestControllerAdvice
@Slf4j
public class GlobalExceptionHandler {
	/**
	 * 权限校验失败 如果请求为ajax返回json，普通请求跳转页面
	 */
	@ExceptionHandler(AuthorizationException.class)
	public Object handleAuthorizationException(HttpServletRequest request, AuthorizationException e) {
		log.error(e.getMessage(), e);
		if (ServletUtils.isAjaxRequest(request)) {
			return AjaxResult.error(PermissionUtils.getMsg(e.getMessage()));
		} else {
			ModelAndView modelAndView = new ModelAndView();
			modelAndView.setViewName("error/unauth");
			return modelAndView;
		}
	}

	/**
	 * 请求方式不支持
	 */
	@ExceptionHandler({ HttpRequestMethodNotSupportedException.class })
	public AjaxResult handleException(HttpRequestMethodNotSupportedException e) {
		log.error(e.getMessage(), e);
		return AjaxResult.error("不支持' " + e.getMethod() + "'请求");
	}

	/**
	 * 拦截未知的运行时异常
	 */
	@ExceptionHandler(RuntimeException.class)
	public AjaxResult notFount(RuntimeException e) {
		log.error("运行时异常:", e);
		return AjaxResult.error("运行时异常:" + e.getMessage());
	}

	/**
	 * 系统异常
	 */
	@ExceptionHandler(Exception.class)
	public AjaxResult handleException(Exception e) {
		log.error(e.getMessage(), e);
		return AjaxResult.error("服务器错误，请联系管理员");
	}

	/**
	 * 业务异常
	 */
	@ExceptionHandler(BusinessException.class)
	public Object businessException(HttpServletRequest request, BusinessException e) {
		log.error(e.getMessage(), e);
		if (ServletUtils.isAjaxRequest(request)) {
			return AjaxResult.error(e.getMessage());
		} else {
			ModelAndView modelAndView = new ModelAndView();
			modelAndView.addObject("errorMessage", e.getMessage());
			modelAndView.setViewName("error/business");
			return modelAndView;
		}
	}

	/**
	 * 演示模式异常
	 */
	@ExceptionHandler(DemoModeException.class)
	public AjaxResult demoModeException(DemoModeException e) {
		return AjaxResult.error("演示模式，不允许操作");
	}

	/**
	 * 演示模式异常
	 */
	@ExceptionHandler(LimitIpException.class)
	public AjaxResult limitIpException(LimitIpException e) {
		return AjaxResult.error("ip受限，不允许任何操作，请联系管理员");
	}

	/**
	 * 演示模式异常
	 */
	@ExceptionHandler(CacheException.class)
	public AjaxResult cacheException(CacheException e) {
		return AjaxResult.error("缓存更新失败，请稍后重试");
	}
}
