// Modules to control application life and create native browser window
const { ipcMain, app, nativeImage, Menu, Tray, BrowserWindow } = require('electron')

// Keep a global reference of the window object, if you don't, the window will
// be closed automatically when the JavaScript object is garbage collected.
let mainWindow
let appIcon = null

function createWindow () {
  // Create the browser window.
  mainWindow = new BrowserWindow({
    width: 800,
    height: 600,
    webPreferences: {
      nodeIntegration: true
    }
  })

  // and load the index.html of the app.
  mainWindow.loadFile('index.html')

  // Open the DevTools.
  // mainWindow.webContents.openDevTools()

  // Emitted when the window is closed.
  mainWindow.on('closed', function () {
    // Dereference the window object, usually you would store windows
    // in an array if your app supports multi windows, this is the time
    // when you should delete the corresponding element.
    mainWindow = null
  })
}

// This method will be called when Electron has finished
// initialization and is ready to create browser windows.
// Some APIs can only be used after this event occurs.
app.whenReady().then(createWindow)

// Quit when all windows are closed.
app.on('window-all-closed', function () {
  // On macOS it is common for applications and their menu bar
  // to stay active until the user quits explicitly with Cmd + Q
  if (process.platform !== 'darwin') {
    app.quit()
  }
})

app.on('activate', function () {
  // On macOS it is common to re-create a window in the app when the
  // dock icon is clicked and there are no other windows open.
  if (mainWindow === null) {
    createWindow()
  }
})

ipcMain.on('put-in-tray', (event) => {
  const icon = nativeImage.createFromDataURL('data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAABAAAAAQACAYAAAB/HSuDAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAB2HAAAdhwBp8J46gAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAACAASURBVHic7N15kF3XfR/437nvve5GN7oBkAIBklq5i7DkJE5s2alJ0fE2ckxSJMVdIABSiy3H9EQpZyJ7UgWLrhkrNZWxnLElkCJBgasIUdwsK7YnobwprjFtRwtAirJCWbZFgCBIoIFuoLvfu2f+gMaxRIDi0n1Pd9/P5x/9IfF9f1Xi7Xfv951zbgQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACwUFLpAQCA5rz91397+HXr37Bueqa7ZrgaHPjm/q/v+eyNPzlTei4AYOEpAABgmbvy1i9/b6+K91aR/6dc1xtSVVV/77+uc0pPRB1/VA/Fx+961/f8abFBAYAFpQAAgGXqho/v/p65Tv5IRH1BRFTf7X8fERFV5/P9yP/q7k0b/t8FHQ4AaJwCAACWoS07nvg3g8HcL0fEyMv9Z3Od+6nb/b++8dTeX/zc1h/uL8B4AEABCgAAWEY2b39qpJ+m7qhyfuer/rCU/mhkqHfJzdec++w8jAYAFKYAAIBl4oa7dr+h34+Hcz146zx+7F/X3c477tx4/p/P42cCAAW8tP2AAMCitun2XT8yN1M/Ns8P/xERr6vm+n943W273jXPnwsANMwKAABY4jbd/uV/lev6w5FSbwFjcl2lj5711Iaf27o11QuYAwAsEAUAACxR7932WG96ZOTXqjq/v7HQVD3a76287O5r3/B8Y5kAwLxQAADAEnT1LV9a1+vGgxHxtqazc1RPdqJ78e1bzn2i6WwA4JVzBgAALDHX3fWV7+/14s+iwMN/RESK+pwc/T+5/vYnLiqRDwC8MgoAAFhCttzxxJaYnflc5Di95Bw56lX9evaBTbft+pWScwAAL50tAACwFOScNm3ffVNO9S/Govv+rh7Kz/avveMXvneq9CQAwIktshsIAOA7bd7+1Oo6Te+MXP9o6VlOJKe0e8XI8EU3X3X210rPAgAcnwIAABax99z+1Q0z+chDEenM0rN8d3lfVENX7th03qOlJwEAXsgZAACwSG3c8filM3nmvy6Nh/+IiLQ2D2b/08YdT9xYehIA4IWsAACARWjTjid+KfdnPxQpLc2yvqruPrIituy8YsNs6VEAgGMUAACwiFy47bHRNSOjd0Q9uLT0LK9WjvSnvZHuO267+rxvlp4FAFAAAMCisXHHF95U1d2Hcq7fUnqW+ZIi7Rl0O5fdufHNny89CwC03dJcVggAy8zmO3b/WBp0/nQ5PfxHROTI66u5ud+77hOPbyo9CwC0nRUAAFDYddt3fSDy4FcjpV7pWRZQrqv00bOe2vBzW7emuvQwANBGCgAAKOS92x7rzYyMbct1f0vpWZqSUudz3dF82a1XbHiu9CwA0DYKAAAo4F3bHju1Ghp+MCK+v0R+6g5H7s+UiI5I1deH0oqLP77pjC+WGQAA2skZAADQsM137n5bNTTyWBR4+M8R0Rk7KXonvTa6q06NKPGWwVy/cbae+vx1d37lyubDAaC9FAAA0KAtn9j97npu8GhEPq3p7FR1YmjNadEZWxMREdXwaHTXnBapU+TogbGYnbl70227fqVEOAC0kS0AANCAC7Y+2n3tm9Z+pKrz+0vkp04vemtOi6i6L/jvcj2I/uTeyLNHCkwWkTrVI9OH4pqdP7vhcJEBAKAlFAAAsMCuueuv1nRmDn0qpfzPS+RXw2PRXbUuXvRrP+foTz0X9fSBxub6dumJXnfkwls3nvWXhQYAgGVPAQAAC2jz9sffMoi5h1KkN5XI74ytic7YSS/5f18fPRz9yWfi2GkBzco5H4jUvfaOLef/duPhANACzgAAgAWycceuywe5//kiD/+piu6q9S/r4T8iohpZGb2TTj/uVoGFllJaHbn/8OYdX/m3jYcDQAtYAQAA8y3ntGn77pvqGHwwpeaP2U+dXnRXrY/UHXrFn5EH/ehP7ok8V+hVgVV1976DQ9d/9sazCw0AAMuPAgAA5tH1D+0b7z//zCcj128vkZ+GVkRv1fr5eb1fztE/tC/qo4de/We9kviIP0vV0MU7Np37t0UGAIBlRgEAAPPk+vufPKN/aPbhyHlDifxqdHV0x06KSPP79T44Mhn9Q/uK3DTkiL1peOidO645948KxAPAsqIAAIB5sOn2x3+8rvv3pBQvb9P9vEjRnVgb1cj4giXUs0eif3BPRK4XLONEcs6z0R36+TuuO+9jjYcDwDKiAACAV+m67bs+kOvBh1OVmj85r+pGb/X6SN3hhc+q+zF34OnI/dmFz3qhXFfpozMrNty484o0KDEAACx1CgAAeIU2b39qpE7Tn4hcX1EiP3WHjx3212myd8jRP7g36pmpBjP/h1R1/iBHunTHpjfvLzIAACxhCgAAeAWu2/G118fgyMMR+XtL5Fcj49EdXzvv+/1fqsHU89E/vD9Sifyq+qvZGLrk3k1n/0Xz4QCwdCkAAOBl2nLH4z/UHww+lXI+tensHBHd8bXRWTHRdPQL1DPT0Z/cW+RcgEgxlTrD7/nExnPuaT4cAJYmBQAAvAybbn/8vbme+/WcUgOb7r9DqqK7al1UQ6ONR59I7s/G3IE9EfVckfhI1X+4Y9P5vxAp5RIDAMBSogAAgJfggq2Pdl/7prUfSXV+f4n81B2K3upTI6rmzxn8rnIdcwf2RJ47UiQ+pc5numvWXn3bxWsPFRkAAJYIBQAAfBcbd3zhlMidB6LOP1QivxpZGd2JU2JRf23nHIOp52IwfaDQANVXOkMjF95+7ZlfLTQAACx6i/hOAgDKu2rHl/5xr44HIsdrS+R3xk6KztiaEtGvSH30cPQnn4ljpxU0K0UcjKr7rh2b3vxbjYcDwBKgAACAE7juE7uvzIP+rRFprOnsVHWiO3FKpEW03/+lyv2Z6B/YE7nuN5+d8yD1hv6PO6477981Hg4Ai5wCAAC+U85p4/bdN+U8+GBKqWo6PnWHojuxLlJ3qOnoeZPrQfQP7ok8d7RIfqo69+yb7G357I1nzxQZAAAWIQUAAPw9P3fn/okDs0/fF5F/okR+NTwa3Yl1Ec33DvMv5xhM7Y/B9MEy8ZH+oh/54nuvf8tfFxkAABYZBQAAfMvGHU++OfdnH4rIZ5fI74yujs7Kk0tEL6jBkckYHNpXJjzFM52h4cs/ce05f1BmAABYPJbBzwsA8Opt3v6Vd+T+7J+UefhP0ZlYtywf/iMiOismorvmtIiq03x4jlMGMzO/965PPP4vmw8HgMXFCgAAWu9d23d9INeDf59SavwJNXV60V21LlJ3uOnoxuXB3LFzAfqzRfJTt3fXma87d/PWH07Nn04IAIuAAgCA1tq8/amRuTx1R4r8zhL5qTsc3dWnRirxy3ghOddRH3o2BkcPFclPneqPVvS6l9x8zbnPFhkAAApSAADQStfctfsNnbl4ONeDt5bIr0bGozu+NiK186t4MPV8DKaeK5KdUvpGztUld15//p8XGQAACmnnXQcArXbt7bt+JNX53oj8mhL53YlTohoZLxG9qOTZ6egf3Bs5182Hp5jOuXrfXddvuLP5cAAoQwEAQKts3P7ET+c8+5GINNR4eNWJ3qr1kXojjUcvVnkwF3MHno4YzBWJT1X3N8/8+nk3bt2aCrQQANAsBQAArfDebY/1poZGfi3l/P4S+ak3HJ2J9ZE63RLxi1uuj60EmJ0uk5+6/6W7ev1lt1+y5kCZAQCgGQoAAJa9q2/50rqqEw+miLeVyE8jK6MzvjZS8vbdFzOYei7qqefLhKfqyaHuiotu23jGV8oMAAALTwEAwLJ25fYv/5Ne5Acix+lNZ+eI6IydFJ2xNU1HL1l5ZirmDu6NFLn57KgOpqpz3Z2bz3u48XAAaICfIgBYtjbd8cSWbq5/v8TDf1Sd6K05zcP/y5SGx2Lo5NdGVM1vlUhRr8qD2QeuvW3XrzQeDgANsAIAgOUn53Tt9t03pah/MUp813V60V19aqROr/Ho5SLXgxgc3BN57miR/LrqPHhoZvW1j7zvtEIHEwDA/FMAALCsbN7+1Op+nr4vcv1jJfLT8Fh0V60LX7HzIOdj5wJMlzmbL6dqdx7uXXj3tef89yIDAMA8c3cCwLKx5bavnj8TRx6ucjqzRH41ujo6K08uEb2sDY5MxuDQviI3LXXkfRFDV959w3mPFogHgHmlAABgWdi44/FL67nB7RF5vPHwVEVn4pSohscaj26Leu5IDA7ujagHjWfnFLNR9X7h7s3n/Xrj4QAwjxQAACx5197+xC/l/uyHUon37HV60V21LlJ3uPHotsmDfgwm90SemykzQNW9e3as3rLzig2zZQYAgFdHAQDAknXhtsdGJ4ZG74h6cGmJ/DS0Irqr1kcU6B3aKuc6Bof2RT56uMwAqfrT1Bu7+M6Nb3y6zAAA8MopAABYkjbu+MKb6kH3oajrt5TIr0ZXRWfs5Ijkq7SEkucCRJX2DLqdy+7d+ObPl4gHgFfKXQsAS841t+3+0Yj63pRz8yfupRSd8bVRjTR/1ADfLs9OR//g3ohcNx+eYjrnzs/cfcP5O5oPB4BXRgEAwJJy9fZdH6jqwa9GpF7j4VUnuqvWR+qNNB7N8eXBXPQPPh3RnysSH5E+evZfb/i5rVtTgRYCAF4eBQAAS8L3bXusd25vxbbI9ZYiA/SGozuxPlKnWySeF5Ojf2BP5NnpMulV93Or145e+tGfesPzRQYAgJdIAQDAonf59l3ru3V+KEX+/hL5aWQ8uuNr7fdf5AZTz8fg8P5IJf5/StXXu50VF+/YdMYXmw8HgJfGnQwAi9q1d+5+Wz1T358in9Z0do6I7vhrolqxquloXqF6ZjoGk8XOBZjq9Uau/8TGs+9rPhwAvjvvLQJg0XrXJ3a/O88MHi3x8B9VJ7qrT/Xwv8RUw6PRXXNapE7zR0REjrG5mSP3XH3brl9pPhwAvjsrAABYdC7Y+mj39Net/UiO/DNR4LsqdYeiu/rUiMp+/yWrHkR/cm/k2SNl8qvqkbnpuGbnz244XGYAAHghBQAAi8o1d/3Vmnz00KdS5H9eIr8aWRmdiVPCV+QykHMMpp6LevpAkfiUqse7o0MXfuKqs79WZAAA+A7ubgBYNK7d/vhb8mDuoYj0pqazc0R0xk6KztiapqNZYPXRwzE49ExEziXiD1TDK66+c+NZ/6lEOAD8fQoAABaFK7fvujwN6ttSxMrGw1MVnVXrohoabTyaZuT+TPQP7okY9EukD6Iz/L/ds+XcXy0QDgB/RwEAQFk5p6tv3X1TjsEHU6TGD6dN3aHoTKyL1B1qOpqG5bofg4N7Is/NlMmvqrsH34gtO7dumC0yAACtpwAAoJjrH9o3fvTZZz6Zc/32EvnV8Gh0JtZFgd6BUnKOwaF9UR89VCQ+VZ0/q1esvOjeq1//zSIDANBqCgAAirj+rifPOHJk9uGIvKFEfjW6OjorTy4RzSJQH5mMwaF9RbJTSnv7ne5l920+74+LDABAaykAAGjclbc//uOp378nRZzUeHhK0Rk/JaqR5o8aYHHJc0eif2BPRK5LpM9Gb+TGezads61AOAAtpQAAoFFX37rrAzkPPpwidRsP73Sju2p9pO5w49EsUnU/+geejtwvsC0/RY5IH+2Pb7hx5xVp0PwAALSNAgCARmze/tTIbJ6+Pdf1lSXyU284uqtOjag6JeJZ1HL0D+6NPDNVJD11On/QrdKlOza9eX+RAQBoDQUAAAvu8m27Xt/t5odzzt9bIr8aGY/O+NqI5GuPExtMPR+Dw/sjFfj3JKX0V6kae8fdW9703xoPB6A13AkBsKCuuuPxH0ozg0/lyKeWyO+Mr41qxUSJaJagPDsd/YN7i5wLkKo0FdXQe+7ZfM49jYcD0AoKAAAWzDW3P/7euj/36xGp+U33VefYfv/eSOPRLG25Pxv9g3siBnNF4lNV/Yd7tpz/C5FSLjEAAMuXAgCAeXfB1ke7p75u7Udyzu8vkZ+6Q9FdfWpE1fw5gywT9SAGk3ujnj1SJD51Op/pT+erdv7shsNFBgBgWVIAADCvNu74wilz/c4DUecfKpFfjayManxtpFSViGc5yTkGU89FPX2gSHxK1VeGq5ELb99y5leLDADAsqMAAGDeXHbrl/7xUI4HcsRrS+RXYydFZ2xNiWiWsfro4RgceiYiN78iP6d0oNNbce3d1535242HA7Ds+HkEgHlx1W27r+zl/LkSD/+p6kR39ake/lkQ1cjK6K45PVKn+S0lKefV9ezUw1dtf/JDjYcDsOxYAQDAq5NzuvLW3TdFHnwwlVh33+lFZ9WpkTq9xqNpmXoQ/ck9EXNHy+Snzj0Hp3pbPnvj2TNlBgBgqVMAAPCKXXvn/om5mafvq3L+iRL5aWg0OhPrIuz3pyk5x+DQvsgzh8rER/qL7sjwRXe96+y/KTIAAEuaAgCAV2TjLU++eTbNPpQin10iP61YHZ2VJ5eIhqiPTMbg0L5IJe6kcjxT94Yvv2/zOX9QIB2AJcxPJgC8bFfeuuvts/noH5d4+M+RoppY7+GfoqoVE9Fdc1rk1Gk+PMUpae7o7111+1d+tvlwAJYyKwAAeFmuvHXXB3I9+HCVUvMnolW96KxaF6k73Hg0HE8ezMVgck9Ef7ZMfqd31943nrv5cz+c+kUGAGBJUQAA8JJs3v7UyHR/6o4q5XeWyE+9FdFZtd5+fxadnOuoDz8b+WiZcwGi6vxhGulces815z5bZgAAlgoFAADf1eU7vvb6avbIIxH5rSXy0/B4dMbXRpkN1/DSDKaej3pqf6Qi/56mb0RVXfLJ68//8wLhACwR7qQAeFFXfXzXj9S5vjeleE3z6Smq8bVRjYw3Hw2vQJ6djsHk3ohcNx+e0nR0ht/7yc1n39V8OABLgQIAgBO66tYnfrquZz+SUhpqPDx1orNqfaTeSOPR8KoM5qJ/4OmIeq7x6Jwjp87Qb7z5G+f8/NatqUALAcBipgAA4AW+b9tjvbM6I78Wkd9fIj/1RqKaWBepav6cQZgXuY7B5N7Is9NF4lPV+y8ja9Zddvslaw4UGQCARUkBAMC3ufqWL60bRDyYUrytRL79/iwn9dRzUU8/XyY8VU/2hlZcdOfGM75SZgAAFht3VwD8nUu3f/mfdPv5gZTi9Kazc46oxk6KztiapqNhQeWZqWPnAkRuPjviYE6963becN7DjYcDsOh4lxIAERFx1fYnN3f79e+XePiP1InO6lM9/LMspeGx6J702ogCW1pSxKpUzz5wxfYnP9R4OACLjhUAAG2Xc7r81t03pah/MUp8L3R60V19WpGHI2hUPYjB5J7Ic0fL5KfOg0f6q6995H2nlTmYAIDiFAAALbZ5+1Orp/rT96VU/1iJ/DQ0Fp1V68LXEa2Rcwymnot8pMzZfDmq3b0VKy+8+9o3/PciAwBQlDsugJa69ravnj/bP/JwSunMEvnV6Jqoxk4qEQ3F1UcmY3BoX6mzLvdV3ZEr7t1y9ueKpANQjDMAAFroyu2PXzo7mPmTIg//qYpqYr2Hf1qtWjER3TWnRaROifi1df/o71y9/cmfKxEOQDlWAAC0zJW3PfFLuT/7oUip+RK404vOxPpI3aHGo2ExynU/6oN7IvdnygzQ7d21evbcLTe/L82VGQCAJikAAFriwm2PjY52Ru/IeXBpifw0NBqdiXURBXoHWMxyriMffjbqo4fKDFB1/3RoaMXFd25849NlBgCgKQoAgBa4/JYvvClF96GI+i0l8tOK1dFZeXKJaFgySp4LkKpqTwwPXfLJd539J82nA9AUBQDAMnfFbbt/tB7U96bIBZ7AU3QmTok0vLL5aFiC8ux0DCb3RuS6RPx0js7PfOo95+8oEQ7AwlMAACxjl9+66wMxGPxqpNRrPLzqRnfV+ojucOPRsKQN5qJ/8OmIQZFt+Tk6Q7+x4Rvn/PzWralICwHAwlEAACxD37ftsd6Z3RXbcl1vKTJAdzg6q9ZHqrpF4mHJy3UMJvdGnp0uEp+q7qPdkdHL7r72Dc8XGQCABaEAAFhmLt++a32eyw+mlH+gRH4aGY/OyrVR6gXnsJzUU8/HYGp/pALXU6qqr/eGJy66612v+1Lj4QAsCHdnAMvIldt3v60e1PdHzqc1Hp4jqvG1Ua2YaDwalrOS5wKkiKmoetffd8N59zUeDsC8UwAALBNX3rb73fWg/x8jYqTx8FRFNbE+qqEVjUdDG+T+TNSTeyOXOBcgR5263f/9vuvP/3fNhwMwnxQAAEvcBVsf7b7mtWs/knL9M1Hg73r61n7/sN8fFlY9iMHknshzR8vkV52HD/ZOvuZ3r1s/VWYAAF4tBQDAEnbNXV9cMztd7Yyof6REfjW8MtL42kipKhEP7ZNz1FPPRX3kQJn81Nnd6Q5d9MnNZ3+tzAAAvBoKAIAl6vJtj78lV3MPRsQZjYfniGrspKjG1jQeDUTkmcMxmHwmInKB9HQgdXtX79xy3n8qEA7Aq6AAAFiC3vnxXZdHXd8akcebzk5VJ6rxdZHs94ei8tzRGBzcE5EHJeLnotP9N5+64fxfKxEOwCujAABYSnJOl338yzeluv5giXX3qTMU1ap1kTpDTUcDx5Hr/rHDAQudC5Cqzt3xN7Fl59YNs0UGAOBlUQAALBHXP7RvfHLv3k9G1G8vkV8NjUY1sS7Cfn9YXHKOwaFnIs8cLpNfdf6sOzZ+0b1Xv/6bZQYA4KVSAAAsAdfc9eQZs1NHH46IDSXyqxWro1p5colo4CWqj0xGfXhfkeyU0t48suLST2086/NFBgDgJVEAACxy77z18R/Pg7l7UsRJzaenqCbWRTU81nw08LLluSPfOhegbj48xWx0ejd+6vo3b2s+HICXQgEAsIhdduuuD+R+/8NVSt3Gwzu9qCbWReoONx4NvAqDuWMlwKDItvycO73ffMtfn3fj1q2pQAsBwItRAAAsQhdsf2rk5P707SkPriwyQG8kOhPrI6pOkXjg1coxmNwbMTNVJj51fz+tGr905xWve67MAAAcjwIAYJG5dNvuU6uqfjBy/f1FBhgZj87KtRHJVwQsdfX081Ef3h+pwPWcU/VXQ0OjF92z6YwvNh4OwHG5uwNYRC7/xOM/WM/070+RT206O0dEZ/yUSCPjTUcDCyjPTkc9ubfIuQApVVNVd+Tdn9xy1r2NhwPwAgoAgEXi8tt2vbfuD349RTS+6T6nTnRWrY/UG2k6GmhAHszG4MCeSPVc89kROXU6/+FT15//C5FSbnwAAP6OAgCgsAu2Pto96bS1H6mifn+RAXojxw77q5o/ZxBoUD2IwaG9EbNHisTnVP3WoeHXXPW7160vdDABAAoAgIIu+egXTqm6nU9Hrv9pifw0vDLS+NpIqSoRDzQt56innot85ECZ/E7niWpo+MKdG8/6yzIDALSbAgCgkMtu+dJbI6eHIuo3Np2dI6IaOymq0TVNRwOLQJ45HIPJZyJF8yvyc8SB1Bu+9v4t5/524+EALecnH4ACLr1l95WR449LPPxH1Ynu6tM9/EOLpeGV0V1zekSBrT8pYnXMzTx82e1f/eXGwwFazgoAgCblnC75+JdvSnX9wVRi3X1nKDqr1kd0eo1HA4tQPYjB5J6IuaNl8lN19/SR4es/e+PZM2UGAGgXBQBAQ669c//Ekelv3pdy/okS+Wl4LKqJdeFPP/Btco768L7IRw+Via+6f94ZGr9o56bX/W2RAQBaxF0gQAOuvOXJN8/Vsw+lqM8ukZ9GV0c1dnKJaGCJqI9MRn14X5mbw5Se6feG3vnQlnP/sEQ8QFsoAAAW2Dtu3vX2lAd3VRHNb7pPVVTjp0QaHms8Glh68tyRGEzujVQPms9OMRu94X/16S3n/mbj4QAtoQAAWECX3LrrAzHX/3BKqfmTtjq9qFatj9QZajwaWLryYC7qyb0R/ULb8qveXc+ddd7mz/1w6pcZAGD5UgAALIALtj81smZ2akeK+vIS+WloRVQT6yMKnDMILH0515EP7Ys8c7jMAJ3uH6w8aeLSHZe+dn+ZAQCWJwUAwDy7fMfXXj84Ov1I5PzWEvlpxapj+/2TP/HAq1NPPx/14f2Rivw9Sd9InZF33P/us/+iQDjAsuTuEGAeXXLLE/886plPpkivaTw8pahWro00Mt54NLB85dnpY1sCct18eErTnd7oe3ZuOfPu5sMBlh8FAMA8uezWXT9dD/ofSTk1v+m+6kY1sS5Sb6TxaKAFBnMxOPh0xGCu8egckVNn+Dfe+jfn/PzWralACwGwfCgAAF6l79v2WO91aeTXqly/v8gA3ZGoVq2LVDV/ziDQIrmOweTeiNnpMvFV9z/n7sw7H9zyDw8UGQBgGVAAALwKF93ypXXdOh6IyD9YIj+NjEe1cq39/kBj6qnnIk8/Xya86jyZRkd+6v5rz/xqmQEAljZ3jACv0IU3f/mfdHP9QEScXiK/WvmaSCtWlYgGWi7PTEV9aG9Ezs2Hp3QgusMbH7j+nN9qPhxgafN+KIBX4NJbn9zcrQe/HwUe/lPViWrVaR7+gWLS8Fh0Vp8eUWLrUc6rY/boQ5fc9tVfbj4cYGmzAgDg5cg5XXzzl29Kuf5gSqn5ErU7HJ1V68vcdAN8p3oQg8k9g/DaUQAAIABJREFUEXNHi8Tn1H1wkFdf+8j7TitzMAHAEqMAAHiJ3rH9L1anue59kfOPlchPwyujmjgl/OkGFpWco55+LvJ0mbP5UtXd1RkevXDndW98qsgAAEuIu0iAl+Dy2756/tzckYdTxJkl8tPYSVGNrikRDfCS1EcmIx/eVyY8Vc/0u8NXPHL92b9fZgCApUEBAPBdvPOWxy/tD/rbU+SJxsNTFdXEukhDo41HA7xcee5I1JN7I+pB8+EpZnNn+F8/+O5z/+/mwwGWBgUAwIu47JYnfqnuz34oUvOHpqbuUKSJdZE6Q01HA7xiue5HntwbudC5ANHp3bW2Pm/Lze9Lc2UGAFi8FAAAx3Hh1sdGO6eO3hG5f2mJ/DQ8GtX4uogC5wwCvFo51xGHn4366KEi+anT+dM0Mnbxpze+8ekiAwAsUgoAgO9w6cd3vyHX+ZFcD95SIj+Nro5q7OQS0QDzKh85GPXhZ4tkp1TtSUNjl3x6y5v+pMgAAIuQAgDg77n45t0/GnlwT8r5NY2HpxRp5SlRjaxsPBpgoeS5I1Ef3BOR6+bDqzTd6Y697/7rz7iz+XCAxUcBAPAtF9286wNV3f/ViNRrPLzqRmfV+ojucOPRAAtuMBeDg09HDIpsy89RDf/GP/jmOT+/dWsq0EIALB4KAKD1vm/bY73Xx/C2nPOWIgP0RqIzsT6i6hSJB2hErqOe3Bt5drpIfKp6j8725y77zPvf+nyRAQAWAQUA0GqXb9+1fm42Pxi5/oES+WlkPKqVayOSP8dAO9TTz0d9eH+kAn/3Uup8vTs0ctHOLWd+qfFwgEXAHSfQWhdu3/226ujc/Sml05rOzhFRrVwbacVE09EA5c1MxeDQ3kg5Nx6dUzrY6Y5c9+kbzn648XCAwrxfCmilSz7+xPXVTP/RIg//VSeqNad7+Afaa3gsqtWnR3SaP3Il5bxqMHvkgYtufeJDjYcDFGYFANAqF2x9tLtq/dqPpFT/TOQCfwO7w1GtWh9RdRuPBlh06kHUk3si5o4Wic9V5+EjIydf87vXrZ8qMgBAwxQAQGtc85tfXDPVqXamXP9IkQFGVkZauTZSsvgK4O/kHPX0cxHTB8rEV53daWj8ogc3v/5rRQYAaJACAGiFi7c9/pac5x6sIs5oOjtHRBo7KarRNU1HAywZeeZw1IeeKXUuwPPd3sqr77/+Tb/TeDhAg/wMBSx7F9+y650p+n9c4uE/qk50Vp/u4R/gu0jDK6NafVrkAq9ETTmvGcwdeuTiW568sfFwgAZZAQAsXzmnCz/25ZuqqD+YCqy7z52h6EysL3LIFcBSlet+5Mm9Ef1S5wL07lqXz9ty8/vSXJEBABaQAgBYli7c+thodeqKuyIP3lFkgN6KqCbWR9jvD/Dy5Rz1oWciZg+Xye/0Hssrxy5++OrXf7PMAAALQwEALDuX3fXkGf3DRx+OHBuKDLBidVRjJxeJBlhO8tHJqA/ti1TijjVVe6M3culD15/1+QLpAAtCAQAsKxdue/zHo567p0pxUuPhqYq08pRIw2ONRwMsV3nuSNQH90SKuvnsFLOpM3LjQ+8+Z1vj4QALQAEALBs/dfOuD1SD/odTSt3Gw6tepIl1kbrDjUcDLHuDuagn90QMZhuPzjlydIZ/4x9985yf37o1Nd9CAMwjBQCw5F2w/amRVTPTt0cMriwygP3+AA3IUU/ujZidKhPf6X1uaNXKy3Ze8brnygwA8OopAIAl7dJtu08dpPrBqOvvLzLA8HhUK9dGmQ2qAO1TTz8feWp/pBJ/d1P1V92R0Yvu33TGF5sPB3j13LECS9Yltz7+g4O5/v0p8qlNZ+dIUa1cG2lkvOlogNbLs9ORD+2NyAXOBYg0FdXQux9+77n3Nh4O8CopAIAl6aKbH39PHsz9x5Si+U33qRPVqvUR3ZHGowE4Jg9mIyb3HvvPprNz5NQd+T8ffvfZ/2uklBsfAOAVUgAAS8oFWx/tjq9f+5EU9fuLDNAdOXbYX9X8OYMAfId6EPWhvRFzRwoNUP3W0dHXXPW7160vdDABwMujAACWjEs++oVTBqnz6Yj6nxYZwH5/gMUn58jTz0U+cqBMfKqeSKMjFz688ay/LDIAwMvgLhZYEi76zS+9NVXpoRz1G5vOzjkijZ0U1eiapqMBeInyzOHIh56JiAIr8lM6UI2MX/Pgpjd+tvlwgJfOO6uARe8dN++6Iqr44xIP/8f2+5/m4R9gkUvDK6NafXpEiS1aOa+uj0w+cuHHv/rLzYcDvHRWAACLV87pwo99+aaI+oMppeYLy87wscP+7PcHWDrqQdSTeyL6R8vkV727hlefd/3OK1LzpxMCfBcKAGBRuvbO/ROHDn/zvoj8E0UGGBqLamJd+DP5P2TnXMOi54iSb8k56sP7ImYOlcmvun8+PDx+0c5Nr/vbMgMAHJ+vCWDRueQ3nnzzXDX7UIr67MbDv7XfP1nyD7Dk5aOTkQ/tK3LHm1J6JvdGL3vkhjP/qPl0gONzBgCwqPyLmx//ibl09I+LPPynKtLEeg//AMtEGpmItPq0yAVueXPOp6S5qf/8L2554mcaDwc4ASsAgEXjpz626wNR9z+cUmp8033qDEWaWBfRGWo6GoCFVve/dS7ATJn87vCdh846Z8vnfjj1ywwAcIwCACjugu1PjYzPTO2IXF9eIj8NjUYaXxdR4JxBAJqRcx1xeF/kmcNlBqi6f9Admbj0gU2v3V9mAAAFAFDYhdu+9vqopx+JyG8tkZ9WrI40dnKJ6CXD2X+w9LjBO7E8/XzUU/sjlTgxMVXf6AyPXfzgljf9t+bDAXw/AAX95Md2/7Oqnrs/pfSa5tPTsVP+h8aaj15CPPzD0uPm7rvLs9ORD+2NyHXz2SlNdXorNj50w1kPNB4OtJ7vCKCIf3Hz4z9R1XP3R47mn8CrblQT6yO6w41HLzUKAFh63Ny9RIPZqA/uiajnGo/OEf3e8Oi7H7j+rE80Hg60mu8IoHEXfexL/7iu8x9GipHGw3sjUY2vj6g6jUcDsMjkOvLknshzR0qk17k3ctFn3n3OZ0qEA+2kAAAade2d+ycmDz/9hZzrNzadnVasijR6ckSJfZ8ALE45Rz21P+LowQLh1f5Bd/gffPY9Z/9NgXCghRp/1RbQbpNTz2wt8fAfqYroz0aefLrx6CUvpYiqE6kzFLkzFKk34o0JsFjkOvLc0WOvt6vnIupBRLZ55+VKEZFTVeBMgPrkbu7/+4i4puFgoKX8DAY05pIdf3nK3NT0UxExWnoWXrmcI9LQSMTweKSR8fBVAk3LkY8ejpiZjDx71KKmpS7nwXBn4nvuf9+bnig9CrD8WQEANGb2yMzG5OF/6fmOHxNTRMTs0YjZo5GnnosYWRWxYlUkqwJgQeVcRxw5GPnIgUjf+qU6RRz/tE6lwNKRUmc2zdwQEb9QehRg+XO3BjSmirio9AzMs3oQMf1cxIG/iShziBa0w9zRY9fZ9HN/9/DPMjKYu7D0CEA7KACAxgzq+ntKz8ACGcxFPvjNyFP77T+GeZann4v6wN9EDJp/XR3NqHN91gXbn2r+zThA6ygAgEZcuO0rr6kin1R6DhbYkQNRH9pzbKky8OrkHPnQ3ojp5yPZ6L+spZQ6K2emzi09B7D8OQMAaEh3NPJM6SFoQJqdjji0N2J8vVcuwiv1/z/8z069wn9+fsdh4fVT7YwcYMEpAIBG9Ku67gxKT8Er8YqeI2anIx/eF2n8lPkeB1ohT+9/5Q//4QzApSjV9k8BC88WAKARP/C3Z30zImZLz0GDZg5FPnqo9BSw9MxORRw5WHoKmtY/+LXSIwDLnwIAaMTWramO1Hmq9Bw0bOrZiLpfegpYOupB5MP7Sk9Bw3Kun/3sjf/M//HAglMAAI3JKX6/9Aw0LNfH3gwAvCR5av+x12vSLrn6w9IjAO3gDACgOb3he/PM9HtLj0Gz8tFDkUbXRHSGSo8Ci9tgLvLMoXnZv28z+dJSp8H9pWcA2sEKAKAxn7nhrEdzSl8sPQfNSilFnj5QegxY9PLRgw7va6Ec+RsrP/fVe0vPAbSDFQBAo3Kn+repP/hMOKR6SZi3XxFnpyLnOlLSO8Nx5Rx55vC8fZw/sEtHyvGhnTuvsO8DaIQ7MaBRn33vhs/m1Pl06TloWK4jZo+UngIWr/5Re//bqK4/95mf/d5bS48BtIcVAEDjOtWa6+r6ubMi199beha+i/ncSDx3JGJobB4/EJaPPDM9f9ebn/+XhFznp7r9o1eVngNoFysAgMY98r7TpjtD3Z+MqvOl0rPQoLmZ0hPA4tU/WnoCGpTr+r/HzNH/+eH/5W17S88CtIsCACji4RvO++aBg51/mlN1fziwuh0Gs6UngEUrD+ZKj0BTcv173f7RH/rtf/0DT5YeBWgfi8SA4t7+sS9fGLnemnL8o9Kz8O3yPFcz6TVnhK8eeKH87Nfm78OSq2wxyvVgdx35V3/nX/6jO0rPArSX7wdg0fjJbbt+IHJcluv8gxGDMyJidYo0WnquNpv3AuCkN0RUjp+Bb5PryPufmr/PUwCUl/PRnAcHItJfp5T+a+T8W7/9L//h/xNWvAGF+X4A4ITe/pu7Hog8eMd8fZ4CAI6jHkR+7uvz93kp/mzlozf9wPx9IC/Xhg0b8tatW+vScwB8J3dhAADLSM4p79y50zsFC9q5c2fpEQCOyyGAAAAA0AIKAAAAAGgBWwAAeFFOrIKF5zoDoAlWAAAAAEALKAAAAACgBWwBAODEcszv2uT5/jxYDlwXADTECgAAAABoAQUAAAAAtIAtAACcUB0RaR4/zypnAIByrAAAAACAFlAAAAAAQAsoAAAAAKAFFAAAAADQAgoAAAAAaAFvAQDghJzaD0uP6xaAE7ECAAAAAFpAAQAAAAAtoAAAAACAFnAGAADNyWGDMnwn1wQADbECAAAAAFrACgAAXpQfJ2Fpcc0CcCJWAAAAAEALKAAAAACgBRQAAAAA0AIKAAAAAGgBhwACcEJ1RKTSQwAvi0MAATgRKwAAAACgBRQAAAAA0AIKAAAAAGgBBQAAAAC0gEMAATixep4/L4cTyuA7uS4AaIgCAIAT8kwCALB82AIAAAAALaAAAAAAgBZQAAAAAEALKAAAAACgBRQAAAAA0ALeAgBAY7xVAI4vzeNnuc4AOBErAAAAAKAFFAAAAADQAgoAAAAAaAEFAAAAALSAAgAAAABawFsAADixXEeexzPFc84R2Rnl8G3m+bqo03y+UwCA5cQKAAAAAGgBBQAAAAC0gAIAAAAAWkABAAAAAC3gEEAAXtR8Htnn+D8AgHKsAAAAAIAWUAAAAABACygAAAAAoAUUAAAAANACCgAAAABoAW8BAOCEcngLAADAcqEAAODFzXcDoAWAb+e6AKAhtgAAAABACygAAAAAoAUUAAAAANACCgAAAABoAYcAAnBC9QJ8prPOAADKsAIAAAAAWkABAAAAAC2gAAAAAIAWUAAAAABACygAAAAAoAW8BQCAxuTwFgA4njSPn+UaA+BErAAAAACAFrACAIAXN58/J1oCAC8039eFawyAE7ACAAAAAFpAAQAAAAAtoAAAAACAFnAGAAAnNN9biR0BAMfnCAAAmmAFAAAAALSAAgAAAABaQAEAAAAALaAAAAAAgBZwCCAAJ1RHRJrHz3MIIByf6wKAJlgBAAAAAC2gAAAAAIAWsAUAgObYAwAvNN/XhGsMgBOwAgAAAABaQAEAAAAALWALAAAvaj5XE9sBAMc339cZAByPFQAAAADQAgoAAAAAaAEFAAAAALSAAgAAAABaQAEAAAAALeAtAACc0HyfJu4tAHB83gIAQBOsAAAAAIAWUAAAAABACygAAAAAoAWcAQDAidUx/5uTbVCGbzff14VrDIATsAIAAAAAWsAKAABOqI75bYotAICF5xoD4ESsAAAAAIAWUAAAAABACygAAAAAoAUUAAAAANACDgEE4EV5CyAsPNcFAE2wAgAAAABaQAEAAAAALaAAAAAAgBZQAAAAAEALOAQQgBOb71P7nAIIL+S6AKAhCgAATmi+n0k858DxuS4AaIItAAAAANACCgAAAABoAQUAAAAAtIACAAAAAFpAAQAAAAAt4C0AALwobwGEhee6AKAJVgAAAABACygAAAAAoAUUAAAAANACCgAAAABoAQUAAAAAtIC3AADwovI8Hk+e8/x+HiwX83qdzd9HAbDMWAEAAAAALaAAAAAAgBZQAAAAAEALKAAAAACgBRQAAAAA0ALeAgDACdURkebx83I4oRwAoBQrAAAAAKAFFAAAAADQAgoAAAAAaAEFAAAAALSAQwABeFHzeWifQwDh+Ob7OgOA41EAAHBidWgAYKHN93Uxn6/uAGBZsQUAAAAAWkABAAAAAC2gAAAAAIAWUAAAAABACzgEEIATmu/z+pwBCMfnLQAANMEKAAAAAGgBBQAAAAC0gAIAAAAAWkABAAAAAC2gAAAAAIAW8BYAAE6ojog0j5/nLQBwfN4CAEATrAAAAACAFrACAIATm++fEi0BgBea72uinufPA2DZsAIAAAAAWkABAAAAAC2gAAAAAIAWcAYAAC9qvk8ndwQAvJDrAoAmWAEAAAAALaAAAAAAgBZQAAAAAEALKAAAAACgBRwCCMAJ5YhI8/x5DjuDhVWXHgCARcsKAAAAAGgBBQAAAAC0gC0AALy4+Vyzbw8AvJDrAoCGWAEAAAAALaAAAAAAgBawBQCAE5rvVclWOsPxuS4AaIIVAAAAANACCgAAAABoAQUAAAAAtIACAAAAAFpAAQAAAAAt4C0AALyo+Tyd3FsA4Pjm+zoDgOOxAgAAAABaQAEAAAAALaAAAAAAgBZwBgAAJ1bP8+c5BABeyHUBQEOsAAAAAIAWsAIAgBOqIyLN4+f5oROOz1sAAGiCFQAAAADQAlYAAPAi5vs3e2sA4IUW4joDgBeyAgAAAABaQAEAAAAALaAAAAAAgBZwBgAAL8oJALDwXBcANMEKAAAAAGgBBQAAAAC0gAIAAAAAWsAZAACc2HxvTHYIALyQ6wKAhigAAGiM5xxYeHXpAQBYtGwBAAAAgBZQAAAAAEALKAAAAACgBRQAAAAA0AIKAAAAAGgBbwEA4IS8BRCaMZ/XhWsMgBOxAgAAAABaQAEAAAAALaAAAAAAgBZQAAAAAEALOAQQgBc336eTOaEMvp3rAoCGKAAAeFGe/2HheQsAAE2wBQAAAABaQAEAAAAALaAAAAAAgBZQAAAAAEALKAAAAACgBbwFAIATquf587wFAI7PdQFAE6wAAAAAgBZQAAAAAEALKAAAAACgBRQAAAAA0AIOAQTghOb7YDKHAMLxuS4AaIICAIAT0wDAwnNNANAQWwAAAACgBRQAAAAA0AIKAAAAAGgBBQAA/H/s3XnQXcd55/df9znn7su7Ai82gvsuS5ZkaSxbNr1MYnlipzSpeMqZiSuuJM54phzbiTSS7MTFVDk1TjwZaTJjO554bI/t8aLEsiRKokiKkWSJWkiKIglShAiAIECA2IF3u+9dz+n8cQmSorC874u3zzn33u+n6i2Ci24/InH6dj/n6acBAAAmAE0AAQCpoQcg4B/PGADgcqgAAAAAAABgApAAAAAAAABgApAAAAAAAABgApAAAAAAAABgApAAAAAAAABgAnALAADgshJJZgs/j1sAgEvbyuci2cLPAgCMFxIAAIAr28qdCRkA4Ltt9XPBMwYAuAyOAAAAAAAAMAFIAAAAAAAAMAFIAAAAAAAAMAFIAAAAAAAAMAFoAggAuKyt7iVGD0Dg0nguAABpoAIAAAAAAIAJQAIAAAAAAIAJQAIAAAAAAIAJQAIAAAAAAIAJQAIAAAAAAIAJwC0AAIAr2sru5NwCAFzaVj9nAABcChUAAAAAAABMACoAAABXRgkA4BfPBQAgJVQAAAAAAAAwAUgAAAAAAAAwATgCAAC4rESS2cLPo9IZuDSaAAIA0kAFAAAAAAAAE4AEAAAAAAAAE4AEAAAAAAAAE4AEAAAAAAAAE4AEAAAAAAAAE4AEAAAAAAAAE4AEAAAAAAAAE4AEAAAAAAAAE4AEAAAAAAAAE4AEAAAAAAAAE4AEAAAAAAAAE4AEAAAAAAAAE4AEAAAAAAAAEyDMOgAAwOSwvRU5E2QdBpArxiVZhwAAmBAkAAAAqQk757MOAQAAYGJxBAAAAAAAgAlAAgAAAAAAgAlAAgAAcFlGHE4GRo5LeG4BAJdEAgAAcHlJ0s46BAAb5VpZRwAAyCcSAACAy7LJ4FTWMQDYGDvon846BgBAPpEAAABcVtBd+3bWMQDYGOu6B7OOAQCQTyQAAACXVbvw9Fcl47KOA8D62ZXjj2QdAwAgn0gAAAAuq/jMX58I4s6+rOMAsD7GDU7MfuujPLMAgEsiAQAAuKxyubwSrr78yazjALA+0dr5T/eXTy1lHQcAIJ9IAAAALuv+++/vzj33Z/+PiQdHs44FwJUZuXbjyIP/4ZFHHlnNOhYAQD6RAAAAXFlr8VRp6dDvZh0GgCuLVk/9cf3cswcl0bcDAHBJJAAAAFf08MMPn5791r//TNBvfSXrWABcmo27z809/Sd/3G63T2YdCwAgv0gAAACuxnWWl48t7PujD9qk/0LWwQD4TsbF56YP3fe+sHf2xS984QuDrOMBAORXkHUAAID8O3r0aOeGhak4sMmz3ambflzGVrKOCYBknFurv/y1X24cefgbDz744PGs4wEA5BsJAADAuhw+fHjl5kbSCQerX+pN3fy9zobzWccETDKTxMebx77wTxoH73vsoYceOizO/gMAroIEAABg3V588cXFO7eXl6PTz3ylN3/3QhKWbsk6JmASBf3WV+ae+9P/vnj0kX0PP/wwm38AwLqQAAAAbMihQ4dau2ZqZ+fPPPrIoDK/Py7N7HU22JZ1XMAksHH/UPnct39r22Mf/hd1re3/7Gc/ezrrmAAAo8NkHQAAYHTdc889tUKhMH3+9p+5pzN180/Fhfq7nA12ZR0XME5skpw2/dWvRctHPt149qMPFZL2+c997nNLWccFABg9JAAAAFvinnvuCbvdbhT/wIcWWtPX3doNS3PWhFSaAZtgjEuCXvtcsHricO0L9x4pFot9OvwDAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAPLGZB3AKHLP3ltQXKx2V7uz3f4g9X+Hq2s9a2xgjp9cbbQGSZD2+OOo3+mP5b/HqBTFWcdwJR/97OHDWccAAEDafuYnbrgh6xguJ+44k2hgN/q/O30uaa/2bc9HTFuhWja5XhONgvmFYGXn7FRvtlHI7N/lmZdW22/+uX/Rymr8cUAC4BLOfPbXdgwCvTVJ4jtipzvi2N3snNsradpJFecUJonLOkxg5P3enzyrxPEsAQAmhzVGv/hzd2UdBjCynFNijFYltSTXNsYsyumQMe6ADbQ/TuyzvXb1mbt+5t7cJqSyFGYdQB6c+8y9jV7Ue3e3P/iJJHH3LPZ6dzlHcgTwzQZGyYAEAABgclhWmMA1MUZWUmP4Y/TKu6S3OmeUDCTJKSqudg7e974njPQ1Y+wXY9t+4Jaf/NfdDMPOjYmdgs595t5GO2j/w34/+S8Hsd7p5DZc6gTg2vzBXzynbo+KPADA5CgWAv03P3tH1mEAE8atGpn7rfTvb/zp2v3G3JtkHVFWJi4B8PL9H/iPe07/uD+If8I5lbKOB5hkf/hX+9XuDLIOAwCA1FTKkX7+Z27LOgxgYhmjk9aYvw5M8Xeu/09+87ms40nbxBwBOPHQr/29fpz8Lyud/tuMmbi8B5BLNuBZBABMlsBw9A3IknNaiJ37pwPX+cWD973/i1EQ/s97f/KfP5J1XGkZ+wTA8Qc/+LPdbvwbK2u92yWJzT+QHyGPIwBgwthwLC8eAkaOMbJJ4n6km/S/dPC+9z0cKvqN63/qn38167h8G9sEwMsPfOj2XpL8Xqs9uCfrWABcmg1pvQEAmCw0AQRyxySJfrzr+j966FPv/4Q1+ic3/L3fPpl1UL6M3er75fvurRz9zPt/u9UdPNXrxfdkHQ+Ay7Nm7KYgAACuKCADAOSSMbJx7N47SLT/hU/9s1937t6xXKiO1f+pkw986J1t03q203fvc84Vso4HwJUFYzUDAQBwdZYEAJBrLnHNQZz85sH7Vr986L4PXJd1PFttbJbfxx/44PtXuoO/jWN3fdaxAFifgCaAAIAJE3L8DRgJzun74yR+4vCnP/CfZh3LVhr5HgCH/+ZXpkyx8OetzuA9WccCYGOiiEUQAGCyRBHJb2CEzPb68ccOfvL9v/PSE9X/4UfuvXfk768e6dX32Yd/fVcchY/0+gmbf2AEFQsjn4MEAGBDSnz3ASPFGNnEuV/a87bV+5/6k/dVs47nWo1sAuDYpz9069Ja95Ek1p1ZxwJgc0oFrkICAEyWIt99wEhKEv14ZUqPvPCJX9tNeLiKAAAgAElEQVSedSzXYiQTAMcf/sC72nH/q3GivVnHAmDzisWRnIIAANg0EgDA6HJOb45N/0v7P/q+G7KOZbNGbvV9+sFfe0t7LfmMc5rJOhYA1yaiERIAYMIUSAAAI805d0tQMl848Nf/bHfWsWzGSK2+Tz7wP92w0u3fnySumXUsAK4diyAAwKQp0AQQGHnOuetcmHzu2Oc+NJt1LBs1MgmA05+/d6HV6z4cJ24h61gAbA3KIAEAk4YGuMDYuK3d6n/68d//hUrWgWzESCQADnzml4qrrdb9ceJG9qwFgO9W4BpAAMCECfnuA8aGk945vav5Z1nHsREjMQNFrvxvBrF7S9ZxANhaHAEAAEyaIgkAYKzEsXvvoU+//1ezjmO9cj8DvfTAB/9htx//11nHAWDrRZyDBABMGJLfwPiJB+63Xnr419+RdRzrkesEwLFPf+jWTnfwe8YYdgnAGCoXo6xDAAAgVcUiCQBgDBU6re5Hj3zqg9NZB3I1uU4AdJP+HzmnetZxAPCDRRAAYNKU+O4DxpJz2tuPBx/JOo6ryW0C4KXPfuC/jRO9K+s4APgTBkYlSiEBABOiUgoVWApbgXEVO/2jw/d98J6s47iSXN5DcuHz906dXWn9r1nHcTXGSJxOGA+W/5aX1evHXv/d1GuhOucH3j4fAIC8qNf8Lr2dcypEJNYvJY6T4eIdI805J+eyjuLyjJEduPjffP7ee9/yI/fem8sFbi4TAIvttX/pnJvPOg5JstaoXAhUKlgVQqtCZBUGRpYJBBPihRMtxYm/mXamVlRrqevt8wEAyIvZWtHr54eB1fXbR+pKcmBTEufUHyTq9RP1Y6e1bqxOL85FcsA5d9f17+j8j5L+t6xjuZTcHQE48rkP3tjvJ/8o6zhq5VA7Zkq6caGinbMlzdQLqpVDFULL5h8TJQr9ThO1Go0AAQCToVr1++7N93c2kBfWGBWjQPVKpJl6QbvnyrppR00LMyVVctBno98fvP/05++tZR3HpeSuAiDuJvdKymxHMFWLNF0rKAzY5AOSFHheS1RK2U/SAACkoVz2u/T2/Z0N5JkxUr0cql4O1e3HWlwdaKnVy+qY7+xqa+2XJeXuWHuuponjD/z6nsEg+QdZjF0uBtq7vaL5ZpHNP/A6vt8mlMtUAAAAJkOl4vc7jwoAYKgYBdo+XdTe7VWVM6oIGMTxrzz++7+QuzM5uZolur3eb0gqpDmmMdJ8s6jdc2UVmDSB7+K7W3G5krtCJAAAvPB9BSAvsYDvVIysds+VNd/023/j0szczO6Zf5rBwFeUmx3vyQfeV42dfjbNMaPQ6rptFU1xBhm4LO8VACUSAACAyeD7CEDIGQDgkqZqkXbPlVO/hnMw6P/jVAdch9zMEt04+C+cc9W0xhtmg0q89QeuwnsPAI4AAAAmRNXzEQD2/8DllYuBds+XU66UMTe+8IkP/FCKA15VbqaJQRz/XFpjFUKrXXNlsqTAOvh+TkqlkDOLAICxF4ZGUeT3CEDE2ha4okJotWe+kmoSwIX6+dQGW4dczBIvP/SB6+LYvSuNscLAaNdc6ZUMqeOHH36u8hN5niCNkZrNVFt/AACQuumpkqzn8mPWt/zwc/WfMJB2zpZkU9oJJ0n83sOf/69K6Yx2dblIAPQGyc/JmBRicdoxU6JBCrABxsj7BNls5GZOBADAi6mG32R3YI33BAMwLoqR1fapdNafLlHTrcz9dCqDrUMuEgCDgX40jXFmGwWVCrn4vwyMlNDzgqLZyKIzKwAA6Wl4/q7jBRewMbVyoEZKt1E56T9KZaB1yLz99uO//wtRkiTvkPM7aRUio+lqYVj5AWBDosCq14+9fX6dmzgAAGOuXit6XYdGgWGdC2zQfLOoVnugOPE7TuLcu/2OsH6ZJwAW9sy8e63Xr/qeseabRRnDrAhsRiEyanX8PT8kAAAA4274Xefvu7QQGq+fD4wja6S5ZkGnLnS9juOMbnnpvl/dteenPnzc60DrkHk9fDfu/13fY5SLVpWi366rwDjz3QiwWiUBAAAYb9Wq3/du3KgDbE69HCoKPR+hcTJ92ff4HWR9sp8pnHmr7yGma3QYB65FwfOiolQKFXB2EQAwpsLQqFj0nADgexTYFGPS2S8mid7mfZB1yDwBkDh3k8/PD6x4+w9cI99vFawxatZpBAgAGE/NRknW+O53lfmyHhhZ9Uooz4+onHSr3xHWJ9OZwn30Pw+c0x6fY6TxHxMYd2FgvF8FODvDVYAAgPE0N+M3yW3t8BpAAJtjjVQt+X1pbIxu9DrAOmXaBPBo7ebb1Ot6rbeolUPREAW4dlFg1E38tUidnSnpwAvePh4AgMzMTJXktwGg9fr5wCSoFK1W2wNvn58kbs+Bz/xS8Zaf/Nd+Ow5eRbYVAG5wh+cRVPTd0AGYEL5LC6emOAIAABhPvr/jaAAIXLuK5z4dkgmCvr3N8yBXlW0CIEnmfX5+IbKylEMBW8J3c6GpBgkAAMB4mmr6/Y7z3awXmARR6P/Iq7OB1/3vemQ6W8TO1Xx+PpMhsHV8VwCUSoGKBRp2AgDGS6kUqOC5IbX3K8yACeF7/xiEtuF1gHXIdIdsrKn7/PyQ61CALeN7QjTGaHaWRoAAgPEyP5PCDQC89AK2hO/9o0tM0+sA65BpE0DFruGzYcmwGyoNUYCtUAj9P09zM2W9fKLldQwAANI0O1uR7+/PNL6jgUkwPD3u71mKkzjzCoBMEwBOrurz8z3/9wMmitHwDUNv4O8mgBkaAQIAxsxU0+uFVypGljUvsEV8V+tYJ68V8OuKIdPBadAHjJRSwfNNAJ4XSQAApG266fd4W5Hyf2DLeN7/y8ll3vCKGQPAuvk+Y1ivF7xPvAAApMUaqV6PvI5R9JycBzBemDEArFvJc5f+MLSanaERIABgPMzNlhR4bipGBQCAjci2CaAkvweWnOfPByZLIZR8P1M7tlV09lzH6xgAAKRhYXtVvr83hxUArHeBreF7/5h9qSspQwDrFlij0HPvjm3zFa+fDwBAWrbN+a1qiwIrWmoB2AgSAAA2pBj5PQYwM8NNAACA8TAz7TcBMLz+DwDWjwQAgA0pRH4XG5VyqFIpB6eTAAC4BpVyqHLZ7/eZ7948AMbPmK+y6QEAbLWS5wSAtUY7tpd1+MiK13EAAPBp50LF+5XXxciItS4wSpKsA6ACAMDGpPG2Yfu2svcxAADwKY2eNiWuAASwQcwaADYkDIwiz1cazXEVIABgxPn+LiuERgEdAAFsUPZHALgFEBg5pciqP4i9ff50syRrpITnFwAwgqw1mmoWvK5Di6FlnQtsNd/7x+xPAFABAGDjyp6PAYSh1dwsVQAAgNG0fb6sIPC7zC5T/g9gEzKuAPCdYqEEAPChVPDfdGj3zppOn+14HQMAAB92LVTk+3tyeP6fdS6wtcZ//0jqEMCGFUIr38cOd9AIEAAwonYsVL1+vjVSMWIZD2DjmDkAbJgx/jsPz82WFYY0NwIAjJYwsJqeKnodg+7/ADaL2QPApvi+DtAGRju3+79CCQCArbRrR1WB59tyfPfiATC+SAAA2JQ0mg/t2lnzPgYAAFtp907/yWsqAABsVvbXANIEEBhJw8VHIsnfW47t89wEAAAYLQvbfDcAdDQABLxJo0l9tkgfAtgUa6RS5LcEsdksqFSizBEAMBrK5VD1ut/3a+VC6L0RL4DxRQIAwKaVi36nEGuM9uziGAAAYDTs3VWRMZ7P/3v+7gUw3phBAGxapej/7fwuz1cpAQCwVXbu8P+dVSEBAOAaZNsDIEkk5/EchHN+Px+YcOXIysh5fcy2z/m9SgkAgK1gjLR9rux17WmNUSm0rG8BX3zvH3NwfIcUIoBNM8b/VUSVSqh5kgAAgJzbPlf23remVDDyfMIAwJgjAQDgmqRRinjDdXXvYwAAcC2u3+v/u6paysEFXgBGGgkAANckjcXInp00AgQA5NvuHRXvY3D+H8C1YhYBcE0KoVHg+T6iRj1SsxF5HQMAgM2aakSq1/x+TwXWqBCydAdwbXJQR+SziYnz/PkApOEbiZX2wOsYN+6t65v7znsdAwCAzbjp+oZ8rzmrpcD7GAB87x+zf4ZJIwK4ZmmUJO7ZyXWAAIB82rMrjfJ//1fvAhh/JAAAXLOq567HkjQ9XVSlzOIHAJAv1UqoZtP/bTWc/wewFZhJAFyzwBqVIr/TiTXcBgAAyJ8br6vJcysclQvWe78dAJOBBACALVFL4e38dSmUWAIAsBF7dvk/olYr56BtF4CxkPFsknj+fJoAAmmplQOdXe57HWN+rqxSMVCnG3sdBwCA9SiVrObn/Jf/10pWrGmBNIz/c0YFAIAtEQVGhdBveWIQGN18A8cAAAD5cMv1dVnrdzldjIzCgPJ/AFsj+3oibgEExkatFOj8it/rAG/aW9Mz+xe9jgEAwHrctLfhfa1ZK4asZ4G0+N4/+i6AXwcqAABsmWoKVxRNTxVVr2WfuwQATLZGPdTUVMH7ONUSy3UAW4cZBcCWKRWsQs85AGOMbr+p6XcQAACu4vabmjLGb2l+GEhFz7fsAJgsOXiN5rumiZopIE3VktVSy2+Tvhuuq+qxp855HQMAgCvZu6cq3+vMWin0PgaA1xv/MwCkFAFsqeFixa9qNdK2FLouAwBwKTvmS6pVI+/jpHHFLoDJQgIAwJYqF6yCFGaW225s+B8EAIBLuPkm/zfSBFYqUf4PYIsxqwDYUsZI9bL/KoA9u6qy3IoEAEiZtUbX7ap6H6dRCeW5xQCACZRxDwDfZyy4BxDIQr1stdjy++wVi1Z7d1d1+KWW13EAAHi9G66rqRBZ+V5j1sv+xwDwRuO/f6QCAMCWKxWsotD/a4vbbuYYAAAgXbfdVPM+RhQauv8D8IKZBYAX9RQaF22fL6lR89+ECQAASWrUIs3PlfyPk8JROgCTiQQAAC8aFf+LF2uN7riVKgAAQDruvqMpm8LB/HqF7v8A/Mg2vUgLAGBsRdaoGFp1+37vO71pb02PP3leccLDDgDwJwiMbthT9b62LBetImtYwwJZ8L1/zEFjz4zrixKRAQDGV6NidWYp9jpGsWB1681TOvAizQABAP7cekM1peZ/gfcxAFzO+O8fOWAEwJt6OdCZpb73cf7OD9yuO/7+T3sfBwAwuZonPy61T3sfp1bihC4Af5hhAHgTWKNK0f8041onFPXOeR8HADCZov55uTX/m/9qKVBgc1AjDGBskQAA4FWz6r/QyBijwuqz3scBAEymwvLTSqH3nxo0/wPgGQkAAF5Vi4GCFGaaaPUFKen5HwgAMFnirqLWYe/DBHb4nQkAPuWgBwBNAIFxZszwjcaF1YHXcVzcVbm1X+3693gdBwAwWcqt5+Ri/wnmZjWUMaxbgez53p9miwoAAN41q6Gc8z/hRYv7ZHIwsQIAxoORU7T0jPdxnHNqVnLwXg7A2CMBAMC7KDCqllLoBTBYVWntgPdxAACTobS6X2bg/5rZWjlUGND8D4B/JAAApKJZTedcY2HpSeWhvAoAMOqcCiv7UhmpSfM/ACnJuNYo8fz59AAA8qJatAqsFPt+7DsXVOwcUbd0veeBAADjrNg+LHUueB8nDPTKlbmsWYHsjf9zSAUAgFQYk86VgJJUXHoqlXEAAOOrtPRkKuMMm/+lMhQAZF0B4BkFAECuNEpW55edfK90zNpJRd0T6hd3eB0HADCeCr0TUvuM/4GcU6MUsF4F8sL3s+i7EnYdqAAAkJootKqW0jnnWF5O580NAGD8lBafSGWcaimg+R+AVJEAAJCqqZSaAZqVo4r6/s9uAgDGS9i/ILN6LJWxpmvjXYwLIH9yMOv4rLPgDACQN5WiVTGUugPPz6aRSsvfUH/2x/2OAwAYK+XFxyTjf/1YiqzKBSPWqkCejP/+kQoAAKlL642HXT6oaHA+lbEAAKMv6l+QWXkhlbHSqogDgNfLQQUAgElTKwWKgoH6cQpvWC48qsG293gfBwAw+sqLX0ulI38UGNVS6okDAK9HAgBA6owZvvk4szzwPpZdeVHX3RnK1vd4HwsAMLqS5SNaPHRESiEBMF3j6j8A2eAIAIBMNCuhApvC6sdI7YP3+x8HADDS2gc/k8rmP7BGjTJv/wFkI+MKAN9NFsa/iQMwqoyRGmWjCy3/F6J2zx9QZfGAzNQt3scCAIyeZPGguhfSOfvfrAQyKTQZBLAZvveP2Zf+UAEAIDNT1TC1aXDt0GdTGgkAMGrW0qoUc07NCm//AWSHBACAzISBUa2UzjTUXTyqwZlnUxkLADA6krNPq7d0NJWxmtVQYZD9G0AAkyvbIwCJOAEATLiZWqTltY5MCt2Q2i/cr/r8ncpD+RUAIA+c1g49mNJ60WmqErI2BfJs/E8AUAEAIFuF0KhRSScX2V85pcHJJ1IZCwCQf/Gpx9VbOZXKWI1yqEKYg9U/gImWg2sAKQEAJt1MLdBK2/+VgJLUOvApNbe9SbKFVMYDAORU0lPrwGeV1lpxuhakNhaAzUqjSX22qAAAkLlCaFRP6UqkuLui7uGHUhkLAJBf3Rce0KCzlMpY9ZLl7T+AXCABACAXZuvpFSStHfmykva51MYDAORL0j6vtaNfSW282UaU2lgAcCUkAADkQhQYNVKqAnBJX90DH09lLABA/nQPfEwu6acyVqMcKKLzP4CcyLgHANcAAHjNTD3Qckq9ANZOPaviuW/Jzt6ZyngAgHxIzj2n9unnUhtvps7Zf2B0jP81AFQAAMiNNKsAjDFa/fYnJBenMh4AIAdcotbzn0xtuGaFt/8A8oUEAIBcSbMXwGDtrPrHvpjaeACAbPWOPqx+63Rq483UcnDhFgC8DgkAALkSplgFIEmtQw9LvZXUxgMAZKS3rNYLn09tuGYlUMjbfwA5QwIAQO7M1sPUTkglg446B9MrBwUAZKNz4BNycTeVsawxvP0HkEtjPjPRBBAYRWEgzdRDnVtJpyFg6/g3FG17i4K5u1IZDwCQruT8fq2deDK18aZrVmEgsQ4FRo3vZzbx/PlXl30CgEsAAFzCVNlqqSUNUujRZ2R05sn/oK8kP6ZBDqZFAMDWCTXQ99v/T8WU1oRhIE1XQtagwKjyvT/NGCtdALlkrdFsLdSppXSqAAquq9vMPn2z96ZUxgMApOOuwtMquk5q483VQxmO/gPIKXoAAMitRiVQKUpvFbXgjmpbcC618QAAfm0Lz2vBvZTaeKXIqJ5iI1sA2CgSAAByba6RXqGSMUZ3B/tklcK5AwCAV9bFuts+LZPi6/g0v7MAYDNyMEvRBADA5ZULRtWi1Oqm8yyX1NJd0be1r39nKuMBAPx4U3G/SmqlNl69bFUuGLH2BEbZ+O8fqQAAkHvzzUhpTsa7zGHNhYupjQcA2FozdlE79WJq4xkjzdWj1MYDgM0iAQAg96LAaKqa3plKK+lNwZMyObiqBQCwMVZO3xM9pTT78E1Vgleu/QOAfCMBAGAkzNYjBSnOWGXX0t3R/vQGBABsibui51Rxq6mNF1hppp6DU7UAsA4Zz1aJ/Jf1jvcZDmBSWCPNN0KdXOynNuZuHdLZcE4nBttSGxMAsHnbg7PabV5Idcz5RihrWG8C48F3D4Ds7wilAgDAyKiXrarF9KYtY4zutk+pZLqpjQkA2Jyy7ekt0TdTXV5Xi5Zr/wCMFBIAAEbKtmakFG90UqSu3l78pqgmAoA8c3pL+E0FSXoJW2Ok+Sal/wBGS7azFicAAGxQaKXZWqizy4PUxmzEZ3VrdEjP929ObUwAwPrdHh3StDuT6piz9VCRNaw1gXEy/icAqAAAMHqmKoEKYboz6E3m25oJuBoQAPJm2i7rRvN8qmMWQqOpCqX/AEZPDuqWfKZYfKdwAGTBGGn7VKijZ7oyKZ0HMM7prcE39IX4hzXIw9QJAFCovt4aPi65OLUxnXPaPlWUofEfMKZ870+zRQUAgJFUioyaKb99Kbg1vTl6OtUxAQCX972FfSq6VqpjTlVDlaIc1PECwCaQAAAwsuabkYKUZ7HtOq694UvpDgoA+C43hEc1746nOmZgpbkGVWAARhczGICRZc3wVoATF/qpjnt38Iz+zjt+SK52XarjAgBesXpUrSc+m2blvyRpWzOU5eU/gBFGAgDASKuVhncwr7RTPP+ZDLT29B+r+f3vlwr11MYFAEiut6rlZ/5ELk43+duoBKqVaPwHYLRlnADw3aSPJoDAJNjWDNTuxRrE6T3vcXdJraf+rapv/2XJkEsFgFS4RO19f6C4fSHVYcPAaL4RiHUlMO7G/x5AegAAGHnWSNub6W/CuxeOqrv//019XACYVN39f6XOucOpjumc08JUROk/gLFAAgDAWKgUbeq3AkhS66WvanDsb1MfFwAmzeDlR9R66WupjztdC1UusPsHMB5IAAAYG/ONUIUw/Wltef/HlSweSn1cAJgUyfIhrTz3sdTHjQKj2Rrn/gGMDxIAAMaGMdLCVAbn8ZNYy0/+oVz3fPpjA8CYc91FrXzzj+XiQepjL0xHstT+Axgj2Xau8t4D0A1/AEyMYijNVAOdX033bqiku6q1J/+dqt/3y5ItpDo2AIytpKe1J/9vxZ3l1IeerQcqhWItCUwS5zz3AEw8fvj6UAEAYOzM1AKVMjiv2V08pvZT/05y2U/uADD6nNpP/6G6i8dSH7kYGU1XKf0HMH5IAAAYO+aVWwGMSf+tTfv0fnWf+/PUxwWAcdN97i/VPvVc6uNaK+2YDmWo/AcwhkgAABhLhdBoYSrKZOzWS4+pd+hTmYwNAOOgf+g+tY6m3/FfkrY1AkUBu38A4ynbHgBK5LkJgOfPB5BntZJRs2K0tJZ+Sf7KgQfVKNQV7fnh1McGgFEWv/RFrRx8UFL6m/CpaqB62Yr1IzCpxn//SAUAgLE23whVjNJfRBojrTz3McWnHk99bAAYVfHpJ7S8/2PKYvNfKhjN1Tn3D2C8kQAAMNaMkXZORwqymO2c0/LTf67kwvMZDA4AoyVZPKjlp/5MLkn/7Zu10o6piHP/AMYeCQAAYy8MpO1T2Zx4cslAy9/8A7m1lzMZHwBGgVs7rpUn/q1cMshk/O3NUCEv/wFMABIAACZCtWg1Vc1mykv6HS0/9jtynXOZjA8AeebaZ7Xy+O8q7ncyGX+6ZlUrsSQGMBkybgIo0QQQQFrm6oE6vUSdfvrzQtxZ1vLX/6Xq3/crspX51McHgDxynXNafuxfKW4vZzJ+uWA0WwvEehHAkO/9Y/ZzDelOABPDmOHdzjajM55xZ1mrj31Ern02mwAAIEeS9jmtPPoRxe3FTMYPrLQwFXLuH8BEyUEFgEcUAAB4g9Aa7ZwOdexcX1ms+gYXKwHe8atUAgCYWK5zTquPfkSDTjabf6Nhg9jQGtaKAF7jez5I/2bq70IFAICJUy5Y7ZiOMhs/7q5o5dEPy62dziwGAMiK65zTytez2/xLw6Z/pQyuiAWArJEAADCRaiWr6YyaAkrDJMDyox8hCQBgorj22cw3/zPVQPUyS2AAkynjIwCJaAIIICtz9UCD2Gmlk009Vtxd1vKjH1HjHb8iU9mWSQwAkBbXPquVDMv+JalWtJqtW7E+BHBp479/JP0JYKJtawSZloFeTAJQCQBgnOVh818MjRamxrv9FQBcDQkAABPNWqMdU6GCDI+Cxt1hY0DXeim7IADAk2T1mFYe/XCmm/+LDWDp+A9g0pEAADDxwsBo10ykLNeFcW9VS1/7V0oWD2QYBQBsreT8fi1//cMadJYyi2HY8T9UmGWmFwByIgd1UPQAAJC9YiQtTAV6+UJfJqNXRMmgreXHf1eNN/+87Pz3ZBIDAGyV5PQ3tPTUn8ol/cxicM5px3SkYiSxJgRwdb73j9nPQ1QAAMAraiWr+Ua2edEk7mnxyT9QfPLrmcYBANdicOQBLT7xR5lu/iVpWyNSrcRyFwAuYkYEgNeZrgaaqQbZBpHEWnryT9U/8DfZxgEAG+bUP/A3WnruE5LJ9k3XbC3QVIbXvQJAHuXgCAAA5MtsPVDipMW1OLsgjNPKC59TpbOo0t0/J5mMkxIAcDVuoM7Tf6i1k09ldpTqombZaqbGvAkAb0QCAAAuYb4RKE6cVjpJpnGsvfy4ku6iym/572TCSqaxAMBlxV2tffP31DmXfSPTeslqW5MlLgBcSsaHXZ3nHgtu+AMAm7C9EShJnFrdbOeRztmDSr7+f6j61l+UKc9lGgsAvJFrndLqN/8v9VdPZx2KqkWj7Y2A9R+AzXGe96c5uIyEg1EAcBnGDG8GKEVZRyL1Vk5q+au/JXf+uaxDAYBXubP7tPz1/z0Xm/9SZLRjOlTGpw8AINeojwKAK7DGaNdMpGPn+uoOso0l7rV1/tHf1dnmu3S6eHe2wQCYcE67ut/Q1NLjWQciSSqE0q6ZMA8v1wAg10gAAMBVWCPtnA517PxA/Qz7AkqSUaK5xS8pDo7pycH3KnEsdwGkK1Rfb48e11RyIutQJElRIO2aDmWZDgHgqjgCAADrEAZGu2cjRTloKm2M0ULyot4dfl4V0846HAATpG5W9O7w85rJ0eZ/92ykMGD3DwDrkXEFgO/u2k5+uzgAmCShlXbPBLmoBJCkanJB7w4e0tPuHToRL2QdDoAxtys4qbvN1xW4jM9DvaIQGu2cDhRa1noAtsr47x+pAACADQgDo90zoQo5OUAVuL7eokd0e/Ccxv0LC0BWnO6KntOb9UhuNv/Dsv9AEW/+AWBDSAAAwAblLQlgJN2ob+ld0SMq2V7W4QAYIyXT1Q8WHtHe5FtZh/KqKJB2z4SU/QPAJpAAAIBNCGy+kgCSNJWc0rvNA9pus7+OC8DoWwhO6YeCh9SIT2UdyquKodGeWTb/ALBZJAAAYJMCa7RnNlIpys9CNFJPbzNf0psLT8kajgQA2DgrpzdF+/RWfVmh62YdzoR3/VQAAB47SURBVKuKodGumUAB7f4BYNNy8O7K5wJ1/Js4AMiWNdKumUDHzw/U6ednvtkVH9BUeFZPxO/QSlLPOhwAI6Jhl/XW4DFVkgtZh/IdStFw8z/c++dnrgUwjnzvT7OVfQKA/T+AEWc1vIP65QsDtXv5mXSq8QX9oB7S4ehO7e/fpmG3AAC4tBvsId1u9snE+Wj0d1EpGs6xVmJdB8Av3/vHHMxh2ScAAGAMWDNcoJ5cGmi1k4PZ/RVGiW6Mn9FscFLfTN6pNVfOOiQAOVM2HX1v+Jim4lO5WJy+XrVotDB18c0/AOBa0QMAALaIMdJCM1Szkr+ptenO6t32Qd0QHs06FAA5cmN4WD9kHxhu/nNmqmK1czqUNez+AWCrZFwBkKQwRs5S2QDGmjHStoZVIZROLw1kcrRwDVxPd7iva0f4gp6Ov0+rrpp1SAAyUrFtvdk8runkZNahXNJMzWq2ZsU6DkC6xv8MOUcAAMCDqYpVYEKdWopz9zUylZzRD9gH9KK5U88Pbs9dfAB8crolPKSb3D5Zl6+z/hctTAWql/JXSQUA44AEAAB4Ui9bhYHRyxcGSnK2yw5crJvcPm0PX9ZTydu1lDSyDgmAZ9N2SW+yj6uWnM86lEuyRtoxFahSZPMPAL6QAAAAj8oFo90zwxsCBmmcetqgWnJO79KDOhler32DN2ugKOuQAGyxQAPdGX5Le9wBKcnhRCQpsNLumUgFVqYA4BXTLAB4VoyM9syGevlCrO4gZ6UAkoycdiSHtat0WoOdP67W1NuyDgnAFiktPaPC8YflestZh3JZhdBo10yokBf/AOBdDhIAvi9azN9iG8DkCQNp92ygk4sDtbr5nJeS3qrsix/X/Nx+le/4BzK1nVmHBGCTXOu4us/9ldbOfDvXK6HXrvnLc5QAJofv/WP2c10OEgAAMBmskXZOhzq/GuvsSpyrGwJer3N2v7pf/k1V9nyfirf8Z1KB/gDAqHD9VfUOfUrto19SEuezyd9F01WruXqQdRgAMFFIAABAymZqgYqR1cnF/DUHvMi5WK2jX1Pn5FOq3vBjCq9/j2T5ygByy8WKX/qiVg7cp6S3lnU0V2SNtL0ZqEanfwBIHas5AMhAtXixL8BA/TjraC4v7rW1/O1PKTr+dVVv+SnZhXdIymflAjCZnOITX9fagfvUb53NOpirKoRGO6cDRQHzCABkIdsEQCJaAACYWIXAaO9cpJOLsVY7+ezMfVF/5YwWn/hDFaceVOW298rM3p11SMDES84+o+7zH1N78XjWoazL8Lx/KGvE+gxAPvneP+Yg95lxBYDv2Z8MAIB8M5J2TFldaElnlvPbF+Ci7uIxdb72f6qy7VaVbvv7Mo0bsw4JmDhu6aA6z39Ma6cP5n7OGHKarVnN1AKxLgOQb+M/R3EEAAByYLpqVYyMTlyIc9sX4CJjjNpnDmjt9G+pPH+zSre+V3bqlqzDAsaeWzmizoFPqn3yGUkaic1/YKWFqVCVQv5jBYBJQAIAAHKiUjDaOx/qxIWBOv2so7k6Y4w6Zw+pfea3SQQAHrnlA+o8/ym1T+/POpQNKUVGC1OW8/4AkCMkAAAgR0Ir7Z4JdaGV6NxqrFwcFruKi4mAztnfVnn+VhVvfI/s7F1ZhwWMvOTcs+q+cL/aZ57XKMwFr3GarQWarlqNQJECAEyUHCQA6AIIAK9njDRTM6oUA51cjHN9S8Abtc88r/aZ5xU1dqhyw48p2PmDkuGqL2DdnFNy+hvqHLpfncWXXv83MgtpI8JAWmgGKhfo9AdgFPneP2bf9DkHCQAAwKWUIqPr5kKdXo610h6thXR/+YSWnvozRYfuV+X6v6twzw/KmULWYQH5lXQVH/uy1l78nPqr57KOZlMaZav5hh12+QcA5BIJAADIMWuGb9NqRadTS/lvEPhG/dVzWnrmLxU+/3GVdrxd0Y3vkSnPZx0WkB/d8+of/bzaR7+kQXct62g2xRppWzNQvcTOHwDyjgQAAIyAWsmoVBidBoFvNOh1tHrkyzJHv6LStjtVvOHHZGfoE4DJ5c4/p96Rh7R28lk5N2KZvdcZNvoLFAVZRwIAWA8SAAAwIi42CDzfSnR+RBoEvpFzidqnnlH71DMq1OdU3vX9srt/WKbQyDo0wDvXbyk+8VW1j3xR/ZVTWYdzjWj0BwCjKAcJAJoAAsB6GSPN1oxqxUCnlxN1+qM7x/VWzqi3/5MyBz6j0vwdKu39UZnZuzSKiQ3gSpKlQ+od/bw6J76hZDDIOpxrVgilhalAxZBGfwDGzfjvH7NPALD/B4ANK4ZGu6cDLbUTnV1JNMIVxHKDgdon9ql9Yp8K9XkVFt6uwnXvlkr0CsDocmtnNDj+JXVefnRkm/p9N6e5eqipihm+9R/heQcALsn3/jEH7ziyTwAAADbFGGmqYlUrWZ1airXWHf3VeG/ljHor90sHPqPy9G4Vdr5Twa53S1Et69CAqxusaXDiUXWPf03dc4eyjmZLlQpG2xuBCmEOVq8AgE0jAQAAIy600q7pQCsdpzPLseLsr5jdAkbtC8fVvvAx2f2fVGnbHYoW3q5g+9ukoJh1cMBr4o7cmSfUffkxdU7vVxKPfon/61krzdUDNcts/AFgHJAAAIAxUS8ZVYuhzizHWm6PfjXARUk80NqJfdKJfbLBn6o4d4sKC29TsOOdUlDKOjxMION6ik8/rd6Jx9U5tU/JoJd1SF5Ui0bzDasoYPMPAOMiBwkAmgAAwFaxRtretKqXnU4vJ+oPxmsOTOK+2qe+pfapb8k++5cqzt6iaNtbFC28Ta7QzDo8jDHTX1T/xDfUP/WUOmefl0vG603/60Wh0baGVaVwceM/XvMIAFye7/1j9mWaOUgAAAC2WqVgtHd2PJoEXk4yeC0ZYJ79CxUauxVt+x5FC2+TqV+XdXgYA2b1qPqnvqH+qafVWToml4zhg/Q6xkjTVavpqpHlbj8AGEskAABgTL3aJLBodG7VabmdfdbZF5c4dRdfUnfxJen5Tyso1VSauVGFubtltr1FKs5kHSJGQX9Fybln1TuzT/2z+9VfW8o6otRUi0bbmoFCm3UkAACfSAAAwJgLA6PtTaN6STqzkqg3vpXLr4o7q2q9/LRaLz8tY/9CxeYuRbO3K5y9U3bmdjlbyDpE5EHckbvwbfXPfkv9s/vVW3l57N/yv1ExMpqvW5ULvPEHgElAAgAAJkSlaLW3aLXacTqzEmsQZx1ROlzi1LlwTJ0Lx6SDn5OxVoX6DkXTNyqcvU129m6uGZwUcUfuwgENzn9b/QsH1V98UfFgAjJilxAG0kzVqlG2otofACZHxgmARDQBBIB01UpStRToQivRhVaiZHxPBlySS2J1l46pu3RMevFvZaxVVNumaPp6hc2bFMzcKlV2iV3RiHNOrnVM8fkDihcPqr90RP3V03KT9hv+DayVpitWU6+e82edBACvGf/9IxUAADCBjF55+1cyOt9yWlqb3E2RSxL1lk+qt3xS0tckSWGhqKh5nYLGXgXNvQqbN8qVt5MUyCvnZNonNVg8rHj5iOLlI+ovHdWg1806slyZqlrNVK0CzvkDwMQiAQAAEywMjLY1jGZqVmeXY610xjvrvV6DXleDMwekMwde/WtBWFBU36GgsVthfa/s1G6Z6m4prGYY6QTqrypZPSa3fEyDlaOKl4+pv3JC8aCXdWS51SgbzdZp8AcAGPcEwPhXcADAlgiNtNAMNFN1Ot9yWm7HMrzt/g7xoKf4whHpwhFJjwz/opPCUl1hbZvC2oKC+m7Z2g7Zyna50qxkgkxjHlkulumcU9I6qaR1QoPl44pXT2rQOqVBZ3VYwoIrcs6pUQ40W7OKLv42ZE0EAFc2AfNkDnoA+EQGAAA2ohBKC02j6Uqg861EKx1HIuBKjDTormjQXZHOHfqOv2WNUVCZVliZlS3PK6jMy5RnZcqzsuU5qTg9uQkCF0vdC0raZ2W65xS3zipeO6Nk7YwG7fOK1y4ocZf5/ua345U5p3p5WOpfCDnjDwAbM/5z5nhXAAAANqUYGe2YCjQ7cLqw5rS8lnD+fYMS55S0zqvfOi/pwHf9/cAamaimsDIlEzVkS1OyxZpssSFFdZliU6bYVFCcUhKUJZvzr+xkIBu3FXcX5bpLw5/eilxvWUl3RUlnSa63rEF7Ua6/qnjCrtvzzjnVy0aztfC1N/4AALxBzlcTAIAsFUKj7Q2jmarV+dVYy21HImCLxImTLlYPXIU1krGRTFiUCUsKCmWZsCQTlmWCopyxw19bKxNWZGRlotd6E1hr5IKinC69MzSKZeKuklc35U6uvybnErl4TYpjJXFHJknkku7w7w26ivttuUFHbtCVS/piT58+I6dG2Wqmxhl/AMDVkQAAAFxVFEjbm4Hm6tLiWqKltlM8uRcHpC5xkuL+8Ke7qn4r64iQNWuGzf2mK1ZhQFIOALA+JAAAAOsWWGm2ZjVTlVY6ThdaiXpx1lEBkyMKpKmKUbNiaYcAANiwHCQAfNcLUo8IAFvNGKlRluolq1bXaXEtUbufdVTA+CpH0lTFqla6uO1nfQMAfoz3/JqDBAAAYFQZI9VKRrVSoN7AaanttLSWyPFuErhmRsOO/lNlo2LEMwUAuHYkAAAAW6IQGs3XjWZrVktriZbWEvUTNi3ARkXWaaoaqFE2sjxCAIAtRAIAALClrJGmq1bTVavuQFpsJVrpUBUAXImRU7Vo1awYVQo8KwAAP0gAAAC8KYbS9qbVfMNqpTM8HtAdZB0VkB+lyKhekhrlgLf9AADvMk4AOMl5bLLg5PfzAQDrYiU1S1KzZNUbOK10hj99bhDABArtsHdGs2xVeHUl5sa97xQAjADP+9McHI2kAgAAkKpCaDRbM5qtSWs9p+V2olb3lbvugTFljVQrGtXLlPgDALJDAgAAkJlKwahSCOTcMBmw2nVa7TiSARgLRlKlaF7d+LPtBwBkjQQAACBzxkjVolG1aDRfd1rrSqtdp1aXZABGi33l93Ltld/Phl0/ACBHcpAA8L2yY+UIAKPEGqlWGp6Rds6o03+tMmCQZB0d8N1CO6xmqZWG5f2vbfpZgwDAaHHyO3dn/72QgwQAAACXZoxULhiVC0bzdak7kFY7iVpdp07fyfB6FRlwzqlcsK9WrRRZTQEARgRfWQCAkVEMpWLNarYmxYnU7ju1OsOjAnH2SXWMMWsu9qyQqkWrMCD5BAAYPSQAAAAjKbDDruq14nAj1h1IrU6izkBa67ocFNlhlNmL1SeRVCla3vIDAMYCX2cAgLFwsTpAGl7h2x04rXWd2n2p3SMhgCszev2G36gY0sAPADB+Mk4AJPLbCMF4/nwAQB4ZI5UiqRQNd3DOGfUGw74BnYFRu5eoH2ccJDIVGKlUMCqFw41/KXrjhp/1AwBMHt9NALPPLFMBAAAYe8ZIxcioGBk1JUmBBrFTu+/U7Uud/rBigCsHx5M1w2RQMRz+sRSJM/wAgImUbQIgjVsWWMwBAC4htEb1olG9+Npf68fDREC759QbDH8dc/XgSAmsVAyNCq+82S+GRlFwiX+Q9QEA4I3GvwCACgAAAC6KAikKXmssKA1vG+jFUrfv1B0MEwM9qgUyZ41UeGWjXwyH1R2FYJgAAAAAl0YCAACAKwisVLZSOTJ6feo+ToaJgN7AqRcPqwd6A6dBzMvlrWIkhcErG/1gmKAZbvoNG30AADZhzBMALMEAAH4EVioXhmXmrxn+ehA7DZJhUqAfO/XjYcKgHzvFieFYwSuscQpfqboIrV79dRTolT+/XK0k3+8AAB98f79kvwAY8wQAAADpCwOjMBjeRHCpA3/OSbGT+gOn2BkN4mGvgTi5+Ecpdq/9OheHBtfFKbDD5EhgzGu/tsNfh4FRYJyi0Cgw4po9AABSRgIAAICUGSOFRgpfrR544x+/k3NS8rqfOHFKkuF7ilf/6JwkoyRxkjGK39CkILnKSwf7hpL6wBrJOVk7vFJ3+OfDf8688s8H1sgavfqzvg09u34AALJCAgAAgJwzZnhv/WvN7C+1iV5fMmGDI2/hZwEAgKzRQgcAAAAAgAmQcQVA9k0QAAAAAAAYHqob70azVAAAAAAAADABSAAAAAAAADABSAAAAAAAADABsu0BkGh4t5Ev43+EAwAAAACwFZzzuz/NweaUCgAAAAAAACYACQAAAAAAACYACQAAAAAAACYACQAAAAAAACZAtk0A6dIHAAAAAMiF8d+fUgEAAAAAAMAEIAEAAAAA4P9v7+5iLb3KOoA/691nPjpNWwQ/ao3BglMZpCo6inSYBiISYsKFibQhMXABdKpWlGiMGElOsIkmEj+SGrDUBKNAwEhMqlYr4tgZito0tWrsMG1TMnZmaimdj3PmzJyz917Li+noQDs9e3DWXvvs9/e7m5v3ebLP3Kz/+6znBXpAAAAAAAA9IAAAAACAHmi8BDC3LQ8AAAARUX8JYKr47Mk0DgCi7u87/0scAQAAuBRqnx9n4P23KwAAAADQAwIAAAAA6IH2VwCqMv8PAADALGh/B8AEAAAAAPSAAAAAAAB6QAAAAAAAPSAAAAAAgB5ovATQkj4AAABmxXyfUU0AAAAAQA8IAAAAAKAHBAAAAADQA213AOQp1JjvKxwAAABcCj04OzZeAhjRi18ZAACADaDm+bT92dcVAAAAAOgBAQAAAAD0gAAAAAAAekAAAAAAAD3QeAngND4DAAAAAJNov6ivJhMAAAAA0AMCAAAAAOgBAQAAAAD0QOMdABHzfscCAACAjaLm+bT92bd9AFD7923/GwMAADDrap8fZ2AHvisAAAAA0AMCAAAAAOgBAQAAAAD0QPsdANVZAgAAAAAmAAAAAKAHBAAAAADQAwIAAAAA6AEBAAAAAPTADCwBtKQPAACAeZdbN2ACAAAAAPpAAAAAAAA90PYKQIm6NwBqPx8AAID5UPv8mCo+e0KNdwDkcEIHAACA+lwBAAAAgB4QAAAAAEAPCAAAAACgBwQAAAAA0AONlwBa0w8AAADTYAIAAAAAekAAAAAAAD0gAAAAAIAeEAAAAABADzReAhgRpeISQDsGAQAAmERJdc+nuf3h1AQAAAAA9IAAAAAAAHpAAAAAAAA9IAAAAACAHmi8BDBXfn77JQsAAAAwC0wAAAAAQA8IAAAAAKAHBAAAAADQA413AETUvadvBwAAAACzoPYOvPWZAAAAAIAeEAAAAABAD7S9ApCj/g0AtwAAAABYT+3zY6r47AmZAAAAAIAeaLwEsPYSBK//AQAAIMIEAAAAAPSCAAAAAAB6QAAAAAAAPSAAAAAAgB5ovAQwov53AAEAAKC12kvw12cCAAAAAHpAAAAAAAA9IAAAAACAHmi8A6BE1Xv6pVgDAAAAQHvtVwCYAAAAAIA+aBsAzEACAgAAAKXy9Hjq0rhuhfU1DQBKKitVnx+p5uMBAACYEznXTQBKiVNVC0ygbQCQ81LN59f+AwIAADAfcqn7ArkbdCeqFphA0yWA3WBwMo/rHdLPPloIAAAAwIsb5YioOEVe+wX4JBpPAIyq/gBro5pPBwAAYF7UPj+WGB2vW2F9TQOAVLqv1nz+6qj+IgcAAAA2tlGOyJV3yOVherZqgQm0nQAYlwNVnx8pVk0BAAAA8CJOD+se/ksp+fJrXlL1/DuJpgHAsTh1oJQyrFljpfIfEgAAgI1tZbXu6HiKOHLN2+6u+hW8STRdArhzz4PDgx+78cmIuLZWjZOnS7x0m3sAAAAAPF8uEctrXdRcIJ+68ni1h1+EphMAZ9X9IYY5xZmqMwYAAABsVCtrKWp/QT6l9GjdCpNpOgEQEdGl9K+5xJtr1ji2kuLbrzQFAAAAwNc6diqqfz2+jMYP1a0wmeYTAINu8LnaNZbO+CQgAAAAX+vUWsSZUf29cZu2brunepEJNA8AvnzoiX+MUk7XrJFSiq8sWwYIAADAWaVEfGVpCufEUp74rnfe+0T9QutrfgXgTYtfPnPwzu98MCLeULPOyjBiaTXiii2uAgAAAPTd8dMphjmi9vx/SrGvaoGL0HwCICJi0MXnp1Hn6aUUo2wSAAAAoM/ODCOeWZ5OrUFK1a+9T2omAoAYpT+NiFy7TC4RR05E9Q2PAAAAzKZxjjh6MkWk+i+HS4lT2zZv+Wz1QhOaiQDglbfe92iUeGAatVZHKY6eSFGEAAAAAL2SS8ThE9ObDB905e6r33nvqakUm0DzHQD/Z/SJiMHrplFpZRjx1FLE1VeUaYQ+AAAANJZLxOHjKVZHEdW/+/eckuPjUyk0oZmYAIiISNuu+uOIsjKtesurKQ6fSK4DAAAAzLnhOOK/jqWpfPLvf5Vy+Lvfu+/e6RVc38wEANt/+p6TKQ2mejfi9DDFoWNdnBlOsyoAAADTcmotxaFjKdbG0x3/TinuSmlKowYTmpkAICJiOCofKiWmehw/lwQ9s5yihPsAAAAA82CcI/57KcWREylymfJZr6QTXd76u9Mtur6ZO/EevHP3n0XET7WovdBFvHRbjiu3TmUhJAAAAJdYiRTHVyKOnU4xrv6tuReWIn5z+y37fq1N9QuboSWAZw3X8uKmzd1PRpTBtGuPcsTTyymeXYm46rISV26NWOhmamIDAACAF7A6ilg6k+Lkajx38G91lksnTo/Kbzcq/qJm8j33l+7c9ekU6abWfUREbN0UcdlCiW2bIzYvCAQAAABaK+XsC9zV0dndbitrMfU7/i/i9utu2f/B1k28kJmbAIiIWFsZvW/Ltk1viYiXtO7lzDDizDDFsdNn/50ixcIgoksRKUp03cz8JwMAAJhbOZcocfZLbsNx624uJD322L8fvb11Fxcys6fXR+/a/f6Sy++07gMAAAAmUKKkt123Z99ftW7kQmbqKwDn+8ST+34/SnqodR8AAACwrlI+O8uH/4gZDgAWFyOnQXp3yeV0614AAADgQnKOp4ajtdta97GemQ0AIiK2v+e+h3JKv9i6DwAAAHghpZTxIOV3fe/PPfBU617WM9MBQETEjj3770xRPtO6DwAAAHi+cvv2Pfff27qLScx8ABARsTbc9O5Sypda9wEAAADnlJL//lNH7/9Q6z4mNbNfAfh6Bz5247Upj/anlK5p3QsAAAA9V8p/nFrZvPu17997vHUrk9owAUBExGN3vfE14/Ha3pTSy1r3AgAAQG8dKlu23PA97/r84daNXIwNFQBERPznHT9842DT5r9OEZe37gUAAIDeeXo8TLt33Lb/YOtGLtaG2AFwvlff9sB942HcVKIste4FAACA/si5HCkLm9+yEQ//ERtwAuCcAx/dtbNL8ZcR8W2tewEAAGDelcdWV+Ot17/v/sdbd/KN2rABQETEI3e84bpu0/hvUqRrW/cCAADAfMq5PLB8au0ndv7yg8+07uX/Y8NdATjfjtv2HxzHwq4SaX/rXgAAAJg/ueRPL3ff/MaNfviP2OATAOcsLkb3jqt3/XqJ/MEupYXW/QAAALCx5VJWIsovverWf/po614ulbkIAM458Ie73pRK/pOU0ne07gUAAICNKZdyYJgXbr7+Z/f9W+teLqUNfQXg671qzxf+YWl57QdKiY9HxLh1PwAAAGwcJcfpcS6/9exK/OC8Hf4j5mwC4HyP3HHD67uF8gcpxWtb9wIAAMBsy7nsXRuln/m+n//igda91DK3AUDEc7sBrnndrWXc/UrXxctb9wMAAMBsKZEezil+Y8ct9/95615qm+sA4JzPvD0G17/59Tenkj7QpfKa1v0AAADQVinxUE7x4R17vvipiCit+5mGXgQA5ywuRnfTt+x6e9eN35tSujFFbGrdEwAAANORo6ykSH87jvyRV9/6z3/Xup9p61UAcL6HP3LDty6UfHPXde8YpPKjUfr7WwAAAMytFHmc4+FI8clnjg3+aPcH9h9r3VIrDr0R8YUP3/Dyb9qWfzyl+LGI9CODVF7RuicAAAC+MSXiydSlfXkcn1vuFu7ZuWff0dY9zQIBwPOlh39v13WbN+cfSl3eETltjxSvKCWu7bp0eYpyWesGAQAA+i6XWCkRJ1NKT0SUx6Okg5HSI8ur3b/s/IX9h1r3N4sEABdhcXGxO3Lk7sF7vv+Kl20ZDq9aS2XTuAy61n0BAADMu0Ea54VYWF0tqyd/9S8u++revXtzROTWfQEAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAPTQ/wDNiYehmpBrPgAAAABJRU5ErkJggg==')

  appIcon = new Tray(icon)

  const contextMenu = Menu.buildFromTemplate([{
    label: 'Remove',
    click: () => {
      event.sender.send('tray-removed')
    }
  }])

  appIcon.setToolTip('Electron Demo in the tray.')
  appIcon.setContextMenu(contextMenu)
})

ipcMain.on('remove-tray', () => {
  appIcon.destroy()
})

// In this file you can include the rest of your app's specific main process
// code. You can also put them in separate files and require them here.
