/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.core;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.hash.HashCode;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import com.google.common.io.BaseEncoding;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Map;
import java.util.TreeMap;
import org.bitcoinj.core.Block;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.ProtocolException;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.core.StoredBlock;
import org.bitcoinj.core.VerificationException;
import org.bitcoinj.store.BlockStore;
import org.bitcoinj.store.BlockStoreException;
import org.bitcoinj.store.FullPrunedBlockStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CheckpointManager {
    private static final Logger log = LoggerFactory.getLogger(CheckpointManager.class);
    private static final String BINARY_MAGIC = "CHECKPOINTS 1";
    private static final String TEXTUAL_MAGIC = "TXT CHECKPOINTS 1";
    private static final int MAX_SIGNATURES = 256;
    protected final TreeMap<Long, StoredBlock> checkpoints = new TreeMap();
    protected final NetworkParameters params;
    protected final Sha256Hash dataHash;
    public static final BaseEncoding BASE64 = BaseEncoding.base64().omitPadding();

    public CheckpointManager(NetworkParameters params, InputStream inputStream) throws IOException {
        this.params = (NetworkParameters)Preconditions.checkNotNull((Object)params);
        Preconditions.checkNotNull((Object)inputStream);
        inputStream = new BufferedInputStream(inputStream);
        inputStream.mark(1);
        int first = inputStream.read();
        inputStream.reset();
        if (first == BINARY_MAGIC.charAt(0)) {
            this.dataHash = this.readBinary(inputStream);
        } else if (first == TEXTUAL_MAGIC.charAt(0)) {
            this.dataHash = this.readTextual(inputStream);
        } else {
            throw new IOException("Unsupported format.");
        }
    }

    private Sha256Hash readBinary(InputStream inputStream) throws IOException {
        FilterInputStream dis = null;
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            DigestInputStream digestInputStream = new DigestInputStream(inputStream, digest);
            dis = new DataInputStream(digestInputStream);
            digestInputStream.on(false);
            byte[] header = new byte[BINARY_MAGIC.length()];
            ((DataInputStream)dis).readFully(header);
            if (!Arrays.equals(header, BINARY_MAGIC.getBytes("US-ASCII"))) {
                throw new IOException("Header bytes did not match expected version");
            }
            int numSignatures = Preconditions.checkPositionIndex((int)((DataInputStream)dis).readInt(), (int)256, (String)"Num signatures out of range");
            for (int i = 0; i < numSignatures; ++i) {
                byte[] sig = new byte[65];
                ((DataInputStream)dis).readFully(sig);
            }
            digestInputStream.on(true);
            int numCheckpoints = ((DataInputStream)dis).readInt();
            Preconditions.checkState((numCheckpoints > 0 ? 1 : 0) != 0);
            int size = 96;
            ByteBuffer buffer = ByteBuffer.allocate(96);
            for (int i = 0; i < numCheckpoints; ++i) {
                if (((DataInputStream)dis).read(buffer.array(), 0, 96) < 96) {
                    throw new IOException("Incomplete read whilst loading checkpoints.");
                }
                StoredBlock block = StoredBlock.deserializeCompact(this.params, buffer);
                buffer.position(0);
                this.checkpoints.put(block.getHeader().getTimeSeconds(), block);
            }
            Sha256Hash dataHash = new Sha256Hash(digest.digest());
            log.info("Read {} checkpoints, hash is {}", (Object)this.checkpoints.size(), (Object)dataHash);
            Sha256Hash sha256Hash = dataHash;
            return sha256Hash;
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
        catch (ProtocolException e) {
            throw new IOException(e);
        }
        finally {
            if (dis != null) {
                dis.close();
            }
            inputStream.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Sha256Hash readTextual(InputStream inputStream) throws IOException {
        Hasher hasher = Hashing.sha256().newHasher();
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader(inputStream, Charsets.US_ASCII));
            String magic = reader.readLine();
            if (!TEXTUAL_MAGIC.equals(magic)) {
                throw new IOException("unexpected magic: " + magic);
            }
            int numSigs = Integer.parseInt(reader.readLine());
            for (int i = 0; i < numSigs; ++i) {
                reader.readLine();
            }
            int numCheckpoints = Integer.parseInt(reader.readLine());
            Preconditions.checkState((numCheckpoints > 0 ? 1 : 0) != 0);
            hasher.putBytes(ByteBuffer.allocate(4).order(ByteOrder.BIG_ENDIAN).putInt(numCheckpoints).array());
            int size = 96;
            ByteBuffer buffer = ByteBuffer.allocate(96);
            for (int i = 0; i < numCheckpoints; ++i) {
                byte[] bytes = BASE64.decode((CharSequence)reader.readLine());
                hasher.putBytes(bytes);
                buffer.position(0);
                buffer.put(bytes);
                buffer.position(0);
                StoredBlock block = StoredBlock.deserializeCompact(this.params, buffer);
                this.checkpoints.put(block.getHeader().getTimeSeconds(), block);
            }
            HashCode hash = hasher.hash();
            log.info("Read {} checkpoints, hash is {}", (Object)this.checkpoints.size(), (Object)hash);
            Sha256Hash sha256Hash = new Sha256Hash(hash.asBytes());
            return sha256Hash;
        }
        finally {
            if (reader != null) {
                reader.close();
            }
        }
    }

    public StoredBlock getCheckpointBefore(long time) {
        try {
            Preconditions.checkArgument((time > this.params.getGenesisBlock().getTimeSeconds() ? 1 : 0) != 0);
            Map.Entry<Long, StoredBlock> entry = this.checkpoints.floorEntry(time);
            if (entry != null) {
                return entry.getValue();
            }
            Block genesis = this.params.getGenesisBlock().cloneAsHeader();
            return new StoredBlock(genesis, genesis.getWork(), 0);
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
    }

    public int numCheckpoints() {
        return this.checkpoints.size();
    }

    public Sha256Hash getDataHash() {
        return this.dataHash;
    }

    public static void checkpoint(NetworkParameters params, InputStream checkpoints, BlockStore store, long time) throws IOException, BlockStoreException {
        Preconditions.checkNotNull((Object)params);
        Preconditions.checkNotNull((Object)store);
        Preconditions.checkArgument((!(store instanceof FullPrunedBlockStore) ? 1 : 0) != 0, (Object)"You cannot use checkpointing with a full store.");
        BufferedInputStream stream = new BufferedInputStream(checkpoints);
        CheckpointManager manager = new CheckpointManager(params, stream);
        StoredBlock checkpoint = manager.getCheckpointBefore(time -= 604800L);
        store.put(checkpoint);
        if (!params.fixTimeWarpBug) {
            store.setChainHead(checkpoint);
        } else if (checkpoint.getHeight() % params.getInterval() == 0) {
            StoredBlock checkpoint2015 = null;
            for (StoredBlock block : manager.checkpoints.values()) {
                if (block.getHeight() != checkpoint.getHeight() - 1) continue;
                checkpoint2015 = block;
                break;
            }
            if (checkpoint2015 == null) {
                throw new RuntimeException("Could not find checkpoint 'mod 2015' block. Something wrong with checkpoint export tool");
            }
            store.put(checkpoint2015);
            store.setChainHead(checkpoint);
        } else if (checkpoint.getHeight() % params.getInterval() == 2015) {
            StoredBlock checkpoint2016 = null;
            for (StoredBlock block : manager.checkpoints.values()) {
                if (block.getHeight() != checkpoint.getHeight() + 1) continue;
                checkpoint2016 = block;
                break;
            }
            if (checkpoint2016 == null) {
                throw new RuntimeException("Could not find checkpoint 'mod 2016' block. Something wrong with checkpoint export tool");
            }
            store.put(checkpoint2016);
            store.setChainHead(checkpoint2016);
        } else {
            throw new RuntimeException("must never happen! Something wrong with checkpoint export tool");
        }
    }
}

