/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.crypto;

import com.google.common.base.Charsets;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Bytes;
import com.lambdaworks.crypto.SCrypt;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.text.Normalizer;
import java.util.Arrays;
import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import org.bitcoinj.core.AddressFormatException;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.core.VersionedChecksummedBytes;
import org.bitcoinj.crypto.DRMWorkaround;

public class BIP38PrivateKey
extends VersionedChecksummedBytes {
    public final NetworkParameters params;
    public final boolean ecMultiply;
    public final boolean compressed;
    public final boolean hasLotAndSequence;
    public final byte[] addressHash;
    public final byte[] content;

    public BIP38PrivateKey(NetworkParameters params, String encoded) throws AddressFormatException {
        super(encoded);
        this.params = params;
        if (this.version != 1) {
            throw new AddressFormatException("Mismatched version number: " + this.version);
        }
        if (this.bytes.length != 38) {
            throw new AddressFormatException("Wrong number of bytes, excluding version byte: " + this.bytes.length);
        }
        this.hasLotAndSequence = (this.bytes[1] & 4) != 0;
        boolean bl = this.compressed = (this.bytes[1] & 0x20) != 0;
        if ((this.bytes[1] & 1) != 0) {
            throw new AddressFormatException("Bit 0x01 reserved for future use.");
        }
        if ((this.bytes[1] & 2) != 0) {
            throw new AddressFormatException("Bit 0x02 reserved for future use.");
        }
        if ((this.bytes[1] & 8) != 0) {
            throw new AddressFormatException("Bit 0x08 reserved for future use.");
        }
        if ((this.bytes[1] & 0x10) != 0) {
            throw new AddressFormatException("Bit 0x10 reserved for future use.");
        }
        int byte0 = this.bytes[0] & 0xFF;
        if (byte0 == 66) {
            if ((this.bytes[1] & 0xC0) != 192) {
                throw new AddressFormatException("Bits 0x40 and 0x80 must be set for non-EC-multiplied keys.");
            }
            this.ecMultiply = false;
            if (this.hasLotAndSequence) {
                throw new AddressFormatException("Non-EC-multiplied keys cannot have lot/sequence.");
            }
        } else if (byte0 == 67) {
            if ((this.bytes[1] & 0xC0) != 0) {
                throw new AddressFormatException("Bits 0x40 and 0x80 must be cleared for EC-multiplied keys.");
            }
            this.ecMultiply = true;
        } else {
            throw new AddressFormatException("Second byte must by 0x42 or 0x43.");
        }
        this.addressHash = Arrays.copyOfRange(this.bytes, 2, 6);
        this.content = Arrays.copyOfRange(this.bytes, 6, 38);
    }

    public ECKey decrypt(String passphrase) throws BadPassphraseException {
        String normalizedPassphrase = Normalizer.normalize(passphrase, Normalizer.Form.NFC);
        ECKey key = this.ecMultiply ? this.decryptEC(normalizedPassphrase) : this.decryptNoEC(normalizedPassphrase);
        Sha256Hash hash = Sha256Hash.createDouble(key.toAddress(this.params).toString().getBytes(Charsets.US_ASCII));
        byte[] actualAddressHash = Arrays.copyOfRange(hash.getBytes(), 0, 4);
        if (!Arrays.equals(actualAddressHash, this.addressHash)) {
            throw new BadPassphraseException();
        }
        return key;
    }

    private ECKey decryptNoEC(String normalizedPassphrase) {
        try {
            byte[] derived = SCrypt.scrypt((byte[])normalizedPassphrase.getBytes(Charsets.UTF_8), (byte[])this.addressHash, (int)16384, (int)8, (int)8, (int)64);
            byte[] key = Arrays.copyOfRange(derived, 32, 64);
            SecretKeySpec keyspec = new SecretKeySpec(key, "AES");
            DRMWorkaround.maybeDisableExportControls();
            Cipher cipher = Cipher.getInstance("AES/ECB/NoPadding");
            cipher.init(2, keyspec);
            byte[] decrypted = cipher.doFinal(this.content, 0, 32);
            for (int i = 0; i < 32; ++i) {
                int n = i;
                decrypted[n] = (byte)(decrypted[n] ^ derived[i]);
            }
            return ECKey.fromPrivate(decrypted, this.compressed);
        }
        catch (GeneralSecurityException x) {
            throw new RuntimeException(x);
        }
    }

    private ECKey decryptEC(String normalizedPassphrase) {
        try {
            byte[] ownerEntropy = Arrays.copyOfRange(this.content, 0, 8);
            byte[] ownerSalt = this.hasLotAndSequence ? Arrays.copyOfRange(ownerEntropy, 0, 4) : ownerEntropy;
            byte[] passFactorBytes = SCrypt.scrypt((byte[])normalizedPassphrase.getBytes(Charsets.UTF_8), (byte[])ownerSalt, (int)16384, (int)8, (int)8, (int)32);
            if (this.hasLotAndSequence) {
                byte[] hashBytes = Bytes.concat((byte[][])new byte[][]{passFactorBytes, ownerEntropy});
                Preconditions.checkState((hashBytes.length == 40 ? 1 : 0) != 0);
                passFactorBytes = Sha256Hash.createDouble(hashBytes).getBytes();
            }
            BigInteger passFactor = new BigInteger(1, passFactorBytes);
            ECKey k = ECKey.fromPrivate(passFactor, true);
            byte[] salt = Bytes.concat((byte[][])new byte[][]{this.addressHash, ownerEntropy});
            Preconditions.checkState((salt.length == 12 ? 1 : 0) != 0);
            byte[] derived = SCrypt.scrypt((byte[])k.getPubKey(), (byte[])salt, (int)1024, (int)1, (int)1, (int)64);
            byte[] aeskey = Arrays.copyOfRange(derived, 32, 64);
            SecretKeySpec keyspec = new SecretKeySpec(aeskey, "AES");
            Cipher cipher = Cipher.getInstance("AES/ECB/NoPadding");
            cipher.init(2, keyspec);
            byte[] encrypted2 = Arrays.copyOfRange(this.content, 16, 32);
            byte[] decrypted2 = cipher.doFinal(encrypted2);
            Preconditions.checkState((decrypted2.length == 16 ? 1 : 0) != 0);
            for (int i = 0; i < 16; ++i) {
                int n = i;
                decrypted2[n] = (byte)(decrypted2[n] ^ derived[i + 16]);
            }
            byte[] encrypted1 = Bytes.concat((byte[][])new byte[][]{Arrays.copyOfRange(this.content, 8, 16), Arrays.copyOfRange(decrypted2, 0, 8)});
            byte[] decrypted1 = cipher.doFinal(encrypted1);
            Preconditions.checkState((decrypted1.length == 16 ? 1 : 0) != 0);
            for (int i = 0; i < 16; ++i) {
                int n = i;
                decrypted1[n] = (byte)(decrypted1[n] ^ derived[i]);
            }
            byte[] seed = Bytes.concat((byte[][])new byte[][]{decrypted1, Arrays.copyOfRange(decrypted2, 8, 16)});
            Preconditions.checkState((seed.length == 24 ? 1 : 0) != 0);
            BigInteger seedFactor = new BigInteger(1, Sha256Hash.createDouble(seed).getBytes());
            Preconditions.checkState((passFactor.signum() >= 0 ? 1 : 0) != 0);
            Preconditions.checkState((seedFactor.signum() >= 0 ? 1 : 0) != 0);
            BigInteger priv = passFactor.multiply(seedFactor).mod(ECKey.CURVE_PARAMS.getN());
            return ECKey.fromPrivate(priv, this.compressed);
        }
        catch (GeneralSecurityException x) {
            throw new RuntimeException(x);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        BIP38PrivateKey other = (BIP38PrivateKey)o;
        return super.equals(other) && Objects.equal((Object)this.params, (Object)other.params);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode((Object[])new Object[]{super.hashCode(), this.params});
    }

    public static final class BadPassphraseException
    extends Exception {
    }
}

