/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.crypto;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.util.Arrays;
import javax.annotation.Nullable;
import org.bitcoinj.core.Address;
import org.bitcoinj.core.AddressFormatException;
import org.bitcoinj.core.Base58;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.core.Utils;
import org.bitcoinj.crypto.ChildNumber;
import org.bitcoinj.crypto.EncryptedData;
import org.bitcoinj.crypto.HDKeyDerivation;
import org.bitcoinj.crypto.HDUtils;
import org.bitcoinj.crypto.KeyCrypter;
import org.bitcoinj.crypto.KeyCrypterException;
import org.spongycastle.crypto.params.KeyParameter;
import org.spongycastle.math.ec.ECPoint;

public class DeterministicKey
extends ECKey {
    private static final long serialVersionUID = 1L;
    private final DeterministicKey parent;
    private final ImmutableList<ChildNumber> childNumberPath;
    private final byte[] chainCode;

    public DeterministicKey(ImmutableList<ChildNumber> childNumberPath, byte[] chainCode, ECPoint publicAsPoint, @Nullable BigInteger priv, @Nullable DeterministicKey parent) {
        super(priv, DeterministicKey.compressPoint((ECPoint)Preconditions.checkNotNull((Object)publicAsPoint)));
        Preconditions.checkArgument((chainCode.length == 32 ? 1 : 0) != 0);
        this.parent = parent;
        this.childNumberPath = (ImmutableList)Preconditions.checkNotNull(childNumberPath);
        this.chainCode = Arrays.copyOf(chainCode, chainCode.length);
    }

    public DeterministicKey(ImmutableList<ChildNumber> childNumberPath, byte[] chainCode, BigInteger priv, @Nullable DeterministicKey parent) {
        super(priv, DeterministicKey.compressPoint(ECKey.CURVE.getG().multiply(priv)));
        Preconditions.checkArgument((chainCode.length == 32 ? 1 : 0) != 0);
        this.parent = parent;
        this.childNumberPath = (ImmutableList)Preconditions.checkNotNull(childNumberPath);
        this.chainCode = Arrays.copyOf(chainCode, chainCode.length);
    }

    public DeterministicKey(ImmutableList<ChildNumber> childNumberPath, byte[] chainCode, KeyCrypter crypter, ECPoint pub, EncryptedData priv, @Nullable DeterministicKey parent) {
        this(childNumberPath, chainCode, pub, null, parent);
        this.encryptedPrivateKey = (EncryptedData)Preconditions.checkNotNull((Object)priv);
        this.keyCrypter = (KeyCrypter)Preconditions.checkNotNull((Object)crypter);
    }

    public DeterministicKey(DeterministicKey keyToClone, DeterministicKey newParent) {
        super(keyToClone.priv, keyToClone.pub);
        this.parent = newParent;
        this.childNumberPath = keyToClone.childNumberPath;
        this.chainCode = keyToClone.chainCode;
        this.encryptedPrivateKey = keyToClone.encryptedPrivateKey;
    }

    public ImmutableList<ChildNumber> getPath() {
        return this.childNumberPath;
    }

    public String getPathAsString() {
        return HDUtils.formatPath(this.getPath());
    }

    private int getDepth() {
        return this.childNumberPath.size();
    }

    public ChildNumber getChildNumber() {
        return this.getDepth() == 0 ? ChildNumber.ZERO : (ChildNumber)this.childNumberPath.get(this.childNumberPath.size() - 1);
    }

    public byte[] getChainCode() {
        return this.chainCode;
    }

    public byte[] getIdentifier() {
        return Utils.sha256hash160(this.getPubKey());
    }

    public byte[] getFingerprint() {
        return Arrays.copyOfRange(this.getIdentifier(), 0, 4);
    }

    @Nullable
    public DeterministicKey getParent() {
        return this.parent;
    }

    public byte[] getPrivKeyBytes33() {
        byte[] bytes33 = new byte[33];
        byte[] priv = this.getPrivKeyBytes();
        System.arraycopy(priv, 0, bytes33, 33 - priv.length, priv.length);
        return bytes33;
    }

    public DeterministicKey getPubOnly() {
        if (this.isPubKeyOnly()) {
            return this;
        }
        return new DeterministicKey(this.getPath(), this.getChainCode(), this.getPubKeyPoint(), null, this.parent);
    }

    static byte[] addChecksum(byte[] input) {
        int inputLength = input.length;
        byte[] checksummed = new byte[inputLength + 4];
        System.arraycopy(input, 0, checksummed, 0, inputLength);
        byte[] checksum = Utils.doubleDigest(input);
        System.arraycopy(checksum, 0, checksummed, inputLength, 4);
        return checksummed;
    }

    @Override
    public DeterministicKey encrypt(KeyCrypter keyCrypter, KeyParameter aesKey) throws KeyCrypterException {
        throw new UnsupportedOperationException("Must supply a new parent for encryption");
    }

    public DeterministicKey encrypt(KeyCrypter keyCrypter, KeyParameter aesKey, @Nullable DeterministicKey newParent) throws KeyCrypterException {
        byte[] privKeyBytes;
        Preconditions.checkNotNull((Object)keyCrypter);
        if (newParent != null) {
            Preconditions.checkArgument((boolean)newParent.isEncrypted());
        }
        Preconditions.checkState(((privKeyBytes = this.getPrivKeyBytes()) != null ? 1 : 0) != 0, (Object)"Private key is not available");
        EncryptedData encryptedPrivateKey = keyCrypter.encrypt(privKeyBytes, aesKey);
        DeterministicKey key = new DeterministicKey(this.childNumberPath, this.chainCode, keyCrypter, this.pub, encryptedPrivateKey, newParent);
        if (newParent == null) {
            key.setCreationTimeSeconds(this.getCreationTimeSeconds());
        }
        return key;
    }

    @Override
    public boolean isEncrypted() {
        return this.priv == null && (super.isEncrypted() || this.parent != null && this.parent.isEncrypted());
    }

    @Override
    @Nullable
    public KeyCrypter getKeyCrypter() {
        if (this.keyCrypter != null) {
            return this.keyCrypter;
        }
        if (this.parent != null) {
            return this.parent.getKeyCrypter();
        }
        return null;
    }

    @Override
    public ECKey.ECDSASignature sign(Sha256Hash input, @Nullable KeyParameter aesKey) throws KeyCrypterException {
        if (this.isEncrypted()) {
            return super.sign(input, aesKey);
        }
        BigInteger privateKey = this.findOrDerivePrivateKey();
        if (privateKey == null) {
            throw new ECKey.MissingPrivateKeyException();
        }
        return super.doSign(input, privateKey);
    }

    @Override
    public DeterministicKey decrypt(KeyCrypter keyCrypter, KeyParameter aesKey) throws KeyCrypterException {
        Preconditions.checkNotNull((Object)keyCrypter);
        if (this.keyCrypter != null && !this.keyCrypter.equals(keyCrypter)) {
            throw new KeyCrypterException("The keyCrypter being used to decrypt the key is different to the one that was used to encrypt it");
        }
        BigInteger privKey = this.findOrDeriveEncryptedPrivateKey(keyCrypter, aesKey);
        DeterministicKey key = new DeterministicKey(this.childNumberPath, this.chainCode, privKey, this.parent);
        if (!Arrays.equals(key.getPubKey(), this.getPubKey())) {
            throw new KeyCrypterException("Provided AES key is wrong");
        }
        if (this.parent == null) {
            key.setCreationTimeSeconds(this.getCreationTimeSeconds());
        }
        return key;
    }

    @Override
    public DeterministicKey decrypt(KeyParameter aesKey) throws KeyCrypterException {
        return (DeterministicKey)super.decrypt(aesKey);
    }

    private BigInteger findOrDeriveEncryptedPrivateKey(KeyCrypter keyCrypter, KeyParameter aesKey) {
        if (this.encryptedPrivateKey != null) {
            return new BigInteger(1, keyCrypter.decrypt(this.encryptedPrivateKey, aesKey));
        }
        DeterministicKey cursor = this.parent;
        while (cursor != null && cursor.encryptedPrivateKey == null) {
            cursor = cursor.parent;
        }
        if (cursor == null) {
            throw new KeyCrypterException("Neither this key nor its parents have an encrypted private key");
        }
        byte[] parentalPrivateKeyBytes = keyCrypter.decrypt(cursor.encryptedPrivateKey, aesKey);
        return this.derivePrivateKeyDownwards(cursor, parentalPrivateKeyBytes);
    }

    @Nullable
    private BigInteger findOrDerivePrivateKey() {
        DeterministicKey cursor = this;
        while (cursor != null && cursor.priv == null) {
            cursor = cursor.parent;
        }
        if (cursor == null) {
            return null;
        }
        return this.derivePrivateKeyDownwards(cursor, cursor.priv.toByteArray());
    }

    private BigInteger derivePrivateKeyDownwards(DeterministicKey cursor, byte[] parentalPrivateKeyBytes) {
        DeterministicKey downCursor = new DeterministicKey(cursor.childNumberPath, cursor.chainCode, cursor.pub, new BigInteger(1, parentalPrivateKeyBytes), cursor.parent);
        ImmutableList path = this.childNumberPath.subList(cursor.getDepth(), this.childNumberPath.size());
        for (ChildNumber num : path) {
            downCursor = HDKeyDerivation.deriveChildKey(downCursor, num);
        }
        Preconditions.checkState((boolean)downCursor.pub.equals(this.pub));
        return (BigInteger)Preconditions.checkNotNull((Object)downCursor.priv);
    }

    public DeterministicKey derive(int child) {
        return HDKeyDerivation.deriveChildKey(this, new ChildNumber(child, true));
    }

    @Override
    public BigInteger getPrivKey() {
        BigInteger key = this.findOrDerivePrivateKey();
        Preconditions.checkState((key != null ? 1 : 0) != 0, (Object)"Private key bytes not available");
        return key;
    }

    public byte[] serializePublic(NetworkParameters params) {
        return this.serialize(params, true);
    }

    public byte[] serializePrivate(NetworkParameters params) {
        return this.serialize(params, false);
    }

    private byte[] serialize(NetworkParameters params, boolean pub) {
        ByteBuffer ser = ByteBuffer.allocate(78);
        ser.putInt(pub ? params.getBip32HeaderPub() : params.getBip32HeaderPriv());
        ser.put((byte)this.getDepth());
        if (this.parent == null) {
            ser.putInt(0);
        } else {
            ser.put(this.parent.getFingerprint());
        }
        ser.putInt(this.getChildNumber().i());
        ser.put(this.getChainCode());
        ser.put(pub ? this.getPubKey() : this.getPrivKeyBytes33());
        Preconditions.checkState((ser.position() == 78 ? 1 : 0) != 0);
        return ser.array();
    }

    public String serializePubB58(NetworkParameters params) {
        return DeterministicKey.toBase58(this.serialize(params, true));
    }

    public String serializePrivB58(NetworkParameters params) {
        return DeterministicKey.toBase58(this.serialize(params, false));
    }

    static String toBase58(byte[] ser) {
        return Base58.encode(DeterministicKey.addChecksum(ser));
    }

    public static DeterministicKey deserializeB58(String base58, NetworkParameters params) {
        return DeterministicKey.deserializeB58(null, base58, params);
    }

    public static DeterministicKey deserializeB58(@Nullable DeterministicKey parent, String base58, NetworkParameters params) {
        try {
            return DeterministicKey.deserialize(params, Base58.decodeChecked(base58), parent);
        }
        catch (AddressFormatException e) {
            throw new IllegalArgumentException(e);
        }
    }

    public static DeterministicKey deserialize(NetworkParameters params, byte[] serializedKey) {
        return DeterministicKey.deserialize(params, serializedKey, null);
    }

    public static DeterministicKey deserialize(NetworkParameters params, byte[] serializedKey, @Nullable DeterministicKey parent) {
        Object path;
        ByteBuffer buffer = ByteBuffer.wrap(serializedKey);
        int header = buffer.getInt();
        if (header != params.getBip32HeaderPriv() && header != params.getBip32HeaderPub()) {
            throw new IllegalArgumentException("Unknown header bytes: " + DeterministicKey.toBase58(serializedKey).substring(0, 4));
        }
        boolean pub = header == params.getBip32HeaderPub();
        byte depth = buffer.get();
        byte[] parentFingerprint = new byte[4];
        buffer.get(parentFingerprint);
        int i = buffer.getInt();
        ChildNumber childNumber = new ChildNumber(i);
        if (parent != null) {
            if (Arrays.equals(parentFingerprint, HDUtils.longTo4ByteArray(0L))) {
                throw new IllegalArgumentException("Parent was provided but this key doesn't have one");
            }
            if (!Arrays.equals(parent.getFingerprint(), parentFingerprint)) {
                throw new IllegalArgumentException("Parent fingerprints don't match");
            }
            path = HDUtils.append(parent.getPath(), childNumber);
            if (path.size() != depth) {
                throw new IllegalArgumentException("Depth does not match");
            }
        } else {
            path = depth >= 1 ? ImmutableList.of((Object)childNumber) : ImmutableList.of();
        }
        byte[] chainCode = new byte[32];
        buffer.get(chainCode);
        byte[] data = new byte[33];
        buffer.get(data);
        Preconditions.checkArgument((!buffer.hasRemaining() ? 1 : 0) != 0, (Object)"Found unexpected data in key");
        if (pub) {
            ECPoint point = ECKey.CURVE.getCurve().decodePoint(data);
            return new DeterministicKey((ImmutableList<ChildNumber>)path, chainCode, point, null, parent);
        }
        return new DeterministicKey((ImmutableList<ChildNumber>)path, chainCode, new BigInteger(1, data), parent);
    }

    @Override
    public long getCreationTimeSeconds() {
        if (this.parent != null) {
            return this.parent.getCreationTimeSeconds();
        }
        return super.getCreationTimeSeconds();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        DeterministicKey other = (DeterministicKey)o;
        return super.equals(other) && Arrays.equals(this.chainCode, other.chainCode) && Objects.equal(this.childNumberPath, other.childNumberPath);
    }

    @Override
    public int hashCode() {
        int result = super.hashCode();
        result = 31 * result + this.childNumberPath.hashCode();
        result = 31 * result + Arrays.hashCode(this.chainCode);
        return result;
    }

    @Override
    public String toString() {
        Objects.ToStringHelper helper = Objects.toStringHelper((Object)this).omitNullValues();
        helper.add("pub", (Object)Utils.HEX.encode(this.pub.getEncoded()));
        helper.add("chainCode", (Object)Utils.HEX.encode(this.chainCode));
        helper.add("path", (Object)this.getPathAsString());
        if (this.creationTimeSeconds > 0L) {
            helper.add("creationTimeSeconds", this.creationTimeSeconds);
        }
        return helper.toString();
    }

    @Override
    public void formatKeyWithAddress(boolean includePrivateKeys, StringBuilder builder, NetworkParameters params) {
        Address address = this.toAddress(params);
        builder.append("  addr:");
        builder.append(address.toString());
        builder.append("  hash160:");
        builder.append(Utils.HEX.encode(this.getPubKeyHash()));
        builder.append("  (");
        builder.append(this.getPathAsString());
        builder.append(")");
        builder.append("\n");
        if (includePrivateKeys) {
            builder.append("  ");
            builder.append(this.toStringWithPrivate(params));
            builder.append("\n");
        }
    }
}

