/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.crypto;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.Utils;
import org.bitcoinj.crypto.ChildNumber;
import org.bitcoinj.crypto.DeterministicKey;
import org.bitcoinj.crypto.HDDerivationException;
import org.bitcoinj.crypto.HDUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongycastle.crypto.macs.HMac;
import org.spongycastle.math.ec.ECPoint;

public final class HDKeyDerivation {
    private static final Logger log = LoggerFactory.getLogger(HDKeyDerivation.class);
    private static final BigInteger RAND_INT = new BigInteger(256, new SecureRandom());
    public static final int MAX_CHILD_DERIVATION_ATTEMPTS = 100;
    public static final HMac MASTER_HMAC_SHA512 = HDUtils.createHmacSha512Digest("Bitcoin seed".getBytes());

    private HDKeyDerivation() {
    }

    public static DeterministicKey createMasterPrivateKey(byte[] seed) throws HDDerivationException {
        Preconditions.checkArgument((seed.length > 8 ? 1 : 0) != 0, (Object)"Seed is too short and could be brute forced");
        byte[] i = HDUtils.hmacSha512(MASTER_HMAC_SHA512, seed);
        Preconditions.checkState((i.length == 64 ? 1 : 0) != 0, (Object)i.length);
        byte[] il = Arrays.copyOfRange(i, 0, 32);
        byte[] ir = Arrays.copyOfRange(i, 32, 64);
        Arrays.fill(i, (byte)0);
        DeterministicKey masterPrivKey = HDKeyDerivation.createMasterPrivKeyFromBytes(il, ir);
        Arrays.fill(il, (byte)0);
        Arrays.fill(ir, (byte)0);
        masterPrivKey.setCreationTimeSeconds(Utils.currentTimeSeconds());
        return masterPrivKey;
    }

    public static DeterministicKey createMasterPrivateKey(ImmutableList<ChildNumber> rootNodeList, byte[] seed) throws HDDerivationException {
        Preconditions.checkArgument((seed.length > 8 ? 1 : 0) != 0, (Object)"Seed is too short and could be brute forced");
        byte[] i = HDUtils.hmacSha512(MASTER_HMAC_SHA512, seed);
        Preconditions.checkState((i.length == 64 ? 1 : 0) != 0, (Object)i.length);
        byte[] il = Arrays.copyOfRange(i, 0, 32);
        byte[] ir = Arrays.copyOfRange(i, 32, 64);
        Arrays.fill(i, (byte)0);
        DeterministicKey masterPrivKey = HDKeyDerivation.createMasterPrivKeyFromBytes(rootNodeList, il, ir);
        Arrays.fill(il, (byte)0);
        Arrays.fill(ir, (byte)0);
        masterPrivKey.setCreationTimeSeconds(Utils.currentTimeSeconds());
        return masterPrivKey;
    }

    public static DeterministicKey createRootNodeWithPrivateKey(ImmutableList<ChildNumber> rootNodeList, byte[] seed) throws HDDerivationException {
        Preconditions.checkArgument((seed.length > 8 ? 1 : 0) != 0, (Object)"Seed is too short and could be brute forced");
        byte[] i = HDUtils.hmacSha512(MASTER_HMAC_SHA512, seed);
        Preconditions.checkState((i.length == 64 ? 1 : 0) != 0, (Object)i.length);
        byte[] il = Arrays.copyOfRange(i, 0, 32);
        byte[] ir = Arrays.copyOfRange(i, 32, 64);
        Arrays.fill(i, (byte)0);
        DeterministicKey masterPrivKey = HDKeyDerivation.createMasterPrivKeyFromBytes((ImmutableList<ChildNumber>)ImmutableList.copyOf(new ArrayList()), il, ir);
        Arrays.fill(il, (byte)0);
        Arrays.fill(ir, (byte)0);
        masterPrivKey.setCreationTimeSeconds(Utils.currentTimeSeconds());
        log.debug("masterPrivKey = " + masterPrivKey);
        DeterministicKey childKey = masterPrivKey;
        for (ChildNumber childNumber : rootNodeList) {
            childKey = HDKeyDerivation.deriveChildKey(childKey, childNumber);
            log.debug("childKey = " + childKey);
        }
        return childKey;
    }

    public static DeterministicKey createMasterPrivKeyFromBytes(byte[] privKeyBytes, byte[] chainCode) throws HDDerivationException {
        BigInteger priv = new BigInteger(1, privKeyBytes);
        HDKeyDerivation.assertNonZero(priv, "Generated master key is invalid.");
        HDKeyDerivation.assertLessThanN(priv, "Generated master key is invalid.");
        return new DeterministicKey((ImmutableList<ChildNumber>)ImmutableList.of(), chainCode, priv, null);
    }

    public static DeterministicKey createMasterPrivKeyFromBytes(ImmutableList<ChildNumber> rootNodeList, byte[] privKeyBytes, byte[] chainCode) throws HDDerivationException {
        BigInteger priv = new BigInteger(1, privKeyBytes);
        HDKeyDerivation.assertNonZero(priv, "Generated master key is invalid.");
        HDKeyDerivation.assertLessThanN(priv, "Generated master key is invalid.");
        return new DeterministicKey(rootNodeList, chainCode, priv, null);
    }

    public static DeterministicKey createMasterPubKeyFromBytes(byte[] pubKeyBytes, byte[] chainCode) {
        return new DeterministicKey((ImmutableList<ChildNumber>)ImmutableList.of(), chainCode, ECKey.CURVE.getCurve().decodePoint(pubKeyBytes), null, null);
    }

    public static DeterministicKey deriveChildKey(DeterministicKey parent, int childNumber) {
        return HDKeyDerivation.deriveChildKey(parent, new ChildNumber(childNumber));
    }

    public static DeterministicKey deriveThisOrNextChildKey(DeterministicKey parent, int childNumber) {
        ChildNumber child = new ChildNumber(childNumber);
        boolean isHardened = child.isHardened();
        for (int nAttempts = 0; nAttempts < 100; ++nAttempts) {
            try {
                child = new ChildNumber(child.num() + nAttempts, isHardened);
                return HDKeyDerivation.deriveChildKey(parent, child);
            }
            catch (HDDerivationException hDDerivationException) {
                continue;
            }
        }
        throw new HDDerivationException("Maximum number of child derivation attempts reached, this is probably an indication of a bug.");
    }

    public static DeterministicKey deriveChildKey(DeterministicKey parent, ChildNumber childNumber) throws HDDerivationException {
        if (parent.isPubKeyOnly()) {
            RawKeyBytes rawKey = HDKeyDerivation.deriveChildKeyBytesFromPublic(parent, childNumber, PublicDeriveMode.NORMAL);
            return new DeterministicKey(HDUtils.append(parent.getPath(), childNumber), rawKey.chainCode, ECKey.CURVE.getCurve().decodePoint(rawKey.keyBytes), null, parent);
        }
        RawKeyBytes rawKey = HDKeyDerivation.deriveChildKeyBytesFromPrivate(parent, childNumber);
        return new DeterministicKey(HDUtils.append(parent.getPath(), childNumber), rawKey.chainCode, new BigInteger(1, rawKey.keyBytes), parent);
    }

    public static RawKeyBytes deriveChildKeyBytesFromPrivate(DeterministicKey parent, ChildNumber childNumber) throws HDDerivationException {
        Preconditions.checkArgument((boolean)parent.hasPrivKey(), (Object)"Parent key must have private key bytes for this method.");
        byte[] parentPublicKey = ECKey.compressPoint(parent.getPubKeyPoint()).getEncoded();
        assert (parentPublicKey.length == 33) : parentPublicKey.length;
        ByteBuffer data = ByteBuffer.allocate(37);
        if (childNumber.isHardened()) {
            data.put(parent.getPrivKeyBytes33());
        } else {
            data.put(parentPublicKey);
        }
        data.putInt(childNumber.i());
        byte[] i = HDUtils.hmacSha512(parent.getChainCode(), data.array());
        assert (i.length == 64) : i.length;
        byte[] il = Arrays.copyOfRange(i, 0, 32);
        byte[] chainCode = Arrays.copyOfRange(i, 32, 64);
        BigInteger ilInt = new BigInteger(1, il);
        HDKeyDerivation.assertLessThanN(ilInt, "Illegal derived key: I_L >= n");
        BigInteger priv = parent.getPrivKey();
        BigInteger ki = priv.add(ilInt).mod(ECKey.CURVE.getN());
        HDKeyDerivation.assertNonZero(ki, "Illegal derived key: derived private key equals 0.");
        return new RawKeyBytes(ki.toByteArray(), chainCode);
    }

    public static RawKeyBytes deriveChildKeyBytesFromPublic(DeterministicKey parent, ChildNumber childNumber, PublicDeriveMode mode) throws HDDerivationException {
        ECPoint Ki;
        Preconditions.checkArgument((!childNumber.isHardened() ? 1 : 0) != 0, (Object)"Can't use private derivation with public keys only.");
        byte[] parentPublicKey = ECKey.compressPoint(parent.getPubKeyPoint()).getEncoded();
        assert (parentPublicKey.length == 33) : parentPublicKey.length;
        ByteBuffer data = ByteBuffer.allocate(37);
        data.put(parentPublicKey);
        data.putInt(childNumber.i());
        byte[] i = HDUtils.hmacSha512(parent.getChainCode(), data.array());
        assert (i.length == 64) : i.length;
        byte[] il = Arrays.copyOfRange(i, 0, 32);
        byte[] chainCode = Arrays.copyOfRange(i, 32, 64);
        BigInteger ilInt = new BigInteger(1, il);
        HDKeyDerivation.assertLessThanN(ilInt, "Illegal derived key: I_L >= n");
        ECPoint G = ECKey.CURVE.getG();
        BigInteger N = ECKey.CURVE.getN();
        switch (mode) {
            case NORMAL: {
                Ki = G.multiply(ilInt).add(parent.getPubKeyPoint());
                break;
            }
            case WITH_INVERSION: {
                Ki = G.multiply(ilInt.add(RAND_INT));
                BigInteger additiveInverse = RAND_INT.negate().mod(N);
                Ki = Ki.add(G.multiply(additiveInverse));
                Ki = Ki.add(parent.getPubKeyPoint());
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        HDKeyDerivation.assertNonInfinity(Ki, "Illegal derived key: derived public key equals infinity.");
        return new RawKeyBytes(Ki.getEncoded(true), chainCode);
    }

    private static void assertNonZero(BigInteger integer, String errorMessage) {
        if (integer.equals(BigInteger.ZERO)) {
            throw new HDDerivationException(errorMessage);
        }
    }

    private static void assertNonInfinity(ECPoint point, String errorMessage) {
        if (point.equals(ECKey.CURVE.getCurve().getInfinity())) {
            throw new HDDerivationException(errorMessage);
        }
    }

    private static void assertLessThanN(BigInteger integer, String errorMessage) {
        if (integer.compareTo(ECKey.CURVE.getN()) > 0) {
            throw new HDDerivationException(errorMessage);
        }
    }

    public static class RawKeyBytes {
        public final byte[] keyBytes;
        public final byte[] chainCode;

        public RawKeyBytes(byte[] keyBytes, byte[] chainCode) {
            this.keyBytes = keyBytes;
            this.chainCode = chainCode;
        }
    }

    public static enum PublicDeriveMode {
        NORMAL,
        WITH_INVERSION;

    }
}

