/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.protocols.channels;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import java.util.Arrays;
import java.util.List;
import javax.annotation.Nullable;
import org.bitcoinj.core.Coin;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.InsufficientMoneyException;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.core.Transaction;
import org.bitcoinj.core.TransactionBroadcaster;
import org.bitcoinj.core.TransactionConfidence;
import org.bitcoinj.core.TransactionInput;
import org.bitcoinj.core.TransactionOutput;
import org.bitcoinj.core.VerificationException;
import org.bitcoinj.core.Wallet;
import org.bitcoinj.crypto.TransactionSignature;
import org.bitcoinj.protocols.channels.PaymentChannelServer;
import org.bitcoinj.protocols.channels.StoredPaymentChannelServerStates;
import org.bitcoinj.protocols.channels.StoredServerChannel;
import org.bitcoinj.protocols.channels.ValueOutOfRangeException;
import org.bitcoinj.script.Script;
import org.bitcoinj.script.ScriptBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PaymentChannelServerState {
    private static final Logger log = LoggerFactory.getLogger(PaymentChannelServerState.class);
    private State state;
    private ECKey clientKey;
    private ECKey serverKey;
    final Wallet wallet;
    private final TransactionBroadcaster broadcaster;
    private Transaction multisigContract = null;
    private Script multisigScript;
    private byte[] bestValueSignature;
    private Coin totalValue;
    private Coin bestValueToMe = Coin.ZERO;
    private Coin feePaidForPayment;
    private TransactionOutput clientOutput;
    private long refundTransactionUnlockTimeSecs;
    private long minExpireTime;
    private StoredServerChannel storedServerChannel = null;
    final SettableFuture<Transaction> closedFuture = SettableFuture.create();

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    PaymentChannelServerState(StoredServerChannel storedServerChannel, Wallet wallet, TransactionBroadcaster broadcaster) throws VerificationException {
        StoredServerChannel storedServerChannel2 = storedServerChannel;
        synchronized (storedServerChannel2) {
            this.wallet = (Wallet)Preconditions.checkNotNull((Object)wallet);
            this.broadcaster = (TransactionBroadcaster)Preconditions.checkNotNull((Object)broadcaster);
            this.multisigContract = (Transaction)Preconditions.checkNotNull((Object)storedServerChannel.contract);
            this.multisigScript = this.multisigContract.getOutput(0).getScriptPubKey();
            this.clientKey = ECKey.fromPublicOnly(this.multisigScript.getChunks().get((int)1).data);
            this.clientOutput = (TransactionOutput)Preconditions.checkNotNull((Object)storedServerChannel.clientOutput);
            this.refundTransactionUnlockTimeSecs = storedServerChannel.refundTransactionUnlockTimeSecs;
            this.serverKey = (ECKey)Preconditions.checkNotNull((Object)storedServerChannel.myKey);
            this.totalValue = this.multisigContract.getOutput(0).getValue();
            this.bestValueToMe = (Coin)Preconditions.checkNotNull((Object)storedServerChannel.bestValueToMe);
            this.bestValueSignature = storedServerChannel.bestValueSignature;
            Preconditions.checkArgument((this.bestValueToMe.equals(Coin.ZERO) || this.bestValueSignature != null ? 1 : 0) != 0);
            this.storedServerChannel = storedServerChannel;
            storedServerChannel.state = this;
            this.state = State.READY;
        }
    }

    public PaymentChannelServerState(TransactionBroadcaster broadcaster, Wallet wallet, ECKey serverKey, long minExpireTime) {
        this.state = State.WAITING_FOR_REFUND_TRANSACTION;
        this.serverKey = (ECKey)Preconditions.checkNotNull((Object)serverKey);
        this.wallet = (Wallet)Preconditions.checkNotNull((Object)wallet);
        this.broadcaster = (TransactionBroadcaster)Preconditions.checkNotNull((Object)broadcaster);
        this.minExpireTime = minExpireTime;
    }

    public synchronized State getState() {
        return this.state;
    }

    public synchronized byte[] provideRefundTransaction(Transaction refundTx, byte[] clientMultiSigPubKey) throws VerificationException {
        Preconditions.checkNotNull((Object)refundTx);
        Preconditions.checkNotNull((Object)clientMultiSigPubKey);
        Preconditions.checkState((this.state == State.WAITING_FOR_REFUND_TRANSACTION ? 1 : 0) != 0);
        log.info("Provided with refund transaction: {}", (Object)refundTx);
        refundTx.verify();
        if (refundTx.getInputs().size() != 1) {
            throw new VerificationException("Refund transaction does not have exactly one input");
        }
        if (refundTx.getInput(0).getSequenceNumber() != 0L) {
            throw new VerificationException("Refund transaction's input's sequence number is non-0");
        }
        if (refundTx.getLockTime() < this.minExpireTime) {
            throw new VerificationException("Refund transaction has a lock time too soon");
        }
        if (refundTx.getOutputs().size() != 1) {
            throw new VerificationException("Refund transaction does not have exactly one output");
        }
        this.refundTransactionUnlockTimeSecs = refundTx.getLockTime();
        this.clientKey = ECKey.fromPublicOnly(clientMultiSigPubKey);
        Script multisigPubKey = ScriptBuilder.createMultiSigOutputScript(2, (List<ECKey>)ImmutableList.of((Object)this.clientKey, (Object)this.serverKey));
        TransactionSignature sig = refundTx.calculateSignature(0, this.serverKey, multisigPubKey, Transaction.SigHash.NONE, true);
        log.info("Signed refund transaction.");
        this.clientOutput = refundTx.getOutput(0);
        this.state = State.WAITING_FOR_MULTISIG_CONTRACT;
        return sig.encodeToBitcoin();
    }

    public synchronized ListenableFuture<PaymentChannelServerState> provideMultiSigContract(final Transaction multisigContract) throws VerificationException {
        Preconditions.checkNotNull((Object)multisigContract);
        Preconditions.checkState((this.state == State.WAITING_FOR_MULTISIG_CONTRACT ? 1 : 0) != 0);
        try {
            multisigContract.verify();
            this.multisigContract = multisigContract;
            this.multisigScript = multisigContract.getOutput(0).getScriptPubKey();
            Script expectedScript = ScriptBuilder.createMultiSigOutputScript(2, Lists.newArrayList((Object[])new ECKey[]{this.clientKey, this.serverKey}));
            if (!Arrays.equals(this.multisigScript.getProgram(), expectedScript.getProgram())) {
                throw new VerificationException("Multisig contract's first output was not a standard 2-of-2 multisig to client and server in that order.");
            }
            this.totalValue = multisigContract.getOutput(0).getValue();
            if (this.totalValue.signum() <= 0) {
                throw new VerificationException("Not accepting an attempt to open a contract with zero value.");
            }
        }
        catch (VerificationException e) {
            log.error("Provided multisig contract did not verify: {}", (Object)multisigContract.toString());
            throw e;
        }
        log.info("Broadcasting multisig contract: {}", (Object)multisigContract);
        this.state = State.WAITING_FOR_MULTISIG_ACCEPTANCE;
        final SettableFuture future = SettableFuture.create();
        Futures.addCallback(this.broadcaster.broadcastTransaction(multisigContract), (FutureCallback)new FutureCallback<Transaction>(){

            public void onSuccess(Transaction transaction) {
                log.info("Successfully broadcast multisig contract {}. Channel now open.", (Object)transaction.getHashAsString());
                try {
                    PaymentChannelServerState.this.wallet.receivePending(multisigContract, null, true);
                }
                catch (VerificationException e) {
                    throw new RuntimeException(e);
                }
                PaymentChannelServerState.this.state = State.READY;
                future.set((Object)PaymentChannelServerState.this);
            }

            public void onFailure(Throwable throwable) {
                log.error(throwable.toString());
                throwable.printStackTrace();
                PaymentChannelServerState.this.state = State.ERROR;
                future.setException(throwable);
            }
        });
        return future;
    }

    private synchronized Wallet.SendRequest makeUnsignedChannelContract(Coin valueToMe) {
        Transaction tx = new Transaction(this.wallet.getParams());
        if (!this.totalValue.subtract(valueToMe).equals(Coin.ZERO)) {
            this.clientOutput.setValue(this.totalValue.subtract(valueToMe));
            tx.addOutput(this.clientOutput);
        }
        tx.addInput(this.multisigContract.getOutput(0));
        return Wallet.SendRequest.forTx(tx);
    }

    public synchronized boolean incrementPayment(Coin refundSize, byte[] signatureBytes) throws VerificationException, ValueOutOfRangeException, InsufficientMoneyException {
        Preconditions.checkState((this.state == State.READY ? 1 : 0) != 0);
        Preconditions.checkNotNull((Object)refundSize);
        Preconditions.checkNotNull((Object)signatureBytes);
        TransactionSignature signature = TransactionSignature.decodeFromBitcoin(signatureBytes, true);
        boolean fullyUsedUp = refundSize.equals(Coin.ZERO);
        if (refundSize.compareTo(this.clientOutput.getMinNonDustValue()) < 0 && !fullyUsedUp) {
            throw new ValueOutOfRangeException("Attempt to refund negative value or value too small to be accepted by the network");
        }
        Coin newValueToMe = this.totalValue.subtract(refundSize);
        if (newValueToMe.signum() < 0) {
            throw new ValueOutOfRangeException("Attempt to refund more than the contract allows.");
        }
        if (newValueToMe.compareTo(this.bestValueToMe) < 0) {
            throw new ValueOutOfRangeException("Attempt to roll back payment on the channel.");
        }
        Transaction walletContract = this.wallet.getTransaction(this.multisigContract.getHash());
        Preconditions.checkNotNull((Object)walletContract, (String)"Wallet did not contain multisig contract {} after state was marked READY", (Object[])new Object[]{this.multisigContract.getHash()});
        if (walletContract.getConfidence().getConfidenceType() == TransactionConfidence.ConfidenceType.DEAD) {
            this.close();
            throw new VerificationException("Multisig contract was double-spent");
        }
        Transaction.SigHash mode = fullyUsedUp ? Transaction.SigHash.NONE : Transaction.SigHash.SINGLE;
        if (signature.sigHashMode() != mode || !signature.anyoneCanPay()) {
            throw new VerificationException("New payment signature was not signed with the right SIGHASH flags.");
        }
        Wallet.SendRequest req = this.makeUnsignedChannelContract(newValueToMe);
        Sha256Hash sighash = req.tx.hashForSignature(0, this.multisigScript, mode, true);
        if (!this.clientKey.verify(sighash, signature)) {
            throw new VerificationException("Signature does not verify on tx\n" + req.tx);
        }
        this.bestValueToMe = newValueToMe;
        this.bestValueSignature = signatureBytes;
        this.updateChannelInWallet();
        return !fullyUsedUp;
    }

    private void signMultisigInput(Transaction tx, Transaction.SigHash hashType, boolean anyoneCanPay) {
        TransactionSignature signature = tx.calculateSignature(0, this.serverKey, this.multisigScript, hashType, anyoneCanPay);
        byte[] mySig = signature.encodeToBitcoin();
        Script scriptSig = ScriptBuilder.createMultiSigInputScriptBytes((List<byte[]>)ImmutableList.of((Object)this.bestValueSignature, (Object)mySig));
        tx.getInput(0).setScriptSig(scriptSig);
    }

    public synchronized ListenableFuture<Transaction> close() throws InsufficientMoneyException {
        if (this.storedServerChannel != null) {
            StoredServerChannel temp = this.storedServerChannel;
            this.storedServerChannel = null;
            StoredPaymentChannelServerStates channels = (StoredPaymentChannelServerStates)this.wallet.getExtensions().get(StoredPaymentChannelServerStates.EXTENSION_ID);
            channels.closeChannel(temp);
            if (this.state.compareTo(State.CLOSING) >= 0) {
                return this.closedFuture;
            }
        }
        if (this.state.ordinal() < State.READY.ordinal()) {
            log.error("Attempt to settle channel in state " + (Object)((Object)this.state));
            this.state = State.CLOSED;
            this.closedFuture.set(null);
            return this.closedFuture;
        }
        if (this.state != State.READY) {
            log.warn("Failed attempt to settle a channel in state " + (Object)((Object)this.state));
            return this.closedFuture;
        }
        Transaction tx = null;
        try {
            Wallet.SendRequest req = this.makeUnsignedChannelContract(this.bestValueToMe);
            tx = req.tx;
            this.signMultisigInput(tx, Transaction.SigHash.NONE, true);
            req.shuffleOutputs = false;
            req.missingSigsMode = Wallet.MissingSigsMode.USE_DUMMY_SIG;
            this.wallet.completeTx(req);
            this.feePaidForPayment = req.tx.getFee();
            log.info("Calculated fee is {}", (Object)this.feePaidForPayment);
            if (this.feePaidForPayment.compareTo(this.bestValueToMe) >= 0) {
                String msg = String.format("Had to pay more in fees (%s) than the channel was worth (%s)", this.feePaidForPayment, this.bestValueToMe);
                throw new InsufficientMoneyException(this.feePaidForPayment.subtract(this.bestValueToMe), msg);
            }
            this.signMultisigInput(tx, Transaction.SigHash.ALL, false);
            tx.verify();
            for (TransactionInput input : tx.getInputs()) {
                input.verify();
            }
        }
        catch (InsufficientMoneyException e) {
            throw e;
        }
        catch (Exception e) {
            log.error("Could not verify self-built tx\nMULTISIG {}\nCLOSE {}", (Object)this.multisigContract, tx != null ? tx : "");
            throw new RuntimeException(e);
        }
        this.state = State.CLOSING;
        log.info("Closing channel, broadcasting tx {}", (Object)tx);
        ListenableFuture<Transaction> future = this.broadcaster.broadcastTransaction(tx);
        Futures.addCallback(future, (FutureCallback)new FutureCallback<Transaction>(){

            public void onSuccess(Transaction transaction) {
                log.info("TX {} propagated, channel successfully closed.", (Object)transaction.getHash());
                PaymentChannelServerState.this.state = State.CLOSED;
                PaymentChannelServerState.this.closedFuture.set((Object)transaction);
            }

            public void onFailure(Throwable throwable) {
                log.error("Failed to settle channel, could not broadcast: {}", (Object)throwable.toString());
                throwable.printStackTrace();
                PaymentChannelServerState.this.state = State.ERROR;
                PaymentChannelServerState.this.closedFuture.setException(throwable);
            }
        });
        return this.closedFuture;
    }

    public synchronized Coin getBestValueToMe() {
        return this.bestValueToMe;
    }

    public synchronized Coin getFeePaid() {
        Preconditions.checkState((this.state == State.CLOSED || this.state == State.CLOSING ? 1 : 0) != 0);
        return this.feePaidForPayment;
    }

    public synchronized Transaction getMultisigContract() {
        Preconditions.checkState((this.multisigContract != null ? 1 : 0) != 0);
        return this.multisigContract;
    }

    public synchronized long getRefundTransactionUnlockTime() {
        Preconditions.checkState((this.state.compareTo(State.WAITING_FOR_MULTISIG_CONTRACT) > 0 && this.state != State.ERROR ? 1 : 0) != 0);
        return this.refundTransactionUnlockTimeSecs;
    }

    private synchronized void updateChannelInWallet() {
        if (this.storedServerChannel != null) {
            this.storedServerChannel.updateValueToMe(this.bestValueToMe, this.bestValueSignature);
            StoredPaymentChannelServerStates channels = (StoredPaymentChannelServerStates)this.wallet.getExtensions().get(StoredPaymentChannelServerStates.EXTENSION_ID);
            this.wallet.addOrUpdateExtension(channels);
        }
    }

    public synchronized void storeChannelInWallet(@Nullable PaymentChannelServer connectedHandler) {
        Preconditions.checkState((this.state == State.READY ? 1 : 0) != 0);
        if (this.storedServerChannel != null) {
            return;
        }
        log.info("Storing state with contract hash {}.", (Object)this.multisigContract.getHash());
        StoredPaymentChannelServerStates channels = (StoredPaymentChannelServerStates)this.wallet.addOrGetExistingExtension(new StoredPaymentChannelServerStates(this.wallet, this.broadcaster));
        this.storedServerChannel = new StoredServerChannel(this, this.multisigContract, this.clientOutput, this.refundTransactionUnlockTimeSecs, this.serverKey, this.bestValueToMe, this.bestValueSignature);
        if (connectedHandler != null) {
            Preconditions.checkState((this.storedServerChannel.setConnectedHandler(connectedHandler, false) == connectedHandler ? 1 : 0) != 0);
        }
        channels.putChannel(this.storedServerChannel);
        this.wallet.addOrUpdateExtension(channels);
    }

    public static enum State {
        WAITING_FOR_REFUND_TRANSACTION,
        WAITING_FOR_MULTISIG_CONTRACT,
        WAITING_FOR_MULTISIG_ACCEPTANCE,
        READY,
        CLOSING,
        CLOSED,
        ERROR;

    }
}

