/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.wallet;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.protobuf.ByteString;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nullable;
import org.bitcoinj.core.BloomFilter;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.Utils;
import org.bitcoinj.crypto.ChildNumber;
import org.bitcoinj.crypto.DeterministicHierarchy;
import org.bitcoinj.crypto.DeterministicKey;
import org.bitcoinj.crypto.EncryptedData;
import org.bitcoinj.crypto.HDKeyDerivation;
import org.bitcoinj.crypto.HDUtils;
import org.bitcoinj.crypto.KeyCrypter;
import org.bitcoinj.crypto.KeyCrypterException;
import org.bitcoinj.crypto.KeyCrypterScrypt;
import org.bitcoinj.crypto.MnemonicCode;
import org.bitcoinj.script.Script;
import org.bitcoinj.store.UnreadableWalletException;
import org.bitcoinj.utils.Threading;
import org.bitcoinj.wallet.BasicKeyChain;
import org.bitcoinj.wallet.DeterministicSeed;
import org.bitcoinj.wallet.EncryptableKeyChain;
import org.bitcoinj.wallet.KeyChain;
import org.bitcoinj.wallet.KeyChainEventListener;
import org.bitcoinj.wallet.MarriedKeyChain;
import org.bitcoinj.wallet.Protos;
import org.bitcoinj.wallet.RedeemData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongycastle.crypto.params.KeyParameter;
import org.spongycastle.math.ec.ECPoint;

public class DeterministicKeyChain
implements EncryptableKeyChain {
    private static final Logger log = LoggerFactory.getLogger(DeterministicKeyChain.class);
    public static final String DEFAULT_PASSPHRASE_FOR_MNEMONIC = "";
    protected final ReentrantLock lock = Threading.lock("DeterministicKeyChain");
    private DeterministicHierarchy hierarchy;
    @Nullable
    private DeterministicKey rootKey;
    @Nullable
    private DeterministicSeed seed;
    private long creationTimeSeconds = MnemonicCode.BIP39_STANDARDISATION_TIME_SECS;
    public static final ImmutableList<ChildNumber> ACCOUNT_ZERO_PATH = ImmutableList.of((Object)ChildNumber.ZERO_HARDENED);
    public static final ImmutableList<ChildNumber> EXTERNAL_PATH = ImmutableList.of((Object)ChildNumber.ZERO_HARDENED, (Object)ChildNumber.ZERO);
    public static final ImmutableList<ChildNumber> INTERNAL_PATH = ImmutableList.of((Object)ChildNumber.ZERO_HARDENED, (Object)ChildNumber.ONE);
    private static final int LAZY_CALCULATE_LOOKAHEAD = -1;
    protected int lookaheadSize = 100;
    protected int lookaheadThreshold = this.calcDefaultLookaheadThreshold();
    private DeterministicKey externalKey;
    private DeterministicKey internalKey;
    private int issuedExternalKeys;
    private int issuedInternalKeys;
    private int keyLookaheadEpoch;
    private final BasicKeyChain basicKeyChain;
    private boolean isFollowing;
    protected int sigsRequiredToSpend = 1;
    private final ImmutableList<ChildNumber> rootNodeList;

    private int calcDefaultLookaheadThreshold() {
        return this.lookaheadSize / 3;
    }

    public static Builder<?> builder() {
        return new Builder();
    }

    public DeterministicKeyChain(SecureRandom random) {
        this(random, 128, DEFAULT_PASSPHRASE_FOR_MNEMONIC, Utils.currentTimeSeconds());
    }

    public DeterministicKeyChain(SecureRandom random, ImmutableList<ChildNumber> rootNodeList) {
        this(random, 128, DEFAULT_PASSPHRASE_FOR_MNEMONIC, Utils.currentTimeSeconds(), rootNodeList);
    }

    public DeterministicKeyChain(SecureRandom random, int bits) {
        this(random, bits, DEFAULT_PASSPHRASE_FOR_MNEMONIC, Utils.currentTimeSeconds());
    }

    public DeterministicKeyChain(SecureRandom random, int bits, String passphrase, long seedCreationTimeSecs) {
        this(new DeterministicSeed(random, bits, passphrase, seedCreationTimeSecs));
    }

    public DeterministicKeyChain(SecureRandom random, int bits, String passphrase, long seedCreationTimeSecs, ImmutableList<ChildNumber> rootPathNode) {
        this(new DeterministicSeed(random, bits, passphrase, seedCreationTimeSecs), rootPathNode);
    }

    public DeterministicKeyChain(byte[] entropy, String passphrase, long seedCreationTimeSecs) {
        this(new DeterministicSeed(entropy, passphrase, seedCreationTimeSecs));
    }

    protected DeterministicKeyChain(DeterministicSeed seed) {
        this(seed, (KeyCrypter)null);
    }

    protected DeterministicKeyChain(DeterministicSeed seed, ImmutableList<ChildNumber> rootNodeList) {
        this(seed, rootNodeList, (KeyCrypter)null);
    }

    protected DeterministicKeyChain(DeterministicSeed seed, ImmutableList<ChildNumber> rootNodeList, @Nullable KeyCrypter crypter) {
        this.seed = seed;
        this.rootNodeList = rootNodeList;
        this.basicKeyChain = new BasicKeyChain(crypter);
        if (!seed.isEncrypted()) {
            log.debug("seed is NOT encrypted, rootNodeList:" + rootNodeList);
            this.rootKey = ACCOUNT_ZERO_PATH.equals(rootNodeList) ? HDKeyDerivation.createMasterPrivateKey((byte[])Preconditions.checkNotNull((Object)seed.getSeedBytes())) : HDKeyDerivation.createRootNodeWithPrivateKey(rootNodeList, (byte[])Preconditions.checkNotNull((Object)seed.getSeedBytes()));
            this.rootKey.setCreationTimeSeconds(seed.getCreationTimeSeconds());
            log.debug("rootKey is:" + this.rootKey);
            this.initializeHierarchyUnencrypted(this.rootKey, rootNodeList);
        } else {
            log.debug("seed IS encrypted");
        }
    }

    public DeterministicKeyChain(DeterministicKey watchingKey, long creationTimeSeconds, ImmutableList<ChildNumber> rootNodeList) {
        Preconditions.checkArgument((boolean)watchingKey.isPubKeyOnly(), (Object)"Private subtrees not currently supported");
        this.basicKeyChain = new BasicKeyChain();
        this.creationTimeSeconds = creationTimeSeconds;
        this.seed = null;
        this.rootNodeList = rootNodeList;
        this.initializeHierarchyUnencrypted(watchingKey, rootNodeList);
    }

    public DeterministicKeyChain(DeterministicKey watchingKey, long creationTimeSeconds) {
        this(watchingKey, creationTimeSeconds, ACCOUNT_ZERO_PATH);
    }

    public DeterministicKeyChain(DeterministicKey watchingKey) {
        this(watchingKey, Utils.currentTimeSeconds());
    }

    protected DeterministicKeyChain(DeterministicKey watchKey, boolean isFollowing) {
        this(watchKey, Utils.currentTimeSeconds());
        this.isFollowing = isFollowing;
    }

    private DeterministicKeyChain(DeterministicKey watchKey, boolean isFollowing, ImmutableList<ChildNumber> rootNode) {
        this(watchKey, Utils.currentTimeSeconds(), rootNode);
        this.isFollowing = isFollowing;
    }

    public static DeterministicKeyChain watchAndFollow(DeterministicKey watchKey) {
        return new DeterministicKeyChain(watchKey, true);
    }

    public static DeterministicKeyChain watch(DeterministicKey accountKey) {
        return DeterministicKeyChain.watch(accountKey, 1369267200L);
    }

    public static DeterministicKeyChain watch(DeterministicKey accountKey, long seedCreationTimeSecs) {
        return new DeterministicKeyChain(accountKey, seedCreationTimeSecs);
    }

    public static DeterministicKeyChain watch(DeterministicKey accountKey, long seedCreationTimeSecs, ImmutableList<ChildNumber> rootNodeList) {
        return new DeterministicKeyChain(accountKey, seedCreationTimeSecs, rootNodeList);
    }

    DeterministicKeyChain(DeterministicSeed seed, @Nullable KeyCrypter crypter) {
        this.seed = seed;
        this.rootNodeList = ACCOUNT_ZERO_PATH;
        this.basicKeyChain = new BasicKeyChain(crypter);
        if (!seed.isEncrypted()) {
            this.rootKey = HDKeyDerivation.createMasterPrivateKey((ImmutableList<ChildNumber>)ImmutableList.builder().build(), (byte[])Preconditions.checkNotNull((Object)seed.getSeedBytes()));
            this.rootKey.setCreationTimeSeconds(seed.getCreationTimeSeconds());
            this.initializeHierarchyUnencrypted(this.rootKey, ACCOUNT_ZERO_PATH);
        }
    }

    private DeterministicKeyChain(KeyCrypter crypter, KeyParameter aesKey, DeterministicKeyChain chain) {
        this(crypter, aesKey, chain, ACCOUNT_ZERO_PATH);
    }

    private DeterministicKeyChain(KeyCrypter crypter, KeyParameter aesKey, DeterministicKeyChain chain, ImmutableList<ChildNumber> rootNodeList) {
        Preconditions.checkNotNull((Object)chain.rootKey);
        Preconditions.checkNotNull((Object)chain.seed);
        Preconditions.checkArgument((!chain.rootKey.isEncrypted() ? 1 : 0) != 0, (Object)"Chain already encrypted");
        this.issuedExternalKeys = chain.issuedExternalKeys;
        this.issuedInternalKeys = chain.issuedInternalKeys;
        this.lookaheadSize = chain.lookaheadSize;
        this.lookaheadThreshold = chain.lookaheadThreshold;
        this.seed = chain.seed.encrypt(crypter, aesKey);
        this.rootNodeList = rootNodeList;
        this.basicKeyChain = new BasicKeyChain(crypter);
        log.debug("unencrypted rootKey: " + chain.rootKey);
        this.rootKey = chain.rootKey.encrypt(crypter, aesKey, null);
        this.hierarchy = new DeterministicHierarchy(this.rootKey);
        this.basicKeyChain.importKey(this.rootKey);
        DeterministicKey account = this.encryptNonLeaf(aesKey, chain, this.rootKey, this.rootKey.getPath());
        log.debug("account: " + account);
        DeterministicKey intermediateKey = null;
        ImmutableList relativeExternalPath = EXTERNAL_PATH;
        ImmutableList relativeInternalPath = INTERNAL_PATH;
        if (DeterministicKeyChain.isTrezorPath(this.rootKey.getPath()) && chain.rootKey.hasPrivKey() && this.rootKey.getPath().size() == 2) {
            intermediateKey = HDKeyDerivation.deriveChildKey(chain.rootKey, new ChildNumber(Integer.MIN_VALUE));
            intermediateKey = intermediateKey.encrypt(crypter, aesKey, account);
            this.hierarchy.putKey(intermediateKey);
            this.basicKeyChain.importKey(intermediateKey);
            System.out.println("DeterministicKeyChain - Added intermediate node: " + intermediateKey);
        }
        if (DeterministicKeyChain.isTrezorPath(this.rootKey.getPath())) {
            relativeExternalPath = ImmutableList.of((Object)ChildNumber.ZERO);
            relativeInternalPath = ImmutableList.of((Object)ChildNumber.ONE);
        }
        DeterministicKey parent = intermediateKey == null ? account : intermediateKey;
        ArrayList externalPathAbsolute = Lists.newArrayList();
        externalPathAbsolute.addAll(parent.getPath());
        externalPathAbsolute.addAll(relativeExternalPath);
        System.out.println("DeterministicKeyChain - externalPathAbsolute: " + externalPathAbsolute);
        this.externalKey = this.encryptNonLeaf(aesKey, chain, parent, (ImmutableList<ChildNumber>)ImmutableList.copyOf((Collection)externalPathAbsolute));
        log.debug("externalKey:" + this.externalKey);
        ArrayList internalPathAbsolute = Lists.newArrayList();
        internalPathAbsolute.addAll(parent.getPath());
        internalPathAbsolute.addAll(relativeInternalPath);
        System.out.println("DeterministicKeyChain - " + internalPathAbsolute);
        this.internalKey = this.encryptNonLeaf(aesKey, chain, parent, (ImmutableList<ChildNumber>)ImmutableList.copyOf((Collection)internalPathAbsolute));
        log.debug("internalKey:" + this.internalKey);
        for (ECKey eckey : chain.basicKeyChain.getKeys()) {
            DeterministicKey key = (DeterministicKey)eckey;
            if (!DeterministicKeyChain.isTrezorPath(key.getPath()) && key.getPath().size() != 3 || DeterministicKeyChain.isTrezorPath(key.getPath()) && key.getPath().size() != 5) continue;
            DeterministicKey parentKey = this.hierarchy.get((List<ChildNumber>)((DeterministicKey)Preconditions.checkNotNull((Object)key.getParent())).getPath(), false, false);
            key = new DeterministicKey(key.getPubOnly(), parentKey);
            this.hierarchy.putKey(key);
            this.basicKeyChain.importKey(key);
        }
    }

    private DeterministicKey encryptNonLeaf(KeyParameter aesKey, DeterministicKeyChain chain, DeterministicKey parent, ImmutableList<ChildNumber> path) {
        DeterministicKey key = chain.hierarchy.get((List<ChildNumber>)path, false, true);
        log.debug("unencrypted key: " + key);
        key = key.encrypt((KeyCrypter)Preconditions.checkNotNull((Object)this.basicKeyChain.getKeyCrypter()), aesKey, parent);
        log.debug("encrypted key: " + key);
        this.hierarchy.putKey(key);
        this.basicKeyChain.importKey(key);
        return key;
    }

    private void initializeHierarchyUnencrypted(DeterministicKey baseKey, ImmutableList<ChildNumber> rootNodeList) {
        ImmutableList emptyRootNodeList = ImmutableList.of((Object)new ChildNumber(0, true));
        if (this.rootKey != null && (baseKey.getPath().isEmpty() || emptyRootNodeList.equals(baseKey.getPath()) || DeterministicKeyChain.isTrezorPath(rootNodeList))) {
            this.addToBasicChain(this.rootKey);
            this.hierarchy = new DeterministicHierarchy(this.rootKey);
            this.addToBasicChain(this.hierarchy.get((List<ChildNumber>)rootNodeList, false, true));
            log.debug("initialize 1 rootKey: " + this.rootKey);
        } else {
            this.rootKey = null;
            this.addToBasicChain(baseKey);
            this.hierarchy = new DeterministicHierarchy(baseKey);
            log.debug("initialize 2 rootKey: " + this.rootKey);
        }
        if (DeterministicKeyChain.isTrezorPath(rootNodeList)) {
            this.externalKey = this.hierarchy.deriveChild((List<ChildNumber>)rootNodeList, false, true, ChildNumber.ZERO);
            this.internalKey = this.hierarchy.deriveChild((List<ChildNumber>)rootNodeList, false, true, ChildNumber.ONE);
        } else {
            this.externalKey = this.hierarchy.deriveChild((List<ChildNumber>)rootNodeList, false, false, ChildNumber.ZERO);
            this.internalKey = this.hierarchy.deriveChild((List<ChildNumber>)rootNodeList, false, false, ChildNumber.ONE);
        }
        log.debug("Adding externalKey: {}", (Object)this.externalKey);
        log.debug("Adding internalKey: {}", (Object)this.internalKey);
        this.addToBasicChain(this.externalKey);
        this.addToBasicChain(this.internalKey);
    }

    @Override
    public DeterministicKey getKey(KeyChain.KeyPurpose purpose) {
        return this.getKeys(purpose, 1).get(0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DeterministicKey> getKeys(KeyChain.KeyPurpose purpose, int numberOfKeys) {
        Preconditions.checkArgument((numberOfKeys > 0 ? 1 : 0) != 0);
        this.lock.lock();
        try {
            DeterministicKey parentKey;
            int index;
            switch (purpose) {
                case RECEIVE_FUNDS: 
                case REFUND: {
                    this.issuedExternalKeys += numberOfKeys;
                    index = this.issuedExternalKeys;
                    parentKey = this.externalKey;
                    break;
                }
                case AUTHENTICATION: 
                case CHANGE: {
                    this.issuedInternalKeys += numberOfKeys;
                    index = this.issuedInternalKeys;
                    parentKey = this.internalKey;
                    break;
                }
                default: {
                    throw new UnsupportedOperationException();
                }
            }
            List<DeterministicKey> lookahead = this.maybeLookAhead(parentKey, index, 0, 0);
            this.basicKeyChain.importKeys(lookahead);
            ArrayList<DeterministicKey> keys = new ArrayList<DeterministicKey>(numberOfKeys);
            for (int i = 0; i < numberOfKeys; ++i) {
                ImmutableList<ChildNumber> path = HDUtils.append(parentKey.getPath(), new ChildNumber(index - numberOfKeys + i, false));
                DeterministicKey k = this.hierarchy.get((List<ChildNumber>)path, false, false);
                this.checkForBitFlip(k);
                keys.add(k);
            }
            ArrayList<DeterministicKey> arrayList = keys;
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void checkForBitFlip(DeterministicKey k) {
        DeterministicKey parent = (DeterministicKey)Preconditions.checkNotNull((Object)k.getParent());
        byte[] rederived = HDKeyDerivation.deriveChildKeyBytesFromPublic((DeterministicKey)parent, (ChildNumber)k.getChildNumber(), (HDKeyDerivation.PublicDeriveMode)HDKeyDerivation.PublicDeriveMode.WITH_INVERSION).keyBytes;
        byte[] actual = k.getPubKey();
        if (!Arrays.equals(rederived, actual)) {
            throw new IllegalStateException(String.format("Bit-flip check failed: %s vs %s", Arrays.toString(rederived), Arrays.toString(actual)));
        }
    }

    private void addToBasicChain(DeterministicKey key) {
        this.basicKeyChain.importKeys((List<? extends ECKey>)ImmutableList.of((Object)key));
    }

    public DeterministicKey markKeyAsUsed(DeterministicKey k) {
        int numChildren = k.getChildNumber().i() + 1;
        if (k.getParent() == this.internalKey) {
            if (this.issuedInternalKeys < numChildren) {
                this.issuedInternalKeys = numChildren;
                this.maybeLookAhead();
            }
        } else if (k.getParent() == this.externalKey && this.issuedExternalKeys < numChildren) {
            this.issuedExternalKeys = numChildren;
            this.maybeLookAhead();
        }
        return k;
    }

    public DeterministicKey findKeyFromPubHash(byte[] pubkeyHash) {
        this.lock.lock();
        try {
            DeterministicKey deterministicKey = (DeterministicKey)this.basicKeyChain.findKeyFromPubHash(pubkeyHash);
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    public DeterministicKey findKeyFromPubKey(byte[] pubkey) {
        this.lock.lock();
        try {
            DeterministicKey deterministicKey = (DeterministicKey)this.basicKeyChain.findKeyFromPubKey(pubkey);
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public DeterministicKey markPubHashAsUsed(byte[] pubkeyHash) {
        this.lock.lock();
        try {
            DeterministicKey k = (DeterministicKey)this.basicKeyChain.findKeyFromPubHash(pubkeyHash);
            if (k != null) {
                this.markKeyAsUsed(k);
            }
            DeterministicKey deterministicKey = k;
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public DeterministicKey markPubKeyAsUsed(byte[] pubkey) {
        this.lock.lock();
        try {
            DeterministicKey k = (DeterministicKey)this.basicKeyChain.findKeyFromPubKey(pubkey);
            if (k != null) {
                this.markKeyAsUsed(k);
            }
            DeterministicKey deterministicKey = k;
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public boolean hasKey(ECKey key) {
        this.lock.lock();
        try {
            boolean bl = this.basicKeyChain.hasKey(key);
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    protected DeterministicKey getKeyByPath(ChildNumber ... path) {
        return this.getKeyByPath((List<ChildNumber>)ImmutableList.copyOf((Object[])path));
    }

    protected DeterministicKey getKeyByPath(List<ChildNumber> path) {
        return this.getKeyByPath(path, false);
    }

    public DeterministicKey getKeyByPath(List<ChildNumber> path, boolean create) {
        return this.hierarchy.get(path, false, create);
    }

    public DeterministicKey getWatchingKey() {
        return this.getKeyByPath((List<ChildNumber>)this.rootNodeList, true).getPubOnly();
    }

    @Override
    public int numKeys() {
        this.lock.lock();
        try {
            this.maybeLookAhead();
            int n = this.basicKeyChain.numKeys();
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int numLeafKeysIssued() {
        this.lock.lock();
        try {
            int n = this.issuedExternalKeys + this.issuedInternalKeys;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public long getEarliestKeyCreationTime() {
        return this.seed != null ? this.seed.getCreationTimeSeconds() : this.creationTimeSeconds;
    }

    @Override
    public void addEventListener(KeyChainEventListener listener) {
        this.basicKeyChain.addEventListener(listener);
    }

    @Override
    public void addEventListener(KeyChainEventListener listener, Executor executor) {
        this.basicKeyChain.addEventListener(listener, executor);
    }

    @Override
    public boolean removeEventListener(KeyChainEventListener listener) {
        return this.basicKeyChain.removeEventListener(listener);
    }

    @Nullable
    public List<String> getMnemonicCode() {
        if (this.seed == null) {
            return null;
        }
        this.lock.lock();
        try {
            List<String> list = this.seed.getMnemonicCode();
            return list;
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean isFollowing() {
        return this.isFollowing;
    }

    @Override
    public List<Protos.Key> serializeToProtobuf() {
        ArrayList result = Lists.newArrayList();
        this.lock.lock();
        try {
            result.addAll(this.serializeMyselfToProtobuf());
        }
        finally {
            this.lock.unlock();
        }
        return result;
    }

    protected List<Protos.Key> serializeMyselfToProtobuf() {
        LinkedList entries = Lists.newLinkedList();
        if (this.seed != null) {
            Protos.Key.Builder mnemonicEntry = BasicKeyChain.serializeEncryptableItem(this.seed);
            mnemonicEntry.setType(Protos.Key.Type.DETERMINISTIC_MNEMONIC);
            DeterministicKeyChain.serializeSeedEncryptableItem(this.seed, mnemonicEntry);
            entries.add(mnemonicEntry.build());
        }
        Map<ECKey, Protos.Key.Builder> keys = this.basicKeyChain.serializeToEditableProtobufs();
        for (Map.Entry<ECKey, Protos.Key.Builder> entry : keys.entrySet()) {
            DeterministicKey key = (DeterministicKey)entry.getKey();
            Protos.Key.Builder proto = entry.getValue();
            proto.setType(Protos.Key.Type.DETERMINISTIC_KEY);
            Protos.DeterministicKey.Builder detKey = proto.getDeterministicKeyBuilder();
            detKey.setChainCode(ByteString.copyFrom((byte[])key.getChainCode()));
            for (ChildNumber num : key.getPath()) {
                log.debug("Serialising a deterministic key with path {}", key.getPath());
                detKey.addPath(num.i());
            }
            if (key.equals(this.externalKey)) {
                detKey.setIssuedSubkeys(this.issuedExternalKeys);
                detKey.setLookaheadSize(this.lookaheadSize);
                detKey.setSigsRequiredToSpend(this.getSigsRequiredToSpend());
            } else if (key.equals(this.internalKey)) {
                detKey.setIssuedSubkeys(this.issuedInternalKeys);
                detKey.setLookaheadSize(this.lookaheadSize);
                detKey.setSigsRequiredToSpend(this.getSigsRequiredToSpend());
            }
            if (entries.isEmpty() && this.isFollowing()) {
                detKey.setIsFollowing(true);
            }
            if (key.getParent() != null) {
                proto.clearCreationTimestamp();
            }
            entries.add(proto.build());
        }
        return entries;
    }

    public static List<DeterministicKeyChain> fromProtobuf(List<Protos.Key> keys, @Nullable KeyCrypter crypter) throws UnreadableWalletException {
        LinkedList chains = Lists.newLinkedList();
        DeterministicSeed seed = null;
        DeterministicKeyChain chain = null;
        int lookaheadSize = -1;
        int sigsRequiredToSpend = 1;
        boolean isTrezor = false;
        boolean foundRootPath = false;
        for (Protos.Key key : keys) {
            boolean isTrezorRootPath;
            DeterministicKey detkey;
            Object passphrase;
            Protos.Key.Type t = key.getType();
            if (t == Protos.Key.Type.DETERMINISTIC_MNEMONIC) {
                if (chain != null) {
                    log.debug("a lookaheadSize = " + lookaheadSize);
                    Preconditions.checkState((lookaheadSize >= 0 ? 1 : 0) != 0);
                    chain.setLookaheadSize(lookaheadSize);
                    chain.setSigsRequiredToSpend(sigsRequiredToSpend);
                    chain.maybeLookAhead();
                    chains.add(chain);
                    chain = null;
                }
                long timestamp = key.getCreationTimestamp() / 1000L;
                passphrase = DEFAULT_PASSPHRASE_FOR_MNEMONIC;
                if (key.hasSecretBytes()) {
                    if (key.hasEncryptedDeterministicSeed()) {
                        throw new UnreadableWalletException("Malformed key proto: " + key.toString());
                    }
                    byte[] seedBytes = null;
                    if (key.hasDeterministicSeed()) {
                        seedBytes = key.getDeterministicSeed().toByteArray();
                    }
                    seed = new DeterministicSeed(key.getSecretBytes().toStringUtf8(), seedBytes, (String)passphrase, timestamp);
                } else if (key.hasEncryptedData()) {
                    if (key.hasDeterministicSeed()) {
                        throw new UnreadableWalletException("Malformed key proto: " + key.toString());
                    }
                    EncryptedData data = new EncryptedData(key.getEncryptedData().getInitialisationVector().toByteArray(), key.getEncryptedData().getEncryptedPrivateKey().toByteArray());
                    EncryptedData encryptedSeedBytes = null;
                    if (key.hasEncryptedDeterministicSeed()) {
                        Protos.EncryptedData encryptedSeed = key.getEncryptedDeterministicSeed();
                        encryptedSeedBytes = new EncryptedData(encryptedSeed.getInitialisationVector().toByteArray(), encryptedSeed.getEncryptedPrivateKey().toByteArray());
                    }
                    seed = new DeterministicSeed(data, encryptedSeedBytes, timestamp);
                } else {
                    throw new UnreadableWalletException("Malformed key proto: " + key.toString());
                }
                if (!log.isDebugEnabled()) continue;
                log.debug("Deserializing: DETERMINISTIC_MNEMONIC: {}", (Object)seed);
                continue;
            }
            if (t != Protos.Key.Type.DETERMINISTIC_KEY) continue;
            if (!key.hasDeterministicKey()) {
                throw new UnreadableWalletException("Deterministic key missing extra data: " + key.toString());
            }
            byte[] chainCode = key.getDeterministicKey().getChainCode().toByteArray();
            LinkedList path = Lists.newLinkedList();
            passphrase = key.getDeterministicKey().getPathList().iterator();
            while (passphrase.hasNext()) {
                int i = (Integer)passphrase.next();
                path.add(new ChildNumber(i));
            }
            ECPoint pubkey = ECKey.CURVE.getCurve().decodePoint(key.getPublicKey().toByteArray());
            ImmutableList immutablePath = ImmutableList.copyOf((Collection)path);
            boolean isAccountKey = false;
            boolean isFollowingKey = false;
            if (key.getDeterministicKey().getIsFollowing()) {
                if (chain != null) {
                    Preconditions.checkState((lookaheadSize >= 0 ? 1 : 0) != 0);
                    chain.setLookaheadSize(lookaheadSize);
                    chain.setSigsRequiredToSpend(sigsRequiredToSpend);
                    chain.maybeLookAhead();
                    chains.add(chain);
                    chain = null;
                    seed = null;
                }
                isFollowingKey = true;
            }
            if (chain == null) {
                boolean isMarried;
                boolean bl = isMarried = !isFollowingKey && !chains.isEmpty() && ((DeterministicKeyChain)chains.get(chains.size() - 1)).isFollowing();
                if (seed == null) {
                    DeterministicKey accountKey = new DeterministicKey((ImmutableList<ChildNumber>)immutablePath, chainCode, pubkey, null, null);
                    chain = isMarried ? new MarriedKeyChain(accountKey) : new DeterministicKeyChain(accountKey, isFollowingKey, (ImmutableList<ChildNumber>)immutablePath);
                    isAccountKey = true;
                    log.debug("B lookaheadSize = " + lookaheadSize);
                } else {
                    chain = isMarried ? new MarriedKeyChain(seed, crypter) : new DeterministicKeyChain(seed, (ImmutableList<ChildNumber>)immutablePath, crypter);
                    chain.lookaheadSize = -1;
                    lookaheadSize = -1;
                }
            }
            log.debug("c");
            DeterministicKey parent = null;
            if (!path.isEmpty() && !isAccountKey) {
                ChildNumber index = (ChildNumber)path.removeLast();
                try {
                    parent = chain.hierarchy.get(path, false, false);
                    path.add(index);
                }
                catch (IllegalArgumentException iae) {
                    log.debug("Ignoring an IllegalArgumentException when trying to get the parent of key with path {}", (Object)path);
                }
                catch (NullPointerException npe) {
                    log.debug("Ignoring a NullPointerException when trying to get the parent of key with path {}", (Object)path);
                }
            }
            if (key.hasSecretBytes()) {
                BigInteger priv = new BigInteger(1, key.getSecretBytes().toByteArray());
                detkey = new DeterministicKey((ImmutableList<ChildNumber>)immutablePath, chainCode, pubkey, priv, parent);
            } else if (key.hasEncryptedData()) {
                Protos.EncryptedData proto = key.getEncryptedData();
                EncryptedData data = new EncryptedData(proto.getInitialisationVector().toByteArray(), proto.getEncryptedPrivateKey().toByteArray());
                Preconditions.checkNotNull((Object)crypter, (Object)"Encountered an encrypted key but no key crypter provided");
                detkey = new DeterministicKey((ImmutableList<ChildNumber>)immutablePath, chainCode, crypter, pubkey, data, parent);
            } else {
                detkey = new DeterministicKey((ImmutableList<ChildNumber>)immutablePath, chainCode, pubkey, null, parent);
            }
            if (key.hasCreationTimestamp()) {
                detkey.setCreationTimeSeconds(key.getCreationTimestamp() / 1000L);
            }
            if (log.isDebugEnabled()) {
                log.debug("Deserializing: DETERMINISTIC_KEY: {}", (Object)detkey);
            }
            boolean bl = isTrezorRootPath = !foundRootPath && (path.size() == 1 || path.size() == 2 || path.size() == 3) && DeterministicKeyChain.isTrezorPath((ImmutableList<ChildNumber>)ImmutableList.copyOf((Collection)path));
            if (path.size() == 0 || isTrezorRootPath) {
                isTrezor = isTrezorRootPath;
                foundRootPath = true;
                chain.rootKey = detkey;
                chain.hierarchy = new DeterministicHierarchy(detkey);
            } else if (!isTrezor && (path.size() == 1 || path.size() == 2) || isTrezor && (path.size() == 5 || path.size() == 6)) {
                if (detkey.getChildNumber().num() == 0) {
                    chain.externalKey = detkey;
                    chain.issuedExternalKeys = key.getDeterministicKey().getIssuedSubkeys();
                    lookaheadSize = Math.max(lookaheadSize, key.getDeterministicKey().getLookaheadSize());
                    sigsRequiredToSpend = key.getDeterministicKey().getSigsRequiredToSpend();
                } else if (detkey.getChildNumber().num() == 1) {
                    chain.internalKey = detkey;
                    chain.issuedInternalKeys = key.getDeterministicKey().getIssuedSubkeys();
                }
            }
            if (chain.hierarchy == null) {
                System.out.println("DeterministicKeyChain#fromProtobuf have a deterministic key but no hierarchy to add it to !");
            } else {
                chain.hierarchy.putKey(detkey);
            }
            chain.basicKeyChain.importKey(detkey);
        }
        if (chain != null) {
            log.debug("c lookaheadSize = " + lookaheadSize);
            Preconditions.checkState((lookaheadSize >= 0 ? 1 : 0) != 0);
            chain.setLookaheadSize(lookaheadSize);
            chain.setSigsRequiredToSpend(sigsRequiredToSpend);
            chain.maybeLookAhead();
            chains.add(chain);
        }
        return chains;
    }

    @Override
    public DeterministicKeyChain toEncrypted(CharSequence password) {
        Preconditions.checkNotNull((Object)password);
        Preconditions.checkArgument((password.length() > 0 ? 1 : 0) != 0);
        Preconditions.checkState((this.seed != null ? 1 : 0) != 0, (Object)"Attempt to encrypt a watching chain.");
        Preconditions.checkState((!this.seed.isEncrypted() ? 1 : 0) != 0);
        KeyCrypterScrypt scrypt = new KeyCrypterScrypt();
        KeyParameter derivedKey = scrypt.deriveKey(password);
        return this.toEncrypted(scrypt, derivedKey);
    }

    @Override
    public DeterministicKeyChain toEncrypted(KeyCrypter keyCrypter, KeyParameter aesKey) {
        return new DeterministicKeyChain(keyCrypter, aesKey, this);
    }

    @Override
    public DeterministicKeyChain toEncrypted(KeyCrypter keyCrypter, KeyParameter aesKey, ImmutableList<ChildNumber> rootNodeList) {
        return new DeterministicKeyChain(keyCrypter, aesKey, this, rootNodeList);
    }

    @Override
    public DeterministicKeyChain toDecrypted(CharSequence password) {
        Preconditions.checkNotNull((Object)password);
        Preconditions.checkArgument((password.length() > 0 ? 1 : 0) != 0);
        KeyCrypter crypter = this.getKeyCrypter();
        Preconditions.checkState((crypter != null ? 1 : 0) != 0, (Object)"Chain not encrypted");
        KeyParameter derivedKey = crypter.deriveKey(password);
        return this.toDecrypted(derivedKey);
    }

    @Override
    public DeterministicKeyChain toDecrypted(KeyParameter aesKey) {
        DeterministicKeyChain chain;
        Preconditions.checkState((this.getKeyCrypter() != null ? 1 : 0) != 0, (Object)"Key chain not encrypted");
        Preconditions.checkState((this.seed != null ? 1 : 0) != 0, (Object)"Can't decrypt a watching chain");
        Preconditions.checkState((boolean)this.seed.isEncrypted());
        String passphrase = DEFAULT_PASSPHRASE_FOR_MNEMONIC;
        DeterministicSeed decSeed = this.seed.decrypt(this.getKeyCrypter(), passphrase, aesKey);
        if (this.rootKey != null && DeterministicKeyChain.isTrezorPath(this.rootKey.getPath())) {
            ArrayList<ChildNumber> trezorRootNodePath = new ArrayList<ChildNumber>();
            trezorRootNodePath.add(new ChildNumber(-2147483604));
            trezorRootNodePath.add(new ChildNumber(Integer.MIN_VALUE));
            trezorRootNodePath.add(new ChildNumber(Integer.MIN_VALUE));
            chain = new DeterministicKeyChain(decSeed, (ImmutableList<ChildNumber>)ImmutableList.copyOf(trezorRootNodePath));
            DeterministicKey internalKeyDecrypted = this.internalKey.decrypt(this.getKeyCrypter(), aesKey);
            chain.hierarchy.putKey(internalKeyDecrypted);
            chain.basicKeyChain.importKey(internalKeyDecrypted);
            System.out.println("DeterministicKeyChain - added internalKeyDecrypted: " + internalKeyDecrypted);
            DeterministicKey externalKeyDecrypted = this.externalKey.decrypt(this.getKeyCrypter(), aesKey);
            chain.hierarchy.putKey(externalKeyDecrypted);
            chain.basicKeyChain.importKey(externalKeyDecrypted);
            System.out.println("DeterministicKeyChain - added externalKeyDecrypted: " + externalKeyDecrypted);
        } else {
            chain = new DeterministicKeyChain(decSeed);
        }
        System.out.println("DeterministicKeyChain - root of decrypted chain: " + chain.toString());
        if (chain.rootKey != null && this.rootKey != null && chain.rootKey.getPath().equals(this.rootKey.getPath()) && !chain.rootKey.getPubKeyPoint().equals(this.rootKey.getPubKeyPoint())) {
            throw new KeyCrypterException("Provided AES key is wrong");
        }
        chain.lookaheadSize = this.lookaheadSize;
        chain.issuedExternalKeys = this.issuedExternalKeys;
        chain.issuedInternalKeys = this.issuedInternalKeys;
        for (ECKey eckey : this.basicKeyChain.getKeys()) {
            DeterministicKey key = (DeterministicKey)eckey;
            if (!DeterministicKeyChain.isTrezorPath(key.getPath()) && key.getPath().size() != 3 || DeterministicKeyChain.isTrezorPath(key.getPath()) && key.getPath().size() != 5) continue;
            Preconditions.checkState((boolean)key.isEncrypted());
            DeterministicKey parent = chain.hierarchy.get((List<ChildNumber>)((DeterministicKey)Preconditions.checkNotNull((Object)key.getParent())).getPath(), false, false);
            key = new DeterministicKey(key.getPubOnly(), parent);
            System.out.println("DeterministicKeyChain - cloned, decrypted key: " + key.toString());
            chain.hierarchy.putKey(key);
            chain.basicKeyChain.importKey(key);
        }
        return chain;
    }

    @Override
    public boolean checkPassword(CharSequence password) {
        Preconditions.checkNotNull((Object)password);
        Preconditions.checkState((this.getKeyCrypter() != null ? 1 : 0) != 0, (Object)"Key chain not encrypted");
        return this.checkAESKey(this.getKeyCrypter().deriveKey(password));
    }

    @Override
    public boolean checkAESKey(KeyParameter aesKey) {
        Preconditions.checkState((this.rootKey != null ? 1 : 0) != 0, (Object)"Can't check password for a watching chain");
        Preconditions.checkNotNull((Object)aesKey);
        Preconditions.checkState((this.getKeyCrypter() != null ? 1 : 0) != 0, (Object)"Key chain not encrypted");
        try {
            return this.rootKey.decrypt(aesKey).getPubKeyPoint().equals(this.rootKey.getPubKeyPoint());
        }
        catch (KeyCrypterException e) {
            return false;
        }
    }

    @Override
    @Nullable
    public KeyCrypter getKeyCrypter() {
        return this.basicKeyChain.getKeyCrypter();
    }

    @Override
    public int numBloomFilterEntries() {
        return this.numKeys() * 2;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public BloomFilter getFilter(int size, double falsePositiveRate, long tweak) {
        this.lock.lock();
        try {
            Preconditions.checkArgument((size >= this.numBloomFilterEntries() ? 1 : 0) != 0);
            this.maybeLookAhead();
            BloomFilter bloomFilter = this.basicKeyChain.getFilter(size, falsePositiveRate, tweak);
            return bloomFilter;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int getLookaheadSize() {
        this.lock.lock();
        try {
            int n = this.lookaheadSize;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void setLookaheadSize(int lookaheadSize) {
        this.lock.lock();
        try {
            boolean readjustThreshold = this.lookaheadThreshold == this.calcDefaultLookaheadThreshold();
            this.lookaheadSize = lookaheadSize;
            if (readjustThreshold) {
                this.lookaheadThreshold = this.calcDefaultLookaheadThreshold();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public void setLookaheadThreshold(int num) {
        this.lock.lock();
        try {
            if (num >= this.lookaheadSize) {
                throw new IllegalArgumentException("Threshold larger or equal to the lookaheadSize");
            }
            this.lookaheadThreshold = num;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int getLookaheadThreshold() {
        this.lock.lock();
        try {
            if (this.lookaheadThreshold >= this.lookaheadSize) {
                int n = 0;
                return n;
            }
            int n = this.lookaheadThreshold;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void maybeLookAhead() {
        this.lock.lock();
        try {
            List<DeterministicKey> keys = this.maybeLookAhead(this.externalKey, this.issuedExternalKeys);
            keys.addAll(this.maybeLookAhead(this.internalKey, this.issuedInternalKeys));
            if (keys.isEmpty()) {
                return;
            }
            ++this.keyLookaheadEpoch;
            this.basicKeyChain.importKeys(keys);
        }
        finally {
            this.lock.unlock();
        }
    }

    private List<DeterministicKey> maybeLookAhead(DeterministicKey parent, int issued) {
        Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread());
        return this.maybeLookAhead(parent, issued, this.getLookaheadSize(), this.getLookaheadThreshold());
    }

    private List<DeterministicKey> maybeLookAhead(DeterministicKey parent, int issued, int lookaheadSize, int lookaheadThreshold) {
        Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread());
        int numChildren = this.hierarchy.getNumChildren(parent.getPath());
        int needed = issued + lookaheadSize + lookaheadThreshold - numChildren;
        if (needed <= lookaheadThreshold) {
            return new ArrayList<DeterministicKey>();
        }
        log.info("{} keys needed for {} = {} issued + {} lookahead size + {} lookahead threshold - {} num children", new Object[]{needed, parent.getPathAsString(), issued, lookaheadSize, lookaheadThreshold, numChildren});
        ArrayList<DeterministicKey> result = new ArrayList<DeterministicKey>(needed);
        long now = System.currentTimeMillis();
        int nextChild = numChildren;
        for (int i = 0; i < needed; ++i) {
            DeterministicKey key = HDKeyDerivation.deriveThisOrNextChildKey(parent, nextChild);
            key = key.getPubOnly();
            this.hierarchy.putKey(key);
            result.add(key);
            nextChild = key.getChildNumber().num() + 1;
        }
        log.info("Took {} msec", (Object)(System.currentTimeMillis() - now));
        return result;
    }

    public void maybeLookAheadScripts() {
    }

    public int getIssuedExternalKeys() {
        this.lock.lock();
        try {
            int n = this.issuedExternalKeys;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int getIssuedInternalKeys() {
        this.lock.lock();
        try {
            int n = this.issuedInternalKeys;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Nullable
    public DeterministicSeed getSeed() {
        this.lock.lock();
        try {
            DeterministicSeed deterministicSeed = this.seed;
            return deterministicSeed;
        }
        finally {
            this.lock.unlock();
        }
    }

    List<ECKey> getKeys(boolean includeLookahead) {
        List<ECKey> keys = this.basicKeyChain.getKeys();
        if (!includeLookahead) {
            int treeSize = this.internalKey.getPath().size();
            LinkedList<ECKey> issuedKeys = new LinkedList<ECKey>();
            for (ECKey key : keys) {
                DeterministicKey detkey = (DeterministicKey)key;
                DeterministicKey parent = detkey.getParent();
                if (parent == null || detkey.getPath().size() <= treeSize || parent.equals(this.internalKey) && detkey.getChildNumber().i() >= this.issuedInternalKeys || parent.equals(this.externalKey) && detkey.getChildNumber().i() >= this.issuedExternalKeys) continue;
                issuedKeys.add(detkey);
            }
            return issuedKeys;
        }
        return keys;
    }

    public List<DeterministicKey> getLeafKeys() {
        ImmutableList.Builder keys = ImmutableList.builder();
        for (ECKey key : this.getKeys(true)) {
            DeterministicKey dKey = (DeterministicKey)key;
            if (dKey.getPath().size() <= 2) continue;
            keys.add((Object)dKey);
        }
        return keys.build();
    }

    static void serializeSeedEncryptableItem(DeterministicSeed seed, Protos.Key.Builder proto) {
        if (seed.isEncrypted() && seed.getEncryptedSeedData() != null) {
            EncryptedData data = seed.getEncryptedSeedData();
            proto.getEncryptedDeterministicSeedBuilder().setEncryptedPrivateKey(ByteString.copyFrom((byte[])data.encryptedBytes)).setInitialisationVector(ByteString.copyFrom((byte[])data.initialisationVector));
            Preconditions.checkState((seed.getEncryptionType() == Protos.Wallet.EncryptionType.ENCRYPTED_SCRYPT_AES ? 1 : 0) != 0);
        } else {
            byte[] secret = seed.getSeedBytes();
            if (secret != null) {
                proto.setDeterministicSeed(ByteString.copyFrom((byte[])secret));
            }
        }
    }

    public int getKeyLookaheadEpoch() {
        this.lock.lock();
        try {
            int n = this.keyLookaheadEpoch;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean isMarried() {
        return false;
    }

    public RedeemData getRedeemData(DeterministicKey followedKey) {
        throw new UnsupportedOperationException();
    }

    public Script freshOutputScript(KeyChain.KeyPurpose purpose) {
        throw new UnsupportedOperationException();
    }

    public String toString(boolean includePrivateKeys, NetworkParameters params) {
        DeterministicKey watchingKey;
        StringBuilder builder2 = new StringBuilder();
        if (this.seed != null) {
            if (this.seed.isEncrypted()) {
                builder2.append(String.format("Seed is encrypted%n", new Object[0]));
            } else if (includePrivateKeys) {
                List<String> words = this.seed.getMnemonicCode();
                builder2.append(String.format("Seed as words: %s%nSeed as hex:   %s%n", Joiner.on((char)' ').join(words), this.seed.toHexString()));
            }
            builder2.append(String.format("Seed birthday: %d  [%s]%n", this.seed.getCreationTimeSeconds(), new Date(this.seed.getCreationTimeSeconds() * 1000L)));
        }
        if ((watchingKey = this.getWatchingKey()).getParent() != null) {
            builder2.append(String.format("Key to watch:  %s%n", watchingKey.serializePubB58(params)));
        }
        this.formatAddresses(includePrivateKeys, params, builder2);
        return builder2.toString();
    }

    protected void formatAddresses(boolean includePrivateKeys, NetworkParameters params, StringBuilder builder2) {
        for (ECKey key : this.getKeys(false)) {
            key.formatKeyWithAddress(includePrivateKeys, builder2, params);
        }
    }

    public void setSigsRequiredToSpend(int sigsRequiredToSpend) {
        this.sigsRequiredToSpend = sigsRequiredToSpend;
    }

    public int getSigsRequiredToSpend() {
        return this.sigsRequiredToSpend;
    }

    @Nullable
    public RedeemData findRedeemDataByScriptHash(ByteString bytes) {
        return null;
    }

    public static boolean isTrezorPath(ImmutableList<ChildNumber> path) {
        return path != null && path.size() > 0 && new ChildNumber(44, true).equals(path.get(0));
    }

    @Nullable
    public DeterministicKey getRootKey() {
        return this.rootKey;
    }

    public static class Builder<T extends Builder<T>> {
        protected SecureRandom random;
        protected int bits = 128;
        protected String passphrase;
        protected long seedCreationTimeSecs;
        protected byte[] entropy;
        protected DeterministicSeed seed;

        protected Builder() {
        }

        protected T self() {
            return (T)this;
        }

        public T entropy(byte[] entropy) {
            this.entropy = entropy;
            return this.self();
        }

        public T seed(DeterministicSeed seed) {
            this.seed = seed;
            return this.self();
        }

        public T random(SecureRandom random, int bits) {
            this.random = random;
            this.bits = bits;
            return this.self();
        }

        public T random(SecureRandom random) {
            this.random = random;
            return this.self();
        }

        public T passphrase(String passphrase) {
            this.passphrase = passphrase;
            return this.self();
        }

        public DeterministicKeyChain build() {
            Preconditions.checkState((this.random != null || this.entropy != null || this.seed != null ? 1 : 0) != 0, (Object)"Must provide either entropy or random");
            Preconditions.checkState((this.passphrase == null || this.seed == null ? 1 : 0) != 0, (Object)"Passphrase must not be specified with seed");
            DeterministicKeyChain chain = this.random != null ? new DeterministicKeyChain(this.random, this.bits, this.getPassphrase(), this.seedCreationTimeSecs) : (this.entropy != null ? new DeterministicKeyChain(this.entropy, this.getPassphrase(), this.seedCreationTimeSecs) : new DeterministicKeyChain(this.seed));
            return chain;
        }

        protected String getPassphrase() {
            return this.passphrase != null ? this.passphrase : DeterministicKeyChain.DEFAULT_PASSPHRASE_FOR_MNEMONIC;
        }
    }
}

