/*
	Apache License
	Version 2.0, January 2004
	http://www.apache.org/licenses/
	
	TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
	
	1. Definitions.
	
	"License" shall mean the terms and conditions for use, reproduction, and
	distribution as defined by Sections 1 through 9 of this document.
	
	"Licensor" shall mean the copyright owner or entity authorized by the copyright
	owner that is granting the License.
	
	"Legal Entity" shall mean the union of the acting entity and all other entities
	that control, are controlled by, or are under common control with that entity.
	For the purposes of this definition, "control" means (i) the power, direct or
	indirect, to cause the direction or management of such entity, whether by
	contract or otherwise, or (ii) ownership of fifty percent (50%) or more of the
	outstanding shares, or (iii) beneficial ownership of such entity.
	
	"You" (or "Your") shall mean an individual or Legal Entity exercising
	permissions granted by this License.
	
	"Source" form shall mean the preferred form for making modifications, including
	but not limited to software source code, documentation source, and configuration
	files.
	
	"Object" form shall mean any form resulting from mechanical transformation or
	translation of a Source form, including but not limited to compiled object code,
	generated documentation, and conversions to other media types.
	
	"Work" shall mean the work of authorship, whether in Source or Object form, made
	available under the License, as indicated by a copyright notice that is included
	in or attached to the work (an example is provided in the Appendix below).
	
	"Derivative Works" shall mean any work, whether in Source or Object form, that
	is based on (or derived from) the Work and for which the editorial revisions,
	annotations, elaborations, or other modifications represent, as a whole, an
	original work of authorship. For the purposes of this License, Derivative Works
	shall not include works that remain separable from, or merely link (or bind by
	name) to the interfaces of, the Work and Derivative Works thereof.
	
	"Contribution" shall mean any work of authorship, including the original version
	of the Work and any modifications or additions to that Work or Derivative Works
	thereof, that is intentionally submitted to Licensor for inclusion in the Work
	by the copyright owner or by an individual or Legal Entity authorized to submit
	on behalf of the copyright owner. For the purposes of this definition,
	"submitted" means any form of electronic, verbal, or written communication sent
	to the Licensor or its representatives, including but not limited to
	communication on electronic mailing lists, source code control systems, and
	issue tracking systems that are managed by, or on behalf of, the Licensor for
	the purpose of discussing and improving the Work, but excluding communication
	that is conspicuously marked or otherwise designated in writing by the copyright
	owner as "Not a Contribution."
	
	"Contributor" shall mean Licensor and any individual or Legal Entity on behalf
	of whom a Contribution has been received by Licensor and subsequently
	incorporated within the Work.
	
	2. Grant of Copyright License.
	
	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable copyright license to reproduce, prepare Derivative Works of,
	publicly display, publicly perform, sublicense, and distribute the Work and such
	Derivative Works in Source or Object form.
	
	3. Grant of Patent License.
	
	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable (except as stated in this section) patent license to make, have
	made, use, offer to sell, sell, import, and otherwise transfer the Work, where
	such license applies only to those patent claims licensable by such Contributor
	that are necessarily infringed by their Contribution(s) alone or by combination
	of their Contribution(s) with the Work to which such Contribution(s) was
	submitted. If You institute patent litigation against any entity (including a
	cross-claim or counterclaim in a lawsuit) alleging that the Work or a
	Contribution incorporated within the Work constitutes direct or contributory
	patent infringement, then any patent licenses granted to You under this License
	for that Work shall terminate as of the date such litigation is filed.
	
	4. Redistribution.
	
	You may reproduce and distribute copies of the Work or Derivative Works thereof
	in any medium, with or without modifications, and in Source or Object form,
	provided that You meet the following conditions:
	
	You must give any other recipients of the Work or Derivative Works a copy of
	this License; and
	You must cause any modified files to carry prominent notices stating that You
	changed the files; and
	You must retain, in the Source form of any Derivative Works that You distribute,
	all copyright, patent, trademark, and attribution notices from the Source form
	of the Work, excluding those notices that do not pertain to any part of the
	Derivative Works; and
	If the Work includes a "NOTICE" text file as part of its distribution, then any
	Derivative Works that You distribute must include a readable copy of the
	attribution notices contained within such NOTICE file, excluding those notices
	that do not pertain to any part of the Derivative Works, in at least one of the
	following places: within a NOTICE text file distributed as part of the
	Derivative Works; within the Source form or documentation, if provided along
	with the Derivative Works; or, within a display generated by the Derivative
	Works, if and wherever such third-party notices normally appear. The contents of
	the NOTICE file are for informational purposes only and do not modify the
	License. You may add Your own attribution notices within Derivative Works that
	You distribute, alongside or as an addendum to the NOTICE text from the Work,
	provided that such additional attribution notices cannot be construed as
	modifying the License.
	You may add Your own copyright statement to Your modifications and may provide
	additional or different license terms and conditions for use, reproduction, or
	distribution of Your modifications, or for any such Derivative Works as a whole,
	provided Your use, reproduction, and distribution of the Work otherwise complies
	with the conditions stated in this License.
	
	5. Submission of Contributions.
	
	Unless You explicitly state otherwise, any Contribution intentionally submitted
	for inclusion in the Work by You to the Licensor shall be under the terms and
	conditions of this License, without any additional terms or conditions.
	Notwithstanding the above, nothing herein shall supersede or modify the terms of
	any separate license agreement you may have executed with Licensor regarding
	such Contributions.
	
	6. Trademarks.
	
	This License does not grant permission to use the trade names, trademarks,
	service marks, or product names of the Licensor, except as required for
	reasonable and customary use in describing the origin of the Work and
	reproducing the content of the NOTICE file.
	
	7. Disclaimer of Warranty.
	
	Unless required by applicable law or agreed to in writing, Licensor provides the
	Work (and each Contributor provides its Contributions) on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied,
	including, without limitation, any warranties or conditions of TITLE,
	NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A PARTICULAR PURPOSE. You are
	solely responsible for determining the appropriateness of using or
	redistributing the Work and assume any risks associated with Your exercise of
	permissions under this License.
	
	8. Limitation of Liability.
	
	In no event and under no legal theory, whether in tort (including negligence),
	contract, or otherwise, unless required by applicable law (such as deliberate
	and grossly negligent acts) or agreed to in writing, shall any Contributor be
	liable to You for damages, including any direct, indirect, special, incidental,
	or consequential damages of any character arising as a result of this License or
	out of the use or inability to use the Work (including but not limited to
	damages for loss of goodwill, work stoppage, computer failure or malfunction, or
	any and all other commercial damages or losses), even if such Contributor has
	been advised of the possibility of such damages.
	
	9. Accepting Warranty or Additional Liability.
	
	While redistributing the Work or Derivative Works thereof, You may choose to
	offer, and charge a fee for, acceptance of support, warranty, indemnity, or
	other liability obligations and/or rights consistent with this License. However,
	in accepting such obligations, You may act only on Your own behalf and on Your
	sole responsibility, not on behalf of any other Contributor, and only if You
	agree to indemnify, defend, and hold each Contributor harmless for any liability
	incurred by, or claims asserted against, such Contributor by reason of your
	accepting any such warranty or additional liability.
	
	END OF TERMS AND CONDITIONS
	
	APPENDIX: How to apply the Apache License to your work
	
	To apply the Apache License to your work, attach the following boilerplate
	notice, with the fields enclosed by brackets "{}" replaced with your own
	identifying information. (Don't include the brackets!) The text should be
	enclosed in the appropriate comment syntax for the file format. We also
	recommend that a file or class name and description of purpose be included on
	the same "printed page" as the copyright notice for easier identification within
	third-party archives.
	
	   Copyright 2020 t-io
	
	   Licensed under the Apache License, Version 2.0 (the "License");
	   you may not use this file except in compliance with the License.
	   You may obtain a copy of the License at
	
	     http://www.apache.org/licenses/LICENSE-2.0
	
	   Unless required by applicable law or agreed to in writing, software
	   distributed under the License is distributed on an "AS IS" BASIS,
	   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	   See the License for the specific language governing permissions and
	   limitations under the License.
*/
package org.tio.core.ssl.facade;

import java.nio.ByteBuffer;

import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tio.core.ChannelContext;
import org.tio.core.ssl.SslVo;
import org.tio.utils.hutool.StrUtil;

class Worker {

	private static Logger log = LoggerFactory.getLogger(Worker.class);

	/*  Uses the SSLEngine and Buffers to perform wrap/unwrap operations.
	 Also, provides access to SSLEngine ops for handshake
	 */
	private final static String		TAG						= "Worker";
	private final SSLEngine			_engine;
	private final Buffers			_buffers;
	private ISSLListener			_sslListener;
	private ISessionClosedListener	_sessionClosedListener	= new DefaultOnCloseListener();
	@SuppressWarnings("unused")
	private String					who;

	private ChannelContext channelContext;

	Worker(final String debugTag, SSLEngine engine, Buffers buffers, ChannelContext channelContext) {
		_engine = engine;
		_buffers = buffers;
		this.channelContext = channelContext;
		this.who = "[Worker:" + debugTag + "]";
	}

	@SuppressWarnings("unused")
	private void debug(final String msg, final String... args) {
		SSLLog.debug(TAG, msg, args);
	}

	void setSessionClosedListener(final ISessionClosedListener scl) {
		_sessionClosedListener = scl;
	}

	void beginHandshake() throws SSLException {
		_engine.beginHandshake();
	}

	SSLEngineResult.HandshakeStatus getHandshakeStatus() {
		return _engine.getHandshakeStatus();
	}

	/**
	 * 只是简单地调一下SSLEngine.getDelegatedTask()
	 * @return
	 */
	Runnable getDelegatedTask() {
		return _engine.getDelegatedTask();
	}

	/**
	 * 加密
	 * @param sslVo
	 * @param plainData
	 * @return
	 * @throws SSLException
	 */
	SSLEngineResult wrap(SslVo sslVo, ByteBuffer plainData) throws SSLException {
		_buffers.prepareForWrap(plainData);
		SSLEngineResult result = doWrap();

		emitWrappedData(sslVo, result);

		switch (result.getStatus()) {
		case BUFFER_UNDERFLOW:
			throw new RuntimeException("BUFFER_UNDERFLOW while wrapping!");
		case BUFFER_OVERFLOW:
			_buffers.grow(BufferType.OUT_CIPHER);
			if (plainData != null && plainData.hasRemaining()) {
				plainData.position(result.bytesConsumed());
				ByteBuffer remainingData = BufferUtils.slice(plainData);
				wrap(sslVo, remainingData);
			}
			break;
		case OK:
			break;
		case CLOSED:
			_sessionClosedListener.onSessionClosed();
			break;
		}
		return result;
	}

	/**
	 * 解密
	 * @param sslVo
	 * @param encryptedData 待解密的数据
	 * @return
	 * @throws SSLException
	 */
	SSLEngineResult unwrap(ByteBuffer encryptedData) throws SSLException {
		ByteBuffer allEncryptedData = _buffers.prependCached(encryptedData);
		_buffers.prepareForUnwrap(allEncryptedData);
		SSLEngineResult result = doUnwrap();
		allEncryptedData.position(result.bytesConsumed());
		ByteBuffer unprocessedEncryptedData = BufferUtils.slice(allEncryptedData);//未处理的数据

		emitPlainData(result);

		switch (result.getStatus()) {
		case BUFFER_UNDERFLOW: //数据不够解密不了，则把剩下的数据存起来，下次继续使用
			_buffers.cache(unprocessedEncryptedData);
			break;
		case BUFFER_OVERFLOW:
			_buffers.grow(BufferType.IN_PLAIN);
			if (unprocessedEncryptedData == null) {
				throw new RuntimeException("Worker.unwrap had " + "buffer_overflow but all data was consumed!!");
			} else {
				//				unwrap(sslVo, unprocessedEncryptedData);
				unwrap(unprocessedEncryptedData);
			}
			break;
		case OK:
			if (unprocessedEncryptedData == null) {
				_buffers.clearCache();
			} else {
				_buffers.cache(unprocessedEncryptedData);
			}
			break;
		case CLOSED:
			_sessionClosedListener.onSessionClosed();
			break;
		}
		if (_buffers.isCacheEmpty() == false && result.getStatus() == SSLEngineResult.Status.OK && result.bytesConsumed() > 0) {
			//			debug("Still data in cahce");
			ByteBuffer byteBuffer = ByteBuffer.allocate(0);
			result = unwrap(byteBuffer);
		}
		return result;
	}

	void setSSLListener(ISSLListener SSLListener) {
		_sslListener = SSLListener;
	}

	void handleEnOfSession(final SSLEngineResult result) {
		if (result.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING) {
			_sessionClosedListener.onSessionClosed();
		}
	}

	void close(boolean properly) {
		_engine.closeOutbound();
		try {
			if (properly) {
				wrap(new SslVo(), null); //sends a TLS close_notify alert
			}
			_engine.closeInbound();
		} catch (SSLException ignore) {
		}

	}

	boolean isCloseCompleted() {
		return _engine.isOutboundDone();
	}

	boolean pendingUnwrap() {
		return !_buffers.isCacheEmpty();
	}
	/* Private */

	private void emitWrappedData(SslVo sslVo, SSLEngineResult result) {
		if (result.bytesProduced() > 0) {
			ByteBuffer internalCipherBuffer = _buffers.get(BufferType.OUT_CIPHER);
			sslVo.setByteBuffer(makeExternalBuffer(internalCipherBuffer));
			_sslListener.onWrappedData(sslVo);
		}
	}

	private void emitPlainData(SSLEngineResult result) {
		if (result.bytesProduced() > 0) {
			ByteBuffer internalPlainBuffer = _buffers.get(BufferType.IN_PLAIN);
			ByteBuffer plainBuffer = (makeExternalBuffer(internalPlainBuffer));
			_sslListener.onPlainData(plainBuffer);
		}

	}

	/**
	 * 加密
	 * @return
	 * @throws SSLException
	 */
	private SSLEngineResult doWrap() throws SSLException {
		try {
			ByteBuffer plainText = _buffers.get(BufferType.OUT_PLAIN);
			ByteBuffer cipherText = _buffers.get(BufferType.OUT_CIPHER);
			if (log.isInfoEnabled()) {
				log.info("{}, doWrap(加密): plainText:{} to cipherText: {}", channelContext, plainText, cipherText);
			}
			return _engine.wrap(plainText, cipherText);
		} catch (SSLException e) {
			throw e;
		}
	}

	/**
	 * 解密
	 * @return
	 * @throws SSLException
	 */
	private SSLEngineResult doUnwrap() throws SSLException {
		ByteBuffer cipherText = _buffers.get(BufferType.IN_CIPHER);
		ByteBuffer plainText = _buffers.get(BufferType.IN_PLAIN);
		try {
			log.info("{}, doUnwrap(解密): 密文buffer:{}, 明文buffer: {}", channelContext, cipherText, plainText);
			return _engine.unwrap(cipherText, plainText);
		} catch (SSLException e) {
			if (log.isInfoEnabled()) {
				byte[] bs = new byte[cipherText.limit()];
				System.arraycopy(cipherText.array(), 0, bs, 0, bs.length);
				log.error(channelContext + ", 解密Error:" + e.toString() + ", byte:" + StrUtil.arrayToString(bs) + ", string:" + new String(bs) + ", buffer:" + cipherText, e);
			}
			throw e;
		}
	}

	private static ByteBuffer makeExternalBuffer(ByteBuffer internalBuffer) {
		ByteBuffer newBuffer = ByteBuffer.allocate(internalBuffer.position());
		internalBuffer.flip();
		BufferUtils.copy(internalBuffer, newBuffer);
		return newBuffer;
	}

}
