<?php

namespace Overtrue\Socialite\Providers;

use Overtrue\Socialite\User;

/**
 * @see https://developer.baidu.com/wiki/index.php?title=docs/oauth [OAuth 2.0 授权机制说明]
 */
class Baidu extends Base
{
    const NAME = 'baidu';
    protected $baseUrl = 'https://openapi.baidu.com';
    protected $version = '2.0';
    protected $scopes = ['basic'];
    protected $display = 'popup';

    /**
     * @param $display
     *
     * @return $this
     */
    public function withDisplay($display)
    {
        $this->display = $display;

        return $this;
    }

    /**
     * @param $scopes
     *
     * @return self
     */
    public function withScopes($scopes)
    {
        $this->scopes = $scopes;

        return $this;
    }

    /**
     * @return string
     */
    protected function getAuthUrl()
    {
        return $this->buildAuthUrlFromBase($this->baseUrl . '/oauth/' . $this->version . '/authorize');
    }

    protected function getCodeFields()
    {
        return [
                'response_type' => 'code',
                'client_id' => $this->getClientId(),
                'redirect_uri' => $this->redirectUrl,
                'scope' => $this->formatScopes($this->scopes, $this->scopeSeparator),
                'display' => $this->display,
            ] + $this->parameters;
    }

    /**
     * @return string
     */
    protected function getTokenUrl()
    {
        return $this->baseUrl . '/oauth/' . $this->version . '/token';
    }

    /**
     * @param $code
     *
     * @return array
     */
    protected function getTokenFields($code)
    {
        return parent::getTokenFields($code) + ['grant_type' => 'authorization_code'];
    }

    /**
     * @param   $token
     *
     * @return array
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    protected function getUserByToken($token)
    {
        $response = $this->getHttpClient()->get(
            $this->baseUrl . '/rest/' . $this->version . '/passport/users/getInfo',
            [
                'query' => [
                    'access_token' => $token,
                ],
                'headers' => [
                    'Accept' => 'application/json',
                ],
            ]
        );

        return json_decode($response->getBody(), true) ?? [];
    }

    /**
     * @param $user
     *
     * @return \Overtrue\Socialite\User
     */
    protected function mapUserToObject($user)
    {
        return new User(
            [
                'id' => $user['userid'] ?? null,
                'nickname' => $user['realname'] ?? null,
                'name' => $user['username'] ?? null,
                'email' => '',
                'avatar' => $user['portrait'] ? 'http://tb.himg.baidu.com/sys/portraitn/item/' . $user['portrait'] : null,
            ]
        );
    }
}
