<?php
/**
 * DbSqlite - Sqlite的数据访问类
 * 
 * 提供最基本的数据访问,使用PDO访问数据库，根据文件名判断sqlite版本
 *
 * @package		PicWind
 * @author		无招之击<houbenbo@gmail> 
 * @version		0.2
 */
class DbSqlite {
	/**
	 * 数据库连接的状态
	 *
	 * @var bool
	 */
	public $ready = false;

	/**
	 * 存储的错误信息
	 *
	 * @var string
	 */
	 private $errormsg = '';

	/**
	 * 是否显示错误信息
	 *
	 * @var string
	 */
	 private $showerror = true;

	/**
	 * 构造函数，建立数据连接
	 *
	 * @param string $dbname Sqlite数据库路径
	 */
	function __construct($dbname) {
		if ( is_file($dbname) ) {
			$sqlite = substr($dbname, strrpos($dbname, '.')) == '.sqlite' ? 'sqlite2:' : 'sqlite:';
			$this->conn = new PDO($sqlite.$dbname);
			$this->conn->setAttribute(PDO::ATTR_ERRMODE,PDO::ERRMODE_EXCEPTION);
			$this->ready = true;
		}
	}

	/**
	 * 执行数据库查询
	 * 
	 * @param string $sql 要执行的Sql语句
	 * @return int|array|flase 返回影响的行数或结果数组或错误
	 */
	function query($sql) {
		if ( !$this->ready ) return false;
		try {
			if ( strtolower(substr(trim($sql),0,6))=="select" ) {
				//执行获取查询
				$query = $this->conn->query($sql);
				return $query->fetchAll(PDO::FETCH_ASSOC);
			} else {
				//返回执行行数
				return $this->conn->exec($sql);
			}
		} catch ( PDOException $e ) {
			$this->errormsg .= $e->getMessage();
			$this->errormsg .= '<code>'.$sql.'</code>';
			$this->printError();
			return false;
		}
	}

	/**
	 * 分页返回数据
	 * 
	 * @param string $sql 获取语句
	 * @param int $recordcount 返回记录数
	 * @param int $pageindex 当前页
	 * @param int $pagesize 页面大小
	 * @return array 数据
	 */
	function pageQuery($sql, &$recordcount, $pageindex=1, $pagesize=20) {
		if ( $pageindex<1 ) $pageindex = 1;
		$countsql = 'select count(*) '.substr($sql,stripos(strtolower($sql),' from '));
		$recordcount = $this->getValue($countsql);
		return $this->query($sql . ' limit ' . ($pageindex - 1) * $pagesize . ',' . $pagesize);
	}
	
	/**
	 * 向数据表中添加记录
	 *
	 * @param string $tablename 数据表的名字
	 * @param array $data 数据array('col1'=>val1,...)
	 * @param string $colstring 字段列表col1|col2|col3...
	 * @return int|false 返回执行结果
	 */
	function dataAdd($tablename,$data,$colstring = '') {
		if ( $colstring != '' ) {
			$cols = explode('|',$colstring);
			$vals = array();
			foreach($cols as $col) $vals[] = $this->escape($data[$col]);
		} else {
			$cols = array_keys($data);
			$vals = $this->escape($data);
		}
		$sql = "insert into $tablename (".implode(',',$cols).") values (".implode(',',$vals).")";
		return $this->query($sql);
	}

	/**
	 * 更新数据
	 *
	 * @param string $tablename 数据表的名字
	 * @param array $data 数据array('col1'=>val1,...)
	 * @param string $where 条件语句
	 * @param string $colstring 字段列表col1|col2|col3...
	 * @return int|false 返回执行结果
	 */
	function  dataUpdate($tablename,$data,$where,$colstring=''){
		if ( $colstring != '' ) $cols = explode('|',$colstring);
		else $cols = array_keys($data);
		$vals = array();
		foreach($cols as $col) $vals[] = $col . "=" . $this->escape($data[$col]) ;
		$sql = "update $tablename set ".implode(',',$vals)." ".$where;
		return $this->query($sql);
	}

	/**
	 * 获取单行数据
	 *
	 * @param string $sql 查询语句
	 * @return array|false 查询结果或false
	 */
	function getRow($sql){
		$re = $this->query($sql);
		return (count($re)>0) ? $re[0] : false;
	}

	/**
	 * 获取单个数据
	 *
	 * @param string $sql 查询语句
	 * @return string|false 查询结果或false
	 */
	function getValue($sql){
		$re = $this->getRow($sql);
		if(!$re) return false;
		foreach($re as $rt) return $rt;
	}

	/**
	 * 开始事务，用于大量集中的数据处理
	 *
	 * 执行此方法之后，如果没有后续操作，将自动回滚事务
	 * @return void
	 */
	function begin() {
		$this->conn->beginTransaction();
	}

	/**
	 * 提交由begin()函数开始的事务
	 *
	 * @return void
	 */
	function commit() {
		$this->conn->commit();
	}

	/**
	 * 对数据进行编码以插入数据库
	 *
	 * @var string|array 要编码的数据
	 * @return string|array 编码后的数据
	 */
	 function escape($data) {
		if ( is_string($data) ) return "'".sqlite_escape_string($data)."'";
		else if ( is_array($data) ) foreach ( $data as $key => $val ) $data[$key] = $this->escape($val);
		else return "'" . $data . "'";
		return $data;
	 }

	/**
	 * 输出错误信息
	 *
	 * @return void
	 */
	 function printError(){
		 if ( !$this->showerror ) return ;
		 echo '<h1>出错了</h1><div>'.$this->errormsg.'</div>';
		 return false;
	 }
}