﻿using System;
using System.Data;
using System.Data.Common;
using System.Collections;
using System.Collections.Generic;
using System.Configuration.Provider;
using System.Collections.Specialized;

namespace Winson.Framework.DBProviders
{
    /// <summary>
    /// 数据库提供基类
    /// </summary>
    internal abstract class DataBaseProvider : ProviderBase
    {
        protected string connStringName;

        public override void Initialize(string name, NameValueCollection config)
        {
            if (config == null)
            {
                throw new ArgumentNullException("没有相关配置!");
            }

            if (String.IsNullOrEmpty(name))
            {
                name = "DBProvider";
            }

            if (String.IsNullOrEmpty(config["description"]))
            {
                config.Remove("description");
                config.Add("description", "创建SQL数据库提供者");
            }

            base.Initialize(name, config);

            if (config["connectionStringName"] == null)
            {
                // 默认的数据库类型
                config["connectionStringName"] = "SqlProvider";
            }
            connStringName = config["connectionStringName"];
            config.Remove("connectionStringName");


            //如果有多余的属于，则抛出异常
            if (config.Count > 0)
            {
                string attr = config.GetKey(0);
                if (!String.IsNullOrEmpty(attr))
                    throw new ProviderException("未指定的属性: " + attr);
            }
        }
        /// <summary>
        /// 当前数据库连接对象
        /// </summary>
        public abstract DbConnection Conn();
        /// <summary>
        /// 是否启用事务
        /// </summary>
        /// <param name="isuse">true:启用  false:不用</param>
        public abstract void UseTransaction(bool isuse);
        /// <summary>
        /// 获取表的主键
        /// </summary>
        /// <param name="tableName">表名</param>
        /// <returns></returns>
        public abstract string GetPrimeKey(string tableName);
        /// <summary>
        /// 生成获取一条实体记录的SQL语句
        /// </summary>
        /// <param name="tableName">表名</param>
        /// <param name="primeKey">关键字</param>
        /// <returns>SQL语句</returns>
        public abstract string GenerateGetOneEntitySql(string tableName, string primeKey);
        /// <summary>
        /// 生成分页的SQL语句
        /// </summary>
        /// <param name="tableName">表名</param>
        /// <param name="limit">每页显示记录数</param>
        /// <param name="currpage">当前页数</param>
        /// <param name="criteria">查询条件,null为查询所有</param>
        /// <param name="orderby">排序字段</param>
        /// <param name="order">排序方式，默认为 ASC</param>
        /// <returns>SQL语句</returns>
        public abstract string GeneratePagerSql(string tableName, int limit, int currpage, string criteria, string orderby, string order);
        /// <summary>
        /// 执行SQL语句
        /// </summary>
        /// <param name="dbAction">执行SQL语句的类型，分别为Insert,Update,Delete</param>
        /// <param name="tableName">要操作的表名</param>
        /// <param name="strSql">SQL语句</param>
        /// <param name="fieldName">字段名列表</param>
        /// <param name="fieldValue">字段参数值</param>
        /// <param name="conditionValue">条件参数值</param>
        /// <returns>返回结果记录数</returns>
        public abstract int ExecuteNonQuery(DBAction dbAction, string tableName, string strSql, ArrayList fieldName, ArrayList fieldValue, NameValueCollection conditionValue);
        /// <summary>
        /// 批量增、删、改操作
        /// </summary>
        /// <param name="dbAction">执行SQL语句的类型，分别为Insert,Update,Delete</param>
        /// <param name="tableName">要操作的表名</param>
        /// <param name="fieldNameList">字段名列表集合</param>
        /// <param name="fieldValueList">字段参数值集合</param>
        /// <returns>成功返回True 否则返回False</returns>
        public abstract bool ExecuteBatch(DBAction dbAction, string tableName, List<ArrayList> fieldNameList, List<ArrayList> fieldValueList);
        /// <summary>
        /// 执行SQL语句，返回第一行第一列的值
        /// </summary>
        /// <param name="strSql">SQL语句</param>
        /// <param name="paramValue">参数集合</param>
        /// <returns>返回第一行,第一列的值</returns>
        public abstract object ExecuteScalar(string strSql, NameValueCollection paramValue);
        /// <summary>
        /// 执行SQL语句，返回DbDataReader
        /// </summary>
        /// <param name="conn">数据库连接对象</param>
        /// <param name="strSql">SQL语句</param>
        /// <param name="paramValue">参数集合</param>
        /// <returns>返回DbDataReader</returns>
        public abstract DbDataReader ExecuteReader(DbConnection conn, string strSql, NameValueCollection paramValue);
        /// <summary>
        /// 执行SQL语句，填充指定表，并返回DataSet
        /// </summary>
        /// <param name="strSql">SQL语句</param>
        /// <param name="tableName">填充的表名</param>
        /// <param name="paramValue">参数集合</param>
        /// <returns>返回DataSet</returns>
        public abstract DataSet ExecuteDataset(string strSql, string tableName, NameValueCollection paramValue);
        /// <summary>
        /// 执行存储过程
        /// </summary>
        /// <param name="procName">存储过程名</param>
        /// <param name="paramValue">参数列表</param>
        /// <returns>成功返回True 否则返回False</returns>
        public abstract bool ExecuteProcedure(string procName, NameValueCollection paramValue);
        /// <summary>
        /// 执行存储过程,返回第一行第一列的值 
        /// </summary>
        /// <param name="procName">存储过程名</param>
        /// <param name="paramValue">参数列表</param>
        /// <returns>返回第一行,第一列的值</returns>
        public abstract object ExecuteProcScalar(string procName, NameValueCollection paramValue);
        /// <summary>
        /// 执行存储过程，返回DataSet
        /// </summary>
        /// <param name="procName">存储过程名</param>
        /// <param name="tableName">填充表名</param>
        /// <param name="paramValue">参数列表</param>
        /// <returns>返回DataSet</returns>
        public abstract DataSet ExecuteProcDataset(string procName, string tableName, NameValueCollection paramValue);
    }
}
