Imports Microsoft.VisualBasic
Imports System
Imports System.Collections
Imports NetShopForge.Library.Tracker

Namespace NetShopForge.Library.Tracker

    ''' <summary>
    ''' Represents a collection of TrackedItem objects.
    ''' </summary>
    Public Class TrackedItemCollection

        Implements System.Collections.ICollection

        Public Const COLLECTION_LIMIT As Integer = 5
        Public Sub Sort()
            items.Sort()
        End Sub


        ' Provides the base object storage used by this class
        Private Items As System.Collections.ArrayList

        ' This version counter is incremented any time the collection changes.
        Private listVersion As Integer

        ''' <summary>
        ''' Initializes a new instance of the TrackedItemCollection> class.
        ''' </summary>
        Public Sub New()
            items = New System.Collections.ArrayList()
            listVersion = 0
        End Sub

        ''' <summary>
        ''' Returns a number that can be used to determine if the collection has changed since some previous
        ''' reference point, since this value changes each time the collection is modified in any way.
        ''' </summary>
        Friend ReadOnly Property Version() As Integer
            Get
                Return listVersion
            End Get
        End Property

        ''' <summary>
        ''' Gets the number objects contained in the collection.
        ''' </summary>
        Public Overridable ReadOnly Property Count() As Integer Implements System.Collections.ICollection.Count
            Get
                Return items.Count
            End Get
        End Property

        ''' <summary>
        ''' Gets a value indicating whether access to the collection is synchronized (thread-safe).
        ''' </summary>
        Public Overridable ReadOnly Property IsSynchronized() As Boolean Implements System.Collections.ICollection.IsSynchronized
            Get
                Return items.IsSynchronized
            End Get
        End Property

        ''' <summary>
        ''' Gets an object that can be used to synchronize access to the collection.
        ''' </summary>
        Public Overridable ReadOnly Property SyncRoot() As Object Implements System.Collections.ICollection.SyncRoot
            Get
                Return items.SyncRoot
            End Get
        End Property

        ''' <summary>
        ''' Returns the TrackedItem at the specified index of the TrackedItemCollection.
        ''' </summary>
        ''' <param name='index'>The zero-based index of the TrackedItem to locate in the collection.</param>
        ''' <returns> The TrackedItem at the specified index of the collection.</returns>
        ''' <exception cref='System.ArgumentOutOfRangeException'><paramref name='index'/> is outside the valid range of indexes for the collection.</exception>
        Default Public Overridable ReadOnly Property Item(ByVal index As Integer) As TrackedItem
            Get
                Return (CType(Me.items(index), TrackedItem))
            End Get
        End Property

        ''' <summary>
        ''' Called when an object is being inserted into the collection.
        ''' </summary>
        Public Event Inserting As TrackedItemCollectionChangedEventHandler

        ''' <summary>
        ''' Called after an object has been inserted into the collection.
        ''' </summary>
        Public Event InsertComplete As TrackedItemCollectionChangedEventHandler

        ''' <summary>
        ''' Called when an object is being removed from the collection.
        ''' </summary>
        Public Event Removing As TrackedItemCollectionChangedEventHandler

        ''' <summary>
        ''' Called after an object has been removed from the collection.
        ''' </summary>
        Public Event RemoveComplete As TrackedItemCollectionChangedEventHandler

        ''' <summary>
        ''' Called when the collection is being cleared.
        ''' </summary>
        Public Event Clearing As TrackedItemCollectionClearedEventHandler

        ''' <summary>
        ''' Called after the collection has been cleared.
        ''' </summary>
        Public Event ClearComplete As TrackedItemCollectionClearedEventHandler

        ''' <summary>
        ''' Internal method called whenever the collection changes to increment the Version property.
        ''' </summary>
        Private Sub IncrementVersion()
            listVersion = (listVersion + 1)
        End Sub

        ''' <summary>
        ''' Returns a <see cref='System.Collections.IEnumerator'/> object that can be used to enumerate
        ''' the entire collection.
        ''' </summary>
        Public Overridable Function GetEnumerator() As System.Collections.IEnumerator Implements System.Collections.ICollection.GetEnumerator
            Return items.GetEnumerator()
        End Function

        ''' <summary>
        ''' Copies the elements of this TrackedItemCollection to a System.Array, starting at a particular System.Array index.
        ''' </summary>
        Public Overridable Sub CopyTo(ByVal array As System.Array, ByVal index As Integer) Implements System.Collections.ICollection.CopyTo
            items.CopyTo(array, index)
        End Sub


        Public Sub AddProduct(ByVal productID As String, ByVal productName As String, ByVal ImagePath As String)
            Dim item As TrackedItem = New TrackedItem()
            item.ItemNumber = productID
            item.ItemName = productName
            item.ImagePath = ImagePath
            item.LastVisited = DateTime.Now
            item.SortingBy = SortFilter.LastVisited
            item.SortingOrder = SortOrder.Descending
            Add(item)
            'Resort ... so controls don't have to worry about it.  
            Sort()
        End Sub
        Public Sub AddCategory(ByVal categoryID As String, ByVal categoryName As String)
            Dim item As TrackedItem = New TrackedItem
            item.ItemID = categoryID
            item.ItemNumber = categoryID.ToString
            item.ItemName = categoryName
            item.LastVisited = DateTime.Now
            item.SortingBy = SortFilter.ItemCount
            item.SortingOrder = SortOrder.Descending
            Add(item)

        End Sub


        Public Sub AddShoppingCart(ByVal shoppingcartInfo As NetShopForge.Library.Cart.CartInfo)
            Dim item As TrackedItem = New TrackedItem()
            item.ItemID = shoppingcartInfo.ProductID
            item.ItemNumber = shoppingcartInfo.ProductID

            item.LastVisited = DateTime.Now
            item.SortingBy = SortFilter.ItemCount
            item.SortingOrder = SortOrder.Descending
            Add(item)
        End Sub




        ''' <summary>
        ''' Adds the specified Favorite to the FavoriteCollection collection.
        ''' </summary>
        ''' <returns>The position into which the new element was inserted.</returns>
        Public Overridable Function Add(ByVal value As TrackedItem) As Integer
            Dim bFound As Boolean = False
            Dim retVal As Integer = 0
            'check to see if this item is in the collection
            Dim index As Integer = 0
            For Each i As TrackedItem In Me
                If i.ItemNumber.Equals(value.ItemNumber) Then
                    i.ItemCount += 1
                    bFound = True
                    retVal = index
                    Exit For
                End If
                index += 1
            Next i

            If (Not bFound) Then
                If (Not Me.InsertingEvent Is Nothing) Then
                    RaiseEvent Inserting(Me, value)
                End If
                Me.IncrementVersion()
                retVal = items.Add(value)
                If (Not Me.InsertCompleteEvent Is Nothing) Then
                    RaiseEvent InsertComplete(Me, value)
                End If

                'check the limit
                If Me.Count > COLLECTION_LIMIT Then
                    Me.RemoveAt(0)
                End If
            End If
            Return retVal
        End Function


        ''' <summary>
        ''' Removes the specified TrackedItem from the TrackedItemCollection collection.
        ''' </summary>
        Public Overridable Sub Remove(ByVal value As TrackedItem)
            If (Not Me.RemovingEvent Is Nothing) Then
                RaiseEvent Removing(Me, value)
            End If
            Me.IncrementVersion()
            items.Remove(value)
            If (Not Me.RemoveCompleteEvent Is Nothing) Then
                RaiseEvent RemoveComplete(Me, value)
            End If
        End Sub

        ''' <summary>
        ''' Removes the TrackedItem object at the specified index from the TrackedItemCollection collection.
        ''' </summary>
        Public Overridable Sub RemoveAt(ByVal index As Integer)
            Dim obj As Object = items(index)
            Me.Remove((CType(obj, TrackedItem)))
        End Sub

        ''' <summary>
        ''' Inserts a TrackedItem object into the TrackedItemCollection collection at the specified position.
        ''' </summary>
        ''' <param name='index'>The zero-based index at which value should be inserted.</param>
        ''' <param name='value'>The TrackedItem to insert into the TrackedItemCollection collection.</param>
        Public Overridable Sub Insert(ByVal index As Integer, ByVal value As TrackedItem)
            If (Not Me.InsertingEvent Is Nothing) Then
                RaiseEvent Inserting(Me, value)
            End If
            Me.IncrementVersion()
            items.Insert(index, value)
            If (Not Me.InsertCompleteEvent Is Nothing) Then
                RaiseEvent InsertComplete(Me, value)
            End If
        End Sub

        ''' <summary>
        ''' determines whether the TrackedItemCollection collection contains a specific TrackedItem object.
        ''' </summary>
        ''' <param name='value'>The TrackedItem object to locate in the TrackedItemCollection collection.</param>
        ''' <returns>True if the System.Object is found in the collection; otherwise, False.</returns>
        Public Overridable Function Contains(ByVal value As TrackedItem) As Boolean
            Return items.Contains(value)
        End Function

        ''' <summary>
        ''' Removes all items from the collection.
        ''' </summary>
        Public Overridable Sub Clear()
            If (Not Me.ClearingEvent Is Nothing) Then
                RaiseEvent Clearing(Me)
            End If
            Me.IncrementVersion()
            items.Clear()
            If (Not Me.ClearCompleteEvent Is Nothing) Then
                RaiseEvent ClearComplete(Me)
            End If
        End Sub

        ''' <summary>
        ''' Creates a shallow copy of the collection.
        ''' </summary>
        Public Overridable Function Clone() As TrackedItemCollection
            Dim obj As TrackedItemCollection = New TrackedItemCollection()
            obj.items = (CType(items.Clone(), System.Collections.ArrayList))
            obj.listVersion = Me.listVersion
            Return obj
        End Function

        ''' <summary>
        ''' Defines the type of event that is called whenever a new object is inserted into or removed from the TrackedItemCollection collection.
        ''' </summary>
        Public Delegate Sub TrackedItemCollectionChangedEventHandler(ByVal sender As Object, ByVal value As TrackedItem)

        ''' <summary>
        ''' Defines the type of event that is called whenever the collection is cleared.
        ''' </summary>
        Public Delegate Sub TrackedItemCollectionClearedEventHandler(ByVal sender As Object)



    End Class
End Namespace





