﻿using System;
using System.Globalization;
using System.IO;
using System.Runtime.InteropServices;
using System.Runtime.Serialization.Formatters.Binary;
using System.Text;

namespace NetworkPortForwarding
{
    public static class ConvertHelper
    {
        #region 去掉字符串中所有空格
        /// <summary>
        /// 去掉字符串中所有空格
        /// </summary>
        /// <param name="str"></param>
        /// <returns></returns>
        public static string ReMoveBlank(string str)
        {
            string ret = "";
            if (str != null)
            {
                // ReSharper disable once GenericEnumeratorNotDisposed
                CharEnumerator cEnumerator = str.GetEnumerator();
                while (cEnumerator.MoveNext())
                {
                    var array = Encoding.ASCII.GetBytes(cEnumerator.Current.ToString());
                    int asciicode = array[0];
                    if (asciicode != 32)
                    {
                        ret += cEnumerator.Current.ToString();
                    }
                }
            }

            return ret;
        }

        /// <summary>
        /// 删除字符串中的指定字符
        /// </summary>
        /// <param name="str"></param>
        /// <param name="desStr"></param>
        /// <returns></returns>
        public static string DeleteChar(string str, char desStr)
        {
            string temp = str;
            string ret = "";
            foreach (char c in temp)
            {
                if (c != Convert.ToChar(desStr))
                {
                    ret += c;
                }
            }
            return ret;
        }
        #endregion

        #region 转换为int类型
        /// <summary>
        /// 将字符串转换为INT
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static int ToInt(string value)
        {
            value = ReMoveBlank(value);
            if (string.IsNullOrEmpty(value)) return 0;

            decimal result;
            try
            {
                decimal.TryParse(value, out result);
            }
            catch
            {
                throw (new Exception());
            }
            return Convert.ToInt32(result);
        }

        /// <summary>
        /// 将十进制数字转换为等效的32位整数
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static int ToInt(decimal value)
        {
            return Convert.ToInt32(value);
        }

        /// <summary>
        /// 将单精度浮点数转换为等效的32位整数
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static int ToInt(float value)
        {
            return Convert.ToInt32(value);
        }

        /// <summary>
        /// 将双精度浮点数转换为等效的32位整数
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static int ToInt(double value)
        {
            return Convert.ToInt32(value);
        }
        /// <summary>
        /// 转整形
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static int ToInt(int value)
        {
            return Convert.ToInt32(value);
        }
        /// <summary>
        /// 将object转换为等效的32位整数
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static int ToInt(object value)
        {
            if (null == value || string.IsNullOrEmpty(value.ToString())) return 0;
            decimal result;
            try
            {
                decimal.TryParse(value.ToString(), out result);
            }
            catch
            {
                throw (new Exception());
            }
            return Convert.ToInt32(result);
        }
        /// <summary>
        /// 字节数组中截取长度转Int类型
        /// </summary>
        /// <param name="bytes">字节数据</param>
        /// <param name="index">数组偏移位</param>
        /// <param name="len">长度</param>
        /// <returns></returns>
        public static int ToInt(byte[] bytes, int index, int len)
        {
            byte[] dst = new byte[len];
            Buffer.BlockCopy(bytes, index, dst, 0, len);
            string floatStr = BytesToString(dst, Encoding.ASCII);
            var ret = ToInt(floatStr);
            return ret;
        }
        /// <summary>
        /// 字节数组中截取长度转Int类型
        /// </summary>
        /// <param name="bytes">字节数组</param>
        /// <param name="index">数组偏移位</param>
        /// <param name="len">长度</param>
        /// <param name="encoding">编码方式</param>
        /// <returns></returns>
        public static int ToInt(byte[] bytes, int index, int len, Encoding encoding)
        {
            byte[] dst = new byte[len];
            Buffer.BlockCopy(bytes, index, dst, 0, len);
            string floatStr = BytesToString(dst, encoding);
            var ret = ToInt(floatStr);
            return ret;
        }
        /// <summary>
        /// 将不存在的值转换为0
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static int ToInt(DBNull value)
        {
            return 0;
        }

        /// <summary>
        /// 将int?转换为等效的32位整数short?
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static int ToInt(int? value)
        {
            if (!value.HasValue) return 0;

            decimal result;
            try
            {
                decimal.TryParse(value.Value.ToString(), out result);
            }
            catch
            {
                throw (new Exception());
            }
            return Convert.ToInt32(result);
        }

        /// <summary>
        /// 将short?转换为等效的32位整数
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static int ToInt(short? value)
        {
            if (!value.HasValue) return 0;
            decimal result;
            try
            {
                decimal.TryParse(value.Value.ToString(), out result);
            }
            catch
            {
                throw (new Exception());
            }
            return Convert.ToInt32(result);
        }
        /// <summary>
        /// 将bool?转换为等效的32位整数
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static int ToInt(bool? value)
        {
            if (!value.HasValue) return 0;
            int result = 0;
            try
            {
                if (value.Value)
                {
                    result = 1;
                }
            }
            catch
            {
                throw (new Exception());
            }
            return result;
        }
        #endregion

        #region 转换为float类型
        /// <summary>
        /// string类型转float类型
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static float ToFloat(string value)
        {
            if (value != null)
            {
                string floatString = ReMoveBlank(value);
                if (float.TryParse(floatString, out var ret))
                    return ret;
                else return 0.0f;
            }
            else return 0.0f;
        }

        /// <summary>
        /// object转Float类型
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static float ToFloat(object value)
        {
            string floatString = ReMoveBlank(value.ToString());
            if (floatString != null)
            {
                if (float.TryParse(floatString, out var ret))
                    return ret;
                else return 0.0f;
            }
            else return 0.0f;
        }

        /// <summary>
        /// 字节数组中截取长度转Float类型
        /// </summary>
        /// <param name="bytes">字节数据</param>
        /// <param name="index">数组偏移位</param>
        /// <param name="len">长度</param>
        /// <returns></returns>
        public static float ToFloat(byte[] bytes, int index, int len)
        {
            var ret = ToFloat(bytes, index, len, Encoding.ASCII);
            return ret;
        }

        /// <summary>
        /// 字节数组中截取长度转Float类型
        /// </summary>
        /// <param name="bytes">字节数据</param>
        /// <param name="index">数组偏移位</param>
        /// <param name="len">长度</param>
        /// <param name="encoding">字符串编码方式</param>
        /// <returns></returns>
        public static float ToFloat(byte[] bytes, int index, int len, Encoding encoding)
        {
            byte[] dst = new byte[len];
            Buffer.BlockCopy(bytes, index, dst, 0, len);
            string floatStr = BytesToString(dst, encoding);
            var ret = ToFloat(floatStr);
            return ret;
        }

        /// <summary>
        /// decimal?转Float类型
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static float ToFloat(decimal? value)
        {
            if (!value.HasValue) return 0.0f;

            if (float.TryParse(value.Value.ToString(CultureInfo.InvariantCulture), out var ret))
                return ret;
            else return 0.0f;

        }
        #endregion

        #region 转换为DateTime类型,世界时间与北京时间转换
        /// <summary>
        /// 将字符串转换成日期时间类型
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static DateTime ToDateTime(string value)
        {
            //value = ReMoveBlank(value);
            if (string.IsNullOrEmpty(value))
                return DateTime.Parse("1899-12-30");

            DateTime ret;
            try
            {
                DateTime.TryParse(value, out ret);
            }
            catch
            {
                throw (new Exception());
            }
            return Convert.ToDateTime(ret);
        }

        /// <summary>
        /// 当前时间转格林尼治时间
        /// </summary>
        /// <param name="dateTime"></param>
        /// <returns></returns>
        public static long DateTimeToTime_t(DateTime dateTime)
        {
            DateTime dt1 = new DateTime(1970, 1, 1, 0, 0, 0);
            TimeSpan ts = dateTime - dt1;
            var timeT = ts.Ticks / 10000000 - 28800;
            return timeT;
        }

        /// <summary>
        /// 格林尼治时间转当前时间
        /// </summary>
        /// <param name="dateTime"></param>
        /// <returns></returns>
        public static DateTime Time_tToDateTime(long dateTime)
        {
            //time_t是世界时间， 比 本地时间 少8小时(即28800秒)
            double seconds = 1259666013 + 28800;
            double secs = Convert.ToDouble(seconds);
            DateTime dt = new DateTime(1970, 1, 1, 0, 0, 0, DateTimeKind.Unspecified).AddSeconds(secs);
            return dt;
        }
        #endregion

        #region 转换为bool类型
        /// <summary>
        /// 将字符串转成bool类型
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static bool ToBool(string value)
        {
            value = ReMoveBlank(value);
            if (string.IsNullOrEmpty(value)) return false;
            if (value.Equals("1")) return true;
            else if (value.Equals("0")) return false;
            else
            {
                bool ret;
                try
                {
                    bool.TryParse(value, out ret);
                }
                catch
                {
                    throw (new Exception());
                }
                return Convert.ToBoolean(ret);
            }
        }
        /// <summary>
        /// 整形转bool类型
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static bool ToBool(int value)
        {
            return ToBool(value.ToString());
        }
        /// <summary>
        /// 将字符串转成bool类型
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static bool ToBool(bool? value)
        {
            if (!value.HasValue) return false;

            bool ret;
            try
            {
                bool.TryParse(value.Value.ToString(), out ret);
            }
            catch
            {
                throw (new Exception());
            }
            return Convert.ToBoolean(ret);
        }
        #endregion

        #region 数据结构与Byte数组之间转换
        /// <summary>
        /// 将object对象转换为Byte数组
        /// </summary>
        /// <param name="obj">源对象</param>
        /// <returns>Byte数组</returns>
        public static byte[] ObjectToBytes(object obj)
        {
            MemoryStream ms = new MemoryStream();
            BinaryFormatter bf = new BinaryFormatter();
            bf.Serialize(ms, obj);
            ms.Close();
            return ms.ToArray();
        }

        /// <summary>
        /// 将Byte数组转换为object对象
        /// </summary>
        /// <param name="bytes">Byte数组</param>
        /// <returns>object对象</returns>
        public static object BytesToObject(byte[] bytes)
        {
            MemoryStream ms = new MemoryStream(bytes, 0, bytes.Length);
            BinaryFormatter bf = new BinaryFormatter();
            return bf.Deserialize(ms);
        }

        /// <summary>
        /// 非托管数据结构转换为Byte数组
        /// </summary>
        /// <param name="obj">非托管数据结构</param>
        /// <returns>Byte数组</returns>
        public static byte[] StructToBytes(object obj)
        {
            //得到结构体大小
            int size = Marshal.SizeOf(obj);
            //创建Byte数组
            byte[] bytes = new byte[size];
            //分配结构体大小的内存空间
            IntPtr ptr = Marshal.AllocHGlobal(size);
            //将结构体拷贝到分配好的内存空间
            Marshal.StructureToPtr(obj, ptr, false);
            //从内存空间拷贝到Byte数组
            Marshal.Copy(ptr, bytes, 0, size);
            //释放内存空间
            Marshal.FreeHGlobal(ptr);
            //返回byte数组
            return bytes;
        }

        /// <summary>
        /// Byte数组转换为非托管数据结构
        /// </summary>
        /// <param name="bytes">Byte数组</param>
        /// <param name="type">结构类型</param>
        /// <returns>非托管数据结构</returns>
        public static object BytesToStruct(byte[] bytes, Type type)
        {
            //得到结构体的大小
            int size = Marshal.SizeOf(type);
            //byte数组长度小于结构体大小
            if (size > bytes.Length)
            {
                return null;
            }
            //分配结构体大小的内存空间
            IntPtr ptr = Marshal.AllocHGlobal(size);
            //将Byte数组拷贝到内存空间
            Marshal.Copy(bytes, 0, ptr, size);

            //将内存空间转换为目标结构体
            object obj = Marshal.PtrToStructure(ptr, type);
            //释放内存空间
            Marshal.FreeHGlobal(ptr);
            //返回结构
            return obj;
        }
        #endregion

        /// <summary>
        /// 字节数组转整形
        /// </summary>
        /// <param name="b"></param>
        /// <returns></returns>
        public static int BytesToInt(byte[] b)
        {
            int intValue = 0;
            try
            {
                for (int i = 0; i < b.Length; i++)
                {
                    intValue += (b[i] & 0xFF) << (8 * i);
                }
            }
            catch
            {
                // ignored
            }

            return intValue;
        }

        /// <summary>
        /// 将byte[]转换成int
        /// </summary>
        /// <param name="data">需要转换成整数的byte数组</param>
        public static int BytesToInt32(byte[] data)
        {
            //如果传入的字节数组长度小于4,则返回0
            if (data.Length < 4)
            {
                return 0;
            }

            //定义要返回的整数
            int num = 0;

            //如果传入的字节数组长度大于4,需要进行处理
            if (data.Length >= 4)
            {
                //创建一个临时缓冲区
                byte[] tempBuffer = new byte[4];

                //将传入的字节数组的前4个字节复制到临时缓冲区
                Buffer.BlockCopy(data, 0, tempBuffer, 0, 4);

                //将临时缓冲区的值转换成整数，并赋给num
                num = BitConverter.ToInt32(tempBuffer, 0);
            }

            //返回整数
            return num;
        }

        /// <summary>
        /// 将整形转成字节数组
        /// </summary>
        /// <returns></returns>
        public static byte[] IntToBytes(int number)
        {
            byte[] b = new byte[4];
            try
            {
                for (int i = 0; i < 4; i++)
                {
                    b[i] = (byte)(number >> 8 * i & 0xFF);
                }
            }
            catch
            {
                // ignored
            }

            return b;
        }

        /// <summary>
        /// 字节数组转短整形
        /// </summary>
        /// <param name="b"></param>
        /// <returns></returns>
        public static short BytesToShort(byte[] b)
        {
            short s = 0;
            try
            {
                short s0 = (short)(b[0] & 0xff);// 最低位 
                short s1 = (short)(b[1] & 0xff);
                s1 <<= 8;
                s = (short)(s0 | s1);
            }
            catch
            {
                // ignored
            }

            return s;
        }

        /// <summary>
        /// 短整形转字节数组
        /// </summary>
        /// <param name="number"></param>
        /// <returns></returns>
        public static byte[] ShortToBytes(short number)
        {
            int temp = number;
            byte[] b = new byte[2];
            try
            {
                for (int i = 0; i < b.Length; i++)
                {
                    b[i] = (byte)(temp & 0xff);//将最低位保存在最低位
                    temp = (byte)(temp >> 8); // 向右移8位 
                }
            }
            catch
            {
                // ignored
            }

            return b;
        }

        /// <summary>
        /// 将小端编码的二进制数组转换为数字整型
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        public static int LittleEndianInt32(byte[] data)
        {
            return BitConverter.ToInt32(data, 0);
        }

        /// <summary>
        /// 将整型转换为小端编码的二进制数组
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        public static byte[] Int32LittleEndian(int data)
        {
            return BitConverter.GetBytes(data);
        }
        /// <summary>
        /// 将大端编码的二进制数组转换为数字整型
        /// </summary>
        /// <param name="data">大端编码的二进制数组</param>
        /// <returns>数字整型</returns>
        public static int BigEndianInt32(byte[] data)
        {
            byte[] temp = (byte[])data.Clone();
            Array.Reverse(temp);
            return BitConverter.ToInt32(temp, 0);
        }
        /// <summary>
        /// 将整型转换为大端编码的二进制数组
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        public static byte[] Int32BigEndian(int data)
        {
            //int n = 439041118; // 十六进制为 1A2B3C5E
            //byte[] ba = System.BitConverter.GetBytes(n);
            //Console.WriteLine(string.Format("小端输出 为 5E-3C-2B-1A：{0}", BitConverter.ToString(ba)));
            //Array.Reverse(ba); // 反转
            //Console.WriteLine(string.Format("大端输出 为 1A-2B-3C-5E：{0}", BitConverter.ToString(ba)));
            byte[] ba = BitConverter.GetBytes(data);
            Array.Reverse(ba); // 反转
            return ba;
        }
        /// <summary>
        /// 转换两个字节为数字
        /// </summary>
        /// <param name="height">高位（前）</param>
        /// <param name="low">低位（后）</param>
        /// <returns></returns>
        public static int ConvertBit2Short(byte height, byte low)
        {
            int nResult = height;
            nResult <<= 8;
            nResult += low;
            return nResult;
        }

        /// <summary>
        /// long类型转成byte数组
        /// </summary>
        /// <param name="number"></param>
        /// <returns></returns>
        public static byte[] LongToBytes(long number)
        {
            long temp = number;
            byte[] b = new byte[8];
            try
            {
                for (int i = 0; i < b.Length; i++)
                {
                    b[i] = (byte)(temp & 0xff);//将最低位保存在最低位
                    temp = (byte)(temp >> 8); // 向右移8位 
                }
            }
            catch
            {
                // ignored
            }

            return b;
        }

        /// <summary>
        /// byte数组转成long
        /// </summary>
        /// <param name="b"></param>
        /// <returns></returns>
        public static long BytesToLong(byte[] b)
        {
            long s = 0;
            try
            {
                long s0 = b[0] & 0xff;// 最低位 
                long s1 = b[1] & 0xff;
                long s2 = b[2] & 0xff;
                long s3 = b[3] & 0xff;
                long s4 = b[4] & 0xff;// 最低位 
                long s5 = b[5] & 0xff;
                long s6 = b[6] & 0xff;
                long s7 = b[7] & 0xff;

                // s0不变 
                s1 <<= 8;
                s2 <<= 16;
                s3 <<= 24;
                s4 <<= 8 * 4;
                s5 <<= 8 * 5;
                s6 <<= 8 * 6;
                s7 <<= 8 * 7;
                s = s0 | s1 | s2 | s3 | s4 | s5 | s6 | s7;
            }
            catch
            {
                // ignored
            }

            return s;
        }

        /// <summary>
        /// 读取文件，并转换为字节流
        /// </summary>
        /// <param name="filename">文件路径及文件名称</param>
        /// <returns></returns>
        public static byte[] FileToBytes(string filename)
        {
            FileStream fs = new FileStream(filename, FileMode.Open, FileAccess.Read);
            byte[] infbytes = new byte[(int)fs.Length];
            fs.Read(infbytes, 0, infbytes.Length);
            fs.Close();
            return infbytes;
        }

        /// <summary>
        /// 将字节流写入文件
        /// </summary>
        /// <param name="filename"></param>
        /// <param name="infbytes"></param>
        public static void BytesToFile(string filename, byte[] infbytes)
        {
            FileStream fs = new FileStream(filename, FileMode.Create, FileAccess.Write);
            fs.Write(infbytes, 0, infbytes.Length);
            fs.Close();
        }

        /// <summary>
        /// 序列化
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public static byte[] RawSerialize(object obj)
        {
            int rawsize = Marshal.SizeOf(obj);
            IntPtr buffer = Marshal.AllocHGlobal(rawsize);
            Marshal.StructureToPtr(obj, buffer, false);
            byte[] rawdatas = new byte[rawsize];
            Marshal.Copy(buffer, rawdatas, 0, rawsize);
            Marshal.FreeHGlobal(buffer);
            return rawdatas;
        }

        /// <summary>
        /// 反序列化
        /// </summary>
        /// <param name="rawdatas"></param>
        /// <param name="strcutType"></param>
        /// <returns></returns>
        public static object RawDeserialize(byte[] rawdatas, Type strcutType)
        {
            Type anytype = strcutType;
            int rawsize = Marshal.SizeOf(anytype);
            if (rawsize > rawdatas.Length) return null;
            IntPtr buffer = Marshal.AllocHGlobal(rawsize);
            Marshal.Copy(rawdatas, 0, buffer, rawsize);
            object retobj = Marshal.PtrToStructure(buffer, anytype);
            Marshal.FreeHGlobal(buffer);
            return retobj;
        }

        /// <summary> 
        /// 字节数组转16进制字符串 
        /// </summary> 
        /// <param name="bytes"></param> 
        /// <returns></returns> 
        public static string BytesToHex(byte[] bytes)
        {
            string returnStr = "";
            if (bytes != null)
            {
                for (int i = 0; i < bytes.Length; i++)
                {
                    returnStr += bytes[i].ToString("X2") + " ";
                }
            }
            return returnStr.Trim();
        }

        /// <summary> 
        /// 从汉字转换到16进制 
        /// </summary> 
        /// <param name="s"></param>
        /// <param name="encode"></param>
        /// <param name="fenge">是否每字符用逗号分隔</param> 
        /// <returns></returns> 
        public static string StrToHex(string s, string encode, bool fenge)
        {
            if ((s.Length % 2) != 0)
            {
                s += " ";//空格 
                //throw new ArgumentException("s is not valid chinese string!"); 
            }
            Encoding chs = Encoding.GetEncoding(encode);
            byte[] bytes = chs.GetBytes(s);
            string str = "";
            for (int i = 0; i < bytes.Length; i++)
            {
                str += $"{bytes[i]:X}";
                if (fenge && (i != bytes.Length - 1))
                {
                    str += ",";
                }
            }
            return str.ToLower();
        }

        /// <summary> 
        ///  从16进制转换成汉字 
        ///  </summary> 
        ///  <param name="hex"></param>
        /// <param name="encode"></param>
        /// <returns></returns> 
        public static string HexToStr(string hex, string encode)
        {
            if (hex == null)
                throw new ArgumentNullException(nameof(hex));
            hex = hex.Replace(",", "");
            hex = hex.Replace("\n", "");
            hex = hex.Replace("\\", "");
            hex = hex.Replace(" ", "");
            if (hex.Length % 2 != 0)
            {
                hex += "20";//空格 
            }
            // 需要将 hex 转换成 byte 数组。 
            byte[] bytes = new byte[hex.Length / 2];
            for (int i = 0; i < bytes.Length; i++)
            {
                try
                {
                    // 每两个字符是一个 byte。 
                    bytes[i] = byte.Parse(hex.Substring(i * 2, 2), NumberStyles.HexNumber);
                }
                catch
                {
                    // Rethrow an exception with custom message. 
                    throw new ArgumentException("hex is not a valid hex number!", nameof(hex));
                }
            }
            Encoding chs = Encoding.GetEncoding(encode);
            return chs.GetString(bytes);
        }

        /// <summary>
        /// 转换字节数组为16进制形式
        /// </summary>
        /// <param name="bytes">字节数组</param>
        /// <returns>16进制表示的字符</returns>
        public static string ConvertBit2Hex(byte[] bytes)
        {
            return BitConverter.ToString(bytes, 0).Replace("-", string.Empty);
        }
        /// <summary>
        /// 将16进制的字符转换为字节数组
        /// </summary>
        /// <param name="hex">16进制字符</param>
        /// <returns>字节数组</returns>
        public static byte[] ConvertHex2Bit(string hex)
        {
            var inputByteArray = new byte[hex.Length / 2];
            for (var x = 0; x < inputByteArray.Length; x++)
            {
                var i = Convert.ToInt32(hex.Substring(x * 2, 2), 16);
                inputByteArray[x] = (byte)i;
            }
            // 输出转换结果
            string str = ConvertBit2Hex(inputByteArray);
            return inputByteArray;
        }

        /// <summary>
        /// 十进制数转换成二、八、十六进制数
        /// </summary>
        /// <param name="intValue">十进制数</param>
        /// <param name="mod">进制</param>
        /// <returns></returns>
        public static string IntToHex(int intValue, int mod)
        {
            string hexValue = string.Empty;
            var temp = intValue;
            while (temp > 0)
            {
                var addValue = temp / mod;
                var modValue = temp % mod;
                char charModValue;
                if (modValue >= 10)
                {
                    charModValue = (char)(modValue + 55);
                }
                else
                {
                    charModValue = (char)(modValue + 48);
                }
                hexValue = charModValue + hexValue;
                temp = addValue;
            }
            return hexValue;
        }

        /// <summary>
        /// 非十进制数转换成十进制
        /// </summary>
        /// <param name="hexValue">非十进制数</param>
        /// <param name="mod">模</param>
        /// <returns>十进制数</returns>
        public static int HexToInt(string hexValue, int mod)
        {
            int value = 0;
            for (int i = 0; i < hexValue.Length; i++)
            {
                int asc = Convert.ToInt32(hexValue[i]);
                if (asc >= 65)
                {
                    value = value + Convert.ToInt32((asc - 55) * Math.Pow(Convert.ToDouble(mod), Convert.ToDouble(hexValue.Length - i - 1)));
                }
                else
                {
                    value = value + Convert.ToInt32((asc - 48) * Math.Pow(Convert.ToDouble(mod), Convert.ToDouble(hexValue.Length - i - 1)));
                }
            }
            return value;
        }

        public static void ShortToHex(ref byte l, ref byte h, short val)
        {
            l = (byte)(val & 0xff);//将最低位保存在最低位
            h = (byte)(val >> 8); // 向右移8位 
        }

        /// <summary>
        /// 转换为16进制字节
        /// </summary>
        /// <param name="hbyte"></param>
        /// <param name="lbyte"></param>
        /// <returns></returns>
        public static byte BytesToHex(char hbyte, char lbyte)
        {
            int hi = 0, li = 0;
            if (hbyte >= '0' && hbyte <= '9')
                hi = (hbyte - '0') * 16;
            else if (hbyte >= 'a' && hbyte <= 'f')
                hi = (hbyte - 'a' + 10) * 16;
            else if (hbyte >= 'A' && hbyte <= 'F')
                hi = (hbyte - 'A' + 10) * 16;

            if (lbyte >= '0' && lbyte <= '9')
                li = lbyte - '0';
            else if (lbyte >= 'a' && lbyte <= 'f')
                li = lbyte - 'a' + 10;
            else if (lbyte >= 'A' && lbyte <= 'F')
                li = lbyte - 'A' + 10;
            return (byte)(hi + li);
        }

        #region 补足位数，不足前补0
        /// <summary>
        /// 指定字符串的固定长度，如果字符串小于固定长度，
        /// 则在字符串的前面补足零，可设置的固定长度最大为9位
        /// </summary>
        /// <param name="text">原始字符串</param>
        /// <param name="limitedLength">字符串的固定长度</param>
        public static string RepairZero(string text, int limitedLength)
        {
            //补足0的字符串
            string temp = "";

            //补足0
            for (int i = 0; i < limitedLength - text.Length; i++)
            {
                temp += "0";
            }

            //连接text
            temp += text;

            //返回补足0的字符串
            return temp;
        }
        #endregion

        #region 各进制数间转换
        /// <summary>
        /// 实现各进制数间的转换。ConvertBase("15",10,16)表示将十进制数15转换为16进制的数。
        /// </summary>
        /// <param name="value">要转换的值,即原值</param>
        /// <param name="from">原值的进制,只能是2,8,10,16四个值。</param>
        /// <param name="to">要转换到的目标进制，只能是2,8,10,16四个值。</param>
        public static string ConvertBase(string value, int from, int to)
        {
            try
            {
                int intValue = Convert.ToInt32(value, from);  //先转成10进制
                string result = Convert.ToString(intValue, to);  //再转成目标进制
                if (to == 2)
                {
                    int resultLength = result.Length;  //获取二进制的长度
                    switch (resultLength)
                    {
                        case 7:
                            result = "0" + result;
                            break;
                        case 6:
                            result = "00" + result;
                            break;
                        case 5:
                            result = "000" + result;
                            break;
                        case 4:
                            result = "0000" + result;
                            break;
                        case 3:
                            result = "00000" + result;
                            break;
                    }
                }
                return result;
            }
            catch
            {
                return "0";
            }
        }
        #endregion

        #region 字符串string与byte[]之间转换
        /// <summary>
        /// 使用指定字符集将string转换成byte[]
        /// </summary>
        /// <param name="text">要转换的字符串</param>
        /// <param name="encoding">字符编码</param>
        public static byte[] StringToBytes(string text, Encoding encoding)
        {
            return encoding.GetBytes(text);
        }

        /// <summary>
        /// 使用指定字符集将byte[]转换成string
        /// </summary>
        /// <param name="bytes">要转换的字节数组</param>
        /// <param name="encoding">字符编码</param>
        public static string BytesToString(byte[] bytes, Encoding encoding)
        {
            return encoding.GetString(bytes);
        }
        #endregion

        #region 字符串string与char[]之间转换
        /// <summary>
        /// 字符串 转换 char数组
        /// </summary>
        /// <param name="inStr"></param>
        /// <param name="inLen"></param>
        /// <returns></returns>
        public static char[] StrToChars(string inStr, int inLen)
        {
            char[] ch = new char[inLen];
            inStr.ToCharArray().CopyTo(ch, 0);
            return ch;
        }

        /// <summary>
        /// char数组 转换 字符串
        /// </summary>
        /// <param name="inStr"></param>
        /// <returns></returns>        
        public static string CharsToStr(char[] inStr)
        {
            var outStr = new string(inStr);
            int i = outStr.IndexOf('\0', 0);
            if (i == -1)
                i = 16;
            return outStr.Substring(0, i);
        }
        #endregion

        #region Base64字符串
        /// <summary>
        /// 将字符串编码为Base64字符串
        /// </summary>
        /// <param name="str"></param>
        /// <returns></returns>
        public static string Base64Encode(string str)
        {
            byte[] barray;
            barray = Encoding.Default.GetBytes(str);
            return Convert.ToBase64String(barray);
        }

        /// <summary>
        /// 将Base64字符串解码为普通字符串
        /// </summary>
        /// <param name="str"></param>
        /// <returns></returns>
        public static string Base64Decode(string str)
        {
            byte[] barray;
            try
            {
                barray = Convert.FromBase64String(str);
                return Encoding.Default.GetString(barray);
            }
            catch
            {
                return str;
            }
        }
        #endregion

        #region 人民币小写转大写
        /// <summary> 
        /// 转换人民币大小金额 
        /// </summary> 
        /// <param name="num">金额</param> 
        /// <returns>返回大写形式</returns> 
        public static string CmycurD(decimal num)
        {
            string str1 = "零壹贰叁肆伍陆柒捌玖";            //0-9所对应的汉字 
            string str2 = "万仟佰拾亿仟佰拾万仟佰拾元角分"; //数字位所对应的汉字 
            var str5 = "";
            int i;    //循环变量 
            int j;    //num的值乘以100的字符串长度 
            string ch2 = "";    //数字位的汉字读法 
            int nzero = 0;  //用来计算连续的零值是几个 
            int temp;            //从原num值中取出的值 

            num = Math.Round(Math.Abs(num), 2, MidpointRounding.AwayFromZero);    //将num取绝对值并四舍五入取2位小数 
            var str4 = ((long)(num * 100)).ToString();
            j = str4.Length;      //找出最高位 
            if (j > 15) { return "溢出"; }
            str2 = str2.Substring(15 - j);   //取出对应位数的str2的值。如：200.55,j为5所以str2=佰拾元角分 

            //循环取出每一位需要转换的值 
            for (i = 0; i < j; i++)
            {
                var str3 = str4.Substring(i, 1);    //从原num值中取出的值 
                temp = Convert.ToInt32(str3);      //转换为数字 
                string ch1;    //数字的汉语读法 
                if (i != (j - 3) && i != (j - 7) && i != (j - 11) && i != (j - 15))
                {
                    //当所取位数不为元、万、亿、万亿上的数字时 
                    if (str3 == "0")
                    {
                        ch1 = "";
                        ch2 = "";
                        nzero = nzero + 1;
                    }
                    else
                    {
                        if (str3 != "0" && nzero != 0)
                        {
                            ch1 = "零" + str1.Substring(temp * 1, 1);
                            ch2 = str2.Substring(i, 1);
                            nzero = 0;
                        }
                        else
                        {
                            ch1 = str1.Substring(temp * 1, 1);
                            ch2 = str2.Substring(i, 1);
                            nzero = 0;
                        }
                    }
                }
                else
                {
                    //该位是万亿，亿，万，元位等关键位 
                    if (str3 != "0" && nzero != 0)
                    {
                        ch1 = "零" + str1.Substring(temp * 1, 1);
                        ch2 = str2.Substring(i, 1);
                        nzero = 0;
                    }
                    else
                    {
                        if (str3 != "0" && nzero == 0)
                        {
                            ch1 = str1.Substring(temp * 1, 1);
                            ch2 = str2.Substring(i, 1);
                            nzero = 0;
                        }
                        else
                        {
                            if (str3 == "0" && nzero >= 3)
                            {
                                ch1 = "";
                                ch2 = "";
                                nzero = nzero + 1;
                            }
                            else
                            {
                                if (j >= 11)
                                {
                                    ch1 = "";
                                    nzero = nzero + 1;
                                }
                                else
                                {
                                    ch1 = "";
                                    ch2 = str2.Substring(i, 1);
                                    nzero = nzero + 1;
                                }
                            }
                        }
                    }
                }
                if (i == (j - 11) || i == (j - 3))
                {
                    //如果该位是亿位或元位，则必须写上 
                    ch2 = str2.Substring(i, 1);
                }
                str5 = str5 + ch1 + ch2;

                if (i == j - 1 && str3 == "0")
                {
                    //最后一位（分）为0时，加上“整” 
                    str5 = str5 + '整';
                }
            }
            if (num == 0)
            {
                str5 = "零元整";
            }
            return str5;
        }


        /// <summary> 
        /// 转换人民币大小金额  (一个重载，将字符串先转换成数字在调用CmycurD)
        /// </summary>
        /// <param name="numstr">用户输入的金额，字符串形式未转成decimal</param>
        /// <returns></returns> 
        public static string CmycurD(string numstr)
        {
            try
            {
                decimal num = Convert.ToDecimal(numstr);
                return CmycurD(num);
            }
            catch
            {
                return "非数字形式！";
            }
        }
        #endregion

        #region byte[]转换为Intptr与IntPtr转byte[]
        /// <summary>
        /// byte[]转换为Intptr
        /// </summary>
        /// <param name="bytes"></param>
        /// <returns></returns>
        public static IntPtr BytesToIntptr(byte[] bytes)
        {
            int size = bytes.Length;
            IntPtr buffer = Marshal.AllocHGlobal(size);
            try
            {
                Marshal.Copy(bytes, 0, buffer, size);
                return buffer;
            }
            finally
            {
                Marshal.FreeHGlobal(buffer);
            }
        }
        /// <summary>
        /// IntPtr转byte[]
        /// </summary>
        /// <param name="buffer">IntPtr</param>
        /// <param name="dataLen">字节数据长度</param>
        /// <returns></returns>
        public static byte[] IntptrToBytes(IntPtr buffer, int dataLen)
        {
            byte[] bytes = new byte[dataLen];
            Marshal.Copy(buffer, bytes, 0, dataLen);
            return bytes;
        }
        #endregion
    }
}
