﻿using System;
using System.Collections.Concurrent;
using System.IO;
using System.Runtime.Remoting;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace NetworkPortForwarding
{
    /// <summary>
    /// 记录日志类
    /// </summary>
    public static class LogHelper
    {
        /// <summary>
        /// App运行日志的路径
        /// </summary>
        public static string LogPath = GetFileAbsolutePath("Log");
        /// <summary>
        /// 错误日志的路径
        /// </summary>
        public static string ErrorLogPath = GetFileAbsolutePath("ErrorLog");
        /// <summary>
        /// 联网日志的路径
        /// </summary>
        public static string NetLogPath = GetFileAbsolutePath("NetLog");
        /// <summary>
        /// 调试日志
        /// </summary>
        public static string DebugLogPath = GetFileAbsolutePath("DebugLog");
        /// <summary>
        /// 其他App日志的路径 AppFileLogPath
        /// </summary>
        public static string AppFileLogPath { set; get; }
        /// <summary>
        /// 其他App日志的路径
        /// </summary>
        public static string AppLogPath = GetFileAbsolutePath(string.IsNullOrEmpty(AppFileLogPath) ? "AppFileLog" : AppFileLogPath);
        /// <summary>
        /// 日志分隔文件大小 2M
        /// </summary>
        private const int LogFileSize = 2 * 1024 * 1024;
        /// <summary>
        /// 日志保存天数
        /// </summary>
        private const int KeepLogDays = 60;
        /// <summary>
        /// 线程安全，先进先出集合
        /// </summary>
        private static readonly ConcurrentQueue<Tuple<string, string>> QueueLog = new ConcurrentQueue<Tuple<string, string>>();
        /// <summary>
        /// 线程锁
        /// </summary>
        private static readonly object WriteLogLock = new object();

        public static bool IsStartWrite = false;
        /// <summary>
        /// 获取文件路径的绝对路径
        /// </summary>
        /// <param name="fileSubPath">文件子目录路径</param>
        /// <param name="fileRootPath">文件根目录路径</param>
        /// <returns></returns>
        private static string GetFileAbsolutePath(string fileSubPath, string fileRootPath = "Log")
        {
            string pathstr;
            fileRootPath = fileRootPath.Trim('\\').Trim();
            if (string.IsNullOrEmpty(fileRootPath) || fileRootPath.Trim() == "")
                pathstr = Directory.GetCurrentDirectory();
            else if (Path.IsPathRooted(fileRootPath) && (string.IsNullOrEmpty(fileSubPath) || fileSubPath.Trim() == ""))
                pathstr = Path.GetFullPath(fileRootPath);
            else pathstr = Path.GetFullPath(Path.Combine(Directory.GetCurrentDirectory(), fileRootPath, fileSubPath));
            return pathstr;
        }
        /// <summary>
        /// 去除文件的只读与隐藏属性
        /// </summary>
        /// <param name="filename"></param>
        private static void SetFileNormal(string filename)
        {
            if (!File.Exists(filename)) return;
            var fabs = File.GetAttributes(filename);
            if ((fabs & FileAttributes.ReadOnly) == FileAttributes.ReadOnly)
                fabs ^= FileAttributes.ReadOnly;
            if ((fabs & FileAttributes.Hidden) == FileAttributes.Hidden)
                fabs ^= FileAttributes.Hidden;
            if ((fabs & FileAttributes.System) == FileAttributes.System)
                fabs ^= FileAttributes.System;

            File.SetAttributes(filename, fabs);//读置文件属性
        }
        /// <summary>
        /// 强制删除一个文件
        /// </summary>
        /// <param name="filename"></param>
        private static void DeleteFile(string filename)
        {
            if (!File.Exists(filename)) return;
            SetFileNormal(filename);//去除只读属性
            File.Delete(filename);
        }

        #region 构造日志帮助类

        static LogHelper()
        {
            Task.Factory.StartNew(action: delegate
            {
                while (true)
                {
                    while (QueueLog.TryDequeue(out var tuple))
                    {
                        WriteLogFile(tuple.Item1, tuple.Item2);
                    }
                    Thread.Sleep(100);
                }
            });
        }
        #endregion

        #region 写文件
        /// <summary>
        /// 写文件
        /// </summary>
        private static void WriteLogFile(string fileType, string logInfo)
        {
            lock (WriteLogLock)
            {
                var logFilePath = LogPath;
                if (fileType.Equals("ErrorLog"))
                    logFilePath = ErrorLogPath;
                else if (fileType.Equals("NetLog"))
                    logFilePath = NetLogPath;
                else if (fileType.Equals("Log"))
                    logFilePath = LogPath;
                else if (fileType.Equals("DebugLog"))
                    logFilePath = DebugLogPath;
                else if (fileType.Equals("AppLog"))
                    logFilePath = AppLogPath;
                var logfileName = $"{logFilePath}\\{DateTime.Now:yyyyMMdd}.log";

                if (!File.Exists(logFilePath))
                {
                    Directory.CreateDirectory(logFilePath);
                }

                //bool isMove = false;
                if (File.Exists(logfileName))
                {
                    var fileinfo = new FileInfo(logfileName);
                    if (fileinfo.Length >= LogFileSize)
                    {
                        //移动原文件为新文件
                        fileinfo.CopyTo($"{logFilePath}\\{DateTime.Now:yyyyMMdd(HHmmss)}.log");
                        fileinfo.Delete();
                    }
                }

                using (var sw = new StreamWriter(logfileName, true, Encoding.GetEncoding("gb2312")))
                {
                    var writeInfo = $"--【{DateTime.Now:yyyy-MM-dd HH:mm:ss.fff}】--------------------**********--------------------\r\n{logInfo}";
                    sw.WriteLine(writeInfo);
                    sw.Flush();
                }
            }
        }
        #endregion

        /// <summary>
        /// 清理日志
        /// </summary>
        public static void ClearLog()
        {
            try
            {
                var oldlogDate = DateTime.Now.AddDays(0 - KeepLogDays);
                var oldlogName = oldlogDate.ToString("yyyy-MM-dd");
                var allLogFiles = Directory.GetFiles(LogPath);
                foreach (var f in allLogFiles)
                {
                    var filename = Path.GetFileNameWithoutExtension(f);
                    if (filename.ToLower().Trim() != "system" && string.CompareOrdinal(filename, oldlogName) <= 0)
                    {
                        DeleteFile(f);
                    }
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
            }
        }

        /// <summary>
        /// 写运行日志
        /// </summary>
        public static void WriteLog(string logInfo)
        {
            QueueLog.Enqueue(new Tuple<string, string>("Log", logInfo));
        }

        /// <summary>
        /// 写错误日志
        /// </summary>
        public static void WriteErrorLog(string logInfo)
        {
            QueueLog.Enqueue(new Tuple<string, string>("ErrorLog", logInfo));
        }
        /// <summary>
        /// 写错误日志
        /// </summary>
        /// <param name="ex"></param>
        public static void WriteErrorLog(Exception ex)
        {
            var logInfo = $"--【{DateTime.Now:yyyy-MM-dd HH:mm:ss.fff}】--------------------**********--------------------\r\n";
            if (!(ex is RemotingException remoteingex))
            {
                logInfo += $"异常描述：\r\n{ex.Message}";
                logInfo += "\r\n";
                logInfo += $"异常位置：\r\n{ex.StackTrace.Trim()}";
                logInfo += "\r\n";
                logInfo += $"异常行位置：\r\n{ex.Source}";
                logInfo += "\r\n";
            }
            else
            {
#if DEBUG
                logInfo += "\r\n";
                logInfo += "----------[Remoting异常]----------------------------------";
                logInfo += $"异常描述：\r\n{ex.Message}";
                logInfo += "\r\n";
                logInfo += $"异常位置：\r\n{ex.StackTrace.Trim()}";
                logInfo += "\r\n";
                logInfo += $"异常行位置：\r\n{ex.Source}";
                logInfo += "\r\n";
#endif
            }
            QueueLog.Enqueue(new Tuple<string, string>("ErrorLog", logInfo));
        }

        /// <summary>
        /// 写联网日志
        /// </summary>
        public static void WriteNetLog(string logInfo)
        {
            QueueLog.Enqueue(new Tuple<string, string>("NetLog", logInfo));
        }

        /// <summary>
        /// 写App日志（其他App日志的路径 AppFileLogPath）
        /// </summary>
        public static void WriteAppLog(string logInfo)
        {
            QueueLog.Enqueue(new Tuple<string, string>("AppLog", logInfo));
        }

        public static void WriteDebugLog(string logInfo)
        {
            QueueLog.Enqueue(new Tuple<string, string>("DebugLog", logInfo));
        }
    }
}
