﻿using HPSocketCS;
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace NetworkPortForwarding
{
    public partial class MainForm : Form
    {
        #region 私有变量定义
        /// <summary>
        /// 服务端
        /// </summary>
        private TcpPullServer _serverSocket;
        /// <summary>
        /// 客户端
        /// </summary>
        private TcpPullClient _clientSocket;
        /// <summary>
        /// 连接的客户端IP地址
        /// </summary>
        private string netClientAddress;
        /// <summary>
        /// 连接的客户端端口号
        /// </summary>
        private int netClientPort;
        /// <summary>
        /// 网络是否连接成功
        /// </summary>
        private bool _isClientNet;
        /// <summary>
        /// 连接的客户端
        /// </summary>
        private List<IntPtr> _clientSockeList;
        #endregion
        public MainForm()
        {
            InitializeComponent();

            _clientSockeList = new List<IntPtr>();
        }

        #region TCP服务连接
        private HandleResult ServerErrors(IntPtr connId, SocketOperation enOperation, int errorCode)
        {
            if (errorCode <= 0) throw new ArgumentOutOfRangeException(nameof(errorCode));
            _serverSocket?.Disconnect(connId);
            return HandleResult.Ok;
        }

        private HandleResult ServerSocketAccept(IntPtr connId, IntPtr pClient)
        {
            _clientSockeList.Add(connId);
            string clientIp = string.Empty;
            ushort clientPort = 0;
            _serverSocket.GetRemoteAddress(connId, ref clientIp, ref clientPort);
            ShowNetDataMsg("客户端TCP连接", 0, $"连接地址：{clientIp}，端口：{clientPort}");
            return HandleResult.Ok;
        }

        private HandleResult ServerSocketClose(IntPtr connId)
        {
            foreach (var item in _clientSockeList)
            {
                if (item == connId)
                {
                    _clientSockeList.Remove(item);
                    break;
                }
            }
            return HandleResult.Ok;
        }

        private HandleResult ServerRecvData(IntPtr connId, int length)
        {
            if (length >= 1)
            {
                IntPtr bufferPtr = Marshal.AllocHGlobal(length);
                try
                {
                    if (_serverSocket.Fetch(connId, bufferPtr, length) == FetchResult.Ok)
                    {
                        var recvBytes = ConvertHelper.IntptrToBytes(bufferPtr, length);
                        string recvString = ConvertHelper.BytesToHex(recvBytes);
                        var ip = string.Empty;
                        ushort port = 0;
                        _serverSocket.GetRemoteAddress(connId, ref ip, ref port);
                        ShowNetDataMsg($"接收，IP:{ip},端口：{port}", length, recvString);

                        if (!_isClientNet)
                        {
                            if (!string.IsNullOrEmpty(netClientAddress) && netClientPort > 0)
                            {
                                _isClientNet = _clientSocket.Connetion(netClientAddress, (ushort)netClientPort);
                            }
                        }
                        _clientSocket.GetListenAddress(ref ip, ref port);
                        _clientSocket.Send(recvBytes, length);
                        ShowNetDataMsg($"发送，IP:{ip},端口：{port}", length, recvString);
                    }
                }
                finally
                {
                    if (bufferPtr != IntPtr.Zero)
                    {
                        Marshal.FreeHGlobal(bufferPtr);
                    }
                }
            }
            return HandleResult.Ok;
        }

        private HandleResult ServerSocketShutdown()
        {
            _isClientNet = false;

            return HandleResult.Ok;
        }
        #endregion

        #region TCP/IP客户端连接
        /// <summary>
        /// 与服务端成功建立连接时，Socket 监听器将收到该通知
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <returns></returns>
        protected HandleResult OnConnect(TcpClient sender)
        {
            _isClientNet = true;

            string serverIp = string.Empty;
            ushort serverPort = 0;
            _clientSocket.GetListenAddress(ref serverIp, ref serverPort);
            ShowNetDataMsg("服务端TCP连接", 0, $"连接地址：{serverIp}，端口：{serverPort}");

            return HandleResult.Ok;
        }
        private HandleResult ClientErrors(TcpClient sender, SocketOperation enOperation, int errorCode)
        {
            if (errorCode <= 0) throw new ArgumentOutOfRangeException(nameof(errorCode));
            sender.Stop();
            return HandleResult.Ok;
        }

        private HandleResult ClientRecvData(TcpPullClient sender, int length)
        {
            if (length >= 5)
            {
                IntPtr bufferPtr = Marshal.AllocHGlobal(length);
                try
                {
                    if (sender.Fetch(bufferPtr, length) == FetchResult.Ok)
                    {
                        var recvBytes = ConvertHelper.IntptrToBytes(bufferPtr, length);
                        string recvString = ConvertHelper.BytesToHex(recvBytes);

                        var ip = string.Empty;
                        ushort port = 0;
                        sender.GetListenAddress(ref ip, ref port);
                        ShowNetDataMsg($"接收，IP:{ip},端口：{port}", length, recvString);
                        if (_clientSockeList.Count > 0)
                        {
                            foreach (var item in _clientSockeList)
                            {
                                _serverSocket.GetRemoteAddress(item, ref ip, ref port);
                                _serverSocket.Send(item, recvBytes, length);
                                ShowNetDataMsg($"发送，IP:{ip},端口：{port}", length, recvString);
                            }
                        }
                    }
                }
                finally
                {
                    if (bufferPtr != IntPtr.Zero)
                    {
                        Marshal.FreeHGlobal(bufferPtr);
                    }
                }
            }
            return HandleResult.Ok;
        }
        #endregion

        //将信息显示到UI线程的方法
        private void ShowNetDataMsg(string state, int dataLen, string dataMsg)
        {
            NetRecviceMsg.BeginInvoke((MethodInvoker)delegate
            {
                var msgList = new StringBuilder();
                msgList.Append($"{DateTime.Now:yyyy-MM-dd HH:mm:ss.fff} {state}，数据长度:{dataLen}\r\n");
                msgList.Append($"数据内容:{dataMsg}\r\n");
                NetRecviceMsg.AppendText(msgList.ToString());

                if (NetRecviceMsg.Lines.GetUpperBound(0) >= 3000)
                {
                    LogHelper.WriteLog(NetRecviceMsg.Text);
                    NetRecviceMsg.Clear();
                }
            });
        }

        #region 文本框处理
        /// <summary>
        /// 只允许键盘输入数字
        /// 不包括“-”号
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        protected void IntBackspace_KeyPress(object sender, KeyPressEventArgs e)
        {
            string pa = "[0-9]|\b"; //匹配数字与退格符
            Regex r = new Regex(pa);
            Match m = r.Match(e.KeyChar.ToString());
            if (m.Success)
                e.Handled = false;
            else
            {
                e.Handled = true;
            }
        }
        /// <summary>
        /// 只允许键盘输入数字
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        protected void KeyBoardInt_KeyPress(object sender, KeyPressEventArgs e)
        {
            e.Handled = true;
            if ((e.KeyChar >= '0' && e.KeyChar <= '9') || (e.KeyChar == (char)8) || (e.KeyChar == '-'))
                e.Handled = false;
        }

        /// <summary>
        /// 允许键盘输入小数
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        /// <param name="wei">小数位数</param>
        protected void KeyBoardFloat_KeyPress(object sender, KeyPressEventArgs e, int wei)
        {
            if (char.IsNumber(e.KeyChar) || e.KeyChar == '.' || e.KeyChar == (char)Keys.Back ||
                e.KeyChar == (char)Keys.Delete)
            {
                e.Handled = false; //让操作生效  
                int j = 0;
                int k = 0;
                bool flag = false;
                if (((TextBox)sender).Text.Length == 0)
                {
                    if (e.KeyChar == '.')
                    {
                        e.Handled = true; //让操作失效  
                    }
                }

                int pPos = 0; //小数点位置
                for (int i = 0; i < ((TextBox)sender).Text.Length; i++)
                {
                    if (((TextBox)sender).Text[i] == '.')
                    {
                        j++;
                        pPos = ((TextBox)sender).Text.IndexOf('.'); //获取当前小数点位置
                        flag = true;
                    }

                    if (flag)
                    {
                        if (char.IsNumber(((TextBox)sender).Text[i]) &&
                            (e.KeyChar != (char)Keys.Back || e.KeyChar == (char)Keys.Delete))
                        {
                            k = ((TextBox)sender).Text.Length - pPos;
                        }
                    }

                    if (j >= 1)
                    {
                        if (e.KeyChar == '.')
                        {
                            e.Handled = true; //让操作失效  
                        }
                    }

                    if (k > wei)
                    {
                        if (char.IsNumber(((TextBox)sender).Text[i]) && e.KeyChar != (char)Keys.Back)
                        {
                            if (((TextBox)sender).Text.Length - ((TextBox)sender).SelectionStart < wei + 1)
                            {
                                if (((TextBox)sender).SelectedText != ((TextBox)sender).Text)
                                {
                                    e.Handled = true; ////让操作失效  
                                }
                            }
                        }
                    }
                }
            }
            else
            {
                e.Handled = true;
            }
        }

        /// <summary>
        /// 判断文本框输入的IP地址是否合法
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        protected void TextBox_Leave(object sender, EventArgs e)
        {
            var regex = new Regex("^[0-9]{1,3}.[0-9]{1,3}.[0-9]{1,3}.[0-9]{1,3}$");
            var blnTest = regex.IsMatch(((TextBox)sender).Text.Trim());
            if (blnTest)
            {
                string[] strTemp = ((TextBox)sender).Text.Trim().Split('.');
                for (int j = 0; j < strTemp.Length; j++)
                {
                    if (Convert.ToInt32(strTemp[j]) > 255)
                    { //大于255则提示，不符合IP格式 
                        MessageBox.Show(this, @"存在不符合IP地址格式，请重新输入！", @"提示", MessageBoxButtons.OK, MessageBoxIcon.Question);
                        if (((TextBox)sender).CanFocus)
                            ((TextBox)sender).Focus();
                        return;
                    }
                }
            }
            else
            {
                //输入非数字则提示，不符合IP格式 
                MessageBox.Show(this, @"存在不符合IP地址格式，请重新输入！", @"提示", MessageBoxButtons.OK, MessageBoxIcon.Question);
                if (((TextBox)sender).CanFocus)
                    ((TextBox)sender).Focus();
            }
        }

        /// <summary>
        /// 光标始终放到最后位置
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        protected void TextBox_OnTextChanged(object sender, EventArgs e)
        {
            if (!(sender is TextBox textBox)) return;
            var str = textBox.Text;
            var len = Encoding.Default.GetByteCount(str);
            var maxlen = textBox.MaxLength;
            var bb = Encoding.Default.GetBytes(str);//得到输入的字符串的数组
            if (len <= maxlen) return;
            textBox.Text = Encoding.Default.GetString(bb, 0, maxlen);//将截断后的字节数组转换成字符串
            textBox.SelectionStart = maxlen;//将光标放置在输入文字的最后
        }

        private void ServerPort_KeyPress(object sender, KeyPressEventArgs e)
        {
            KeyBoardInt_KeyPress(sender, e);
        }

        private void ClientIP_Leave(object sender, EventArgs e)
        {
            TextBox_Leave(sender, e);
        }

        private void ClearMsg_Click(object sender, EventArgs e)
        {
            if (!string.IsNullOrEmpty(NetRecviceMsg.Text))
                LogHelper.WriteLog(NetRecviceMsg.Text);
            NetRecviceMsg.Clear();
        }

        private void StartBtn_Click(object sender, EventArgs e)
        {
            netClientAddress = ClientIP.Text.Trim();
            netClientPort = ConvertHelper.ToInt(ConvertHelper.ReMoveBlank(ClientPort.Text));
            var netServerPort = ConvertHelper.ToInt(ConvertHelper.ReMoveBlank(ServerPort.Text));
            if (StartBtn.Text.Equals("启 动"))
            {
                _clientSocket = new TcpPullClient();
                _clientSocket.OnConnect += OnConnect;
                //接收到数据
                _clientSocket.OnReceive += ClientRecvData;
                //客户端已连出错
                _clientSocket.OnError += ClientErrors;

                if (!string.IsNullOrEmpty(netClientAddress) && netClientPort > 0)
                {
                    _isClientNet = _clientSocket.Connetion(netClientAddress, (ushort)netClientPort);
                }

                _serverSocket = new TcpPullServer
                {
                    IpAddress = "0.0.0.0",
                    Port = (ushort)netServerPort
                };
                //连接出错
                _serverSocket.OnError += ServerErrors;
                //客户端连接请求被接受后触发
                _serverSocket.OnAccept += ServerSocketAccept;
                //连接关闭后触发（服务端的连接通常是多个，只要某一个连接关闭了都会触发）
                _serverSocket.OnClose += ServerSocketClose;
                //接收到数据
                _serverSocket.OnReceive += ServerRecvData;
                //组件停止后触发
                _serverSocket.OnShutdown += ServerSocketShutdown;
                _serverSocket.Start();
                StartBtn.Text = "停 止";
            }
            else
            {
                if (_clientSocket != null)
                    _clientSocket.Stop();
                if (_serverSocket != null)
                    _serverSocket.Stop();
                StartBtn.Text = "启 动";
            }
        }
        #endregion

        readonly Sunisoft.IrisSkin.SkinEngine SkinEngine = new Sunisoft.IrisSkin.SkinEngine();
        private void MainForm_Load(object sender, EventArgs e)
        {
            //加载皮肤
            SkinEngine.SkinFile = Application.StartupPath + "//SteelBlack.ssk";
            SkinEngine.Active = true;
        }
    }
}
