function Vector(x, y) {
    // constructors
    // Vector(x, y)         -> {x: x, y: y}
    // Vector()             -> {x: 0, y: 0}
    // Vector(vector)       -> {x: vector.x, y: vector.y}
    // Vector({x: x, y: y}) -> {x: x, y: y}
    if (! (this instanceof Vector)) {
        return new Vector(x, y);
    }
    if (typeof x === "object") {
        this.x = x.x;
        this.y = x.y;
    } else if (x === undefined || y === undefined) {
        this.x = 0;
        this.y = 0;
    } else {
        this.x = x;
        this.y = y;
    }
}

Vector.prototype = {
    get 0() {
        return this.x;
    },

    get 1() {
        return this.y;
    }
};

Vector.prototype.add = function (vector2) {
    this.x += vector2.x;
    this.y += vector2.y;
    return this;
};

Vector.prototype.subtract = function (vector2) {
    this.x -= vector2.x;
    this.y -= vector2.y;
    return this;
};

Vector.prototype.multiply = function (vector2) {
    this.x *= vector2.x;
    this.y *= vector2.y;
    return this;
};

Vector.prototype.divide = function (vector2) {
    this.x /= vector2.x;
    this.y /= vector2.y;
    return this;
};

Vector.prototype.magnitude = function () {
    return Math.sqrt(Math.pow(this.x, 2) + Math.pow(this.y, 2));
};

Vector.prototype.normalized = function () {
    let mag = this.magnitude();
    if (mag === 0) {
        return Vector(0, 0);
    }
    return Vector(this.x / mag, this.y / mag);
};

Vector.prototype.distanceBetween = function (vector2) {
    return Math.sqrt(Math.pow(this.x - vector2.x, 2) + Math.pow(this.y - vector2.y, 2));
};

Vector.prototype.angleBetween = function (vector2) {
    // Math.atan2(endy - vec.y, endx - vec.x)
    return Math.atan2(vector2.y - this.y, vector2.x - this.x);
};

Vector.prototype.dot = function (vector2) {
    return this.x * vector2.x + this.y * vector2.y;
};

Vector.prototype.scale = function (scalar) {
    this.x *= scalar;
    this.y *= scalar;
    return this;
};

Vector.prototype.heading = function () {
    return Math.atan2(this.y, this.x);
};

Vector.prototype.lengthSq =  function () {
    return this.dot(this);
};

Vector.prototype.visualize = function (ctx, origin, exaggerate) {
    ctx.beginPath();
    ctx.moveTo(origin.x, origin.y);
    let scale = exaggerate? 5 : 1;
    ctx.lineTo(origin.x + (this.x * scale), origin.y + (this.y * scale));
    ctx.stroke();
    return this;
};

