<?php

namespace shangfan\model\base;

use framework\Request;
use framework\Exception;
use shangfan\model\base\MemberUser;
use shangfan\model\base\OrderGoods;
use shangfan\model\base\MemberAddress;
use shangfan\model\base\MemberPayRecord;
use framework\paginate\Paginate;
use shangfan\model\base\ShopPayRecord;
use shangfan\model\base\OrderMark;

/**
 * -----------------------------------------------------------------------------
 * shangfan商城
 * 商范软件 版权所有，并保留所有权利。
 * 官网地址：http://www.shangfanruanjian.com
 * -----------------------------------------------------------------------------
 * Order
 */
class Order extends BaseModel {

    public $table = 'shangfan_order';
    public $key = 'id';

    /**
     * add
     * @param Request $request
     * @param MemberUser $memberUser
     * @return type
     */
    public static function add(Request $request, MemberUser $memberUser) {
        $shopingcartOptionIds = $request->post('shopingcartOptionIds', []);
        if (!count($shopingcartOptionIds)) {
            return statusFailure('请选择购物车商品创建订单');
        }
        if (!$memberUser) {
            return statusFailure('请先登录');
        }
        $shopingcartOptionIdsStr = implode(',', $shopingcartOptionIds);
        $success = ShopShopingcartOption::update("update {table} set is_order=2 where member_user_id={$memberUser->id} and id in({$shopingcartOptionIdsStr}) and is_order=0 and status=1;");
        if (!$success) {
            return statusFailure('更新购物车状态失败，请重试');
        }
        $shopShopingcartOption = ShopShopingcartOption::get("select shop_user_id from {table} where member_user_id={$memberUser->id} and is_order=2 and status=1;");
        if (!count($shopShopingcartOption)) {
            return statusFailure('获取购物车状态失败，请重试');
        }
        $shopUserIds = [];
        foreach ($shopShopingcartOption as $val) {
            if (!in_array($val->shop_user_id, $shopUserIds)) {
                $shopUserIds[] = $val->shop_user_id;
            }
        }
        if (!count($shopUserIds)) {
            return statusFailure('获取商家失败，请重试');
        }
        $orderNumberConf = [];
        foreach ($shopUserIds as $val) {
            $success = Order::shopOrderCreate($val, $memberUser);
            if ($success['status']) {
                $orderNumberConf[] = $success['data']['orderNumber'];
            }
        }
        return statusSuccess('创建订单成功', ['orderNumberConf' => implode(',', $orderNumberConf)]);
    }

    /**
     * shopOrder
     * @param type $shopUserId
     * @param MemberUser $memberUser
     * @return type
     */
    public static function shopOrderCreate($shopUserId, MemberUser $memberUser) {
        $shopShopingcartOption = ShopShopingcartOption::get("select * from {table} where member_user_id={$memberUser->id} and shop_user_id={$shopUserId} and is_order=2 and status=1;");
        if (!count($shopShopingcartOption)) {
            return statusSuccess();
        }
        $orderNumber = Order::createOrderNumber();
        $totalAmount = 0;
        foreach ($shopShopingcartOption as $val) {
            $shopGoods = ShopGoods::detailMd5($val->shop_goods_md5);
            if (!$shopGoods) {
                continue;
            }
            $shopGoodsPrice = ShopGoodsPrice::goodsOptionPrice($val->ids_option);
            if (!$shopGoodsPrice) {
                continue;
            }
            $orderGoodsTotalAmount = bcmul($shopGoodsPrice, $val->number, 2);
            $success = OrderGoods::insert([
                        'order_number' => $orderNumber,
                        'shop_goods_md5' => $val->shop_goods_md5,
                        'member_user_id' => $val->member_user_id,
                        'ids_option' => $val->ids_option,
                        'number' => $val->number,
                        'price_member' => $shopGoodsPrice,
                        'total_amount' => $orderGoodsTotalAmount,
                        'create_time' => time(),
            ]);
            if (!$success) {
                return statusFailure('创建订单商品失败');
            }
            $totalAmount += $orderGoodsTotalAmount;
        }
        $success = Order::insert([
                    'order_number' => $orderNumber,
                    'member_user_id' => $memberUser->id,
                    'shop_user_id' => $shopUserId,
                    'total_amount' => $totalAmount,
                    'pay_type' => 1,
                    'create_time' => time(),
        ]);
        if (!$success) {
            return statusFailure('创建订单商品失败');
        }
        Order::shopingcartRefresh($memberUser->id, $shopUserId);
        return statusSuccess('', ['orderNumber' => $orderNumber]);
    }

    /**
     * shopingcartRefresh
     * @param type $memberUserId
     * @param type $shopUserId
     * @return type
     */
    public static function shopingcartRefresh($memberUserId, $shopUserId) {
        $success = ShopShopingcartOption::update("update {table} set is_order=1 where member_user_id={$memberUserId} and shop_user_id={$shopUserId} and is_order=2 and status=1;");
        if (!$success) {
            return statusFailure('更新购物车订单状态失败，请重试');
        }
        $count = ShopShopingcartOption::count("select count(id) as count from {table} where member_user_id={$memberUserId} and shop_user_id={$shopUserId} and is_order=0 and status=1;");
        if (!$count) {
            ShopShopingcart::update("update {table} set status=0 where member_user_id={$memberUserId} and shop_user_id={$shopUserId};");
        }
        return statusSuccess();
    }

    /**
     * createOrderNumber
     * @return type
     */
    public static function createOrderNumber() {
        $orderNumberStr = substr(date('YmdHis'), 2, 12);
        $fp = fopen(frameworkPath() . '/dat/shangfan.order.dat', 'r+');
        if (!$fp || !flock($fp, LOCK_EX)) {
            return $orderNumberStr . rand(100, 999);
        }
        $orderNumberStr .= substr(microtime(), 2, 3);
        flock($fp, LOCK_UN);
        return $orderNumberStr;
    }

    /**
     * listsOrderDetail
     * @param type $orderNumberArr
     * @return type
     */
    public static function listsOrderDetail($orderNumberArr) {
        if (!is_array($orderNumberArr) || !count($orderNumberArr)) {
            return [];
        }
        $whereIn = sqlWhrerInStringBuild($orderNumberArr);
        $order = Order::get("select * from {table} where order_number in ($whereIn);");
        if (!count($order)) {
            return [];
        }
        foreach ($order as &$val) {
            $shopInfo = ShopInfo::detailShopUserId($val->shop_user_id);
            $val->shopName = $shopInfo ? $shopInfo->name : '-';
            $val->listsOrderGoods = OrderGoods::listsOrderGoods($val->order_number);
            $orderMark = OrderMark::detail($val->order_number);
            $val->orderMark = $orderMark ? $orderMark->mark : '';
        }
        return $order;
    }

    /**
     * detail
     * @param type $orderNumber
     * @return type
     */
    public static function detail($orderNumber) {
        return Order::first("select * from {table} where order_number='{$orderNumber}';");
    }

    /**
     * initMemberAddress
     * @param Order $order
     */
    public static function initMemberAddress(Order $order) {
        if (!$order->member_address_id) {
            $detailDefault = MemberAddress::detailDefault($order->member_user_id);
            if (!$detailDefault) {
                $detailDefault = MemberAddress::detailLimit($order->member_user_id);
            }
            $order->member_address_id = $detailDefault->id;
            $order->save();
        }
        return $order;
    }

    /**
     * orderMemberAddress
     * @param type $orderNumberArr
     * @return type
     */
    public static function orderMemberAddress($orderNumberArr) {
        if (!is_array($orderNumberArr) || !count($orderNumberArr)) {
            return null;
        }
        $memberAddressId = 0;
        foreach ($orderNumberArr as $val) {
            $order = Order::detail($val);
            if (!$order) {
                continue;
            }
            $order = Order::initMemberAddress($order);
            if ($order && $order->member_address_id) {
                $memberAddressId = $order->member_address_id;
            }
        }
        $memberAddress = MemberAddress::detail($memberAddressId);
        if (!$memberAddress) {
            return null;
        }
        $memberAddress->provinceName = SystemCity::strCityName($memberAddress->province_id);
        $memberAddress->cityName = SystemCity::strCityName($memberAddress->city_id);
        $memberAddress->countyName = SystemCity::strCityName($memberAddress->county_id);
        $memberAddress->phone = stringPhoneReplace($memberAddress->phone);
        return $memberAddress;
    }

    /**
     * orderTotalAmount
     * @param type $orderNumberArr
     * @return int
     */
    public static function orderTotalAmount($orderNumberArr) {
        if (!is_array($orderNumberArr) || !count($orderNumberArr)) {
            return 0;
        }
        $whereIn = sqlWhrerInStringBuild($orderNumberArr);
        $order = Order::first("select sum(total_amount) total_amount from {table} where order_number in ({$whereIn});");
        if (!$order || !$order->total_amount) {
            return 0;
        }
        return $order->total_amount;
    }

    /**
     * paytypeUpdate
     * @param Request $request
     * @param MemberUser $memberUser
     * @return type
     * @throws Exception
     */
    public static function paytypeUpdate(Request $request, MemberUser $memberUser) {
        $payType = intval($request->post('payType', 0));
        if (!$payType) {
            throw new Exception('缺少支付方式参数');
        }
        switch ($payType) {
            case 1:
                if (systemConfig('pcPayBalance') != '1') {
                    return statusFailure('余额支付通道未开启');
                }
                break;
            case 2:
                if (systemConfig('pcPayWechat') != '1') {
                    return statusFailure('微信支付通道未开启');
                }
                break;
            case 2:
                if (systemConfig('pcPayAlipay') != '1') {
                    return statusFailure('支付宝支付通道未开启');
                }
                break;
        }
        $orderNumberStr = $request->post('orderNumberStr', '');
        if (!$orderNumberStr) {
            throw new Exception('缺少订单号参数');
        }
        $orderNumber = explode(',', $orderNumberStr);
        if (!is_array($orderNumber) || !count($orderNumber)) {
            throw new Exception('订单参数有误');
        }
        foreach ($orderNumber as $val) {
            $order = Order::detail($val);
            if (!$order) {
                throw new Exception('未找到订单');
            }
            if ($order->is_pay || $order->pay_time) {
                return statusFailure('订单已经支付过了，请勿重复支付');
            }
            if ($order->is_pay || $order->pay_time) {
                return statusFailure('订单已经支付过了，请勿重复支付');
            }
            if ($order->member_user_id != $memberUser->id) {
                continue;
            }
            $order->pay_type = $payType;
            $order->save();
        }
        return statusSuccess();
    }

    /**
     * addressUpdate
     * @param Request $request
     * @param MemberUser $memberUser
     * @return type
     * @throws Exception
     */
    public static function addressUpdate(Request $request, MemberUser $memberUser) {
        $memberAddressId = intval($request->post('memberAddressId', 0));
        if (!$memberAddressId) {
            throw new Exception('缺少收货地址参数');
        }
        $orderNumberStr = $request->post('orderNumberStr', '');
        if (!$orderNumberStr) {
            throw new Exception('缺少订单号参数');
        }
        $orderNumber = explode(',', $orderNumberStr);
        if (!is_array($orderNumber) || !count($orderNumber)) {
            throw new Exception('订单参数有误');
        }
        foreach ($orderNumber as $val) {
            $order = Order::detail($val);
            if (!$order) {
                throw new Exception('未找到订单');
            }
            if ($order->member_user_id != $memberUser->id) {
                continue;
            }
            $order->member_address_id = $memberAddressId;
            $order->save();
        }
        return statusSuccess();
    }

    /**
     * orderPayBalance
     * @param type $orderNumberStr
     * @param MemberUser $memberUser
     * @return type
     * @throws Exception
     */
    public static function orderPayBalance($orderNumberStr, MemberUser $memberUser) {
        if (systemConfig('pcPayBalance') != '1') {
            return statusFailure('余额支付通道未开启');
        }
        $orderNumberArr = explode(',', $orderNumberStr);
        if (!is_array($orderNumberArr) || !count($orderNumberArr)) {
            return statusFailure('订单参数有误');
        }
        if (!Order::orderCheckAddress($orderNumberArr)) {
            return statusFailure('请选择收货地址');
        }
        $orderTotalAmount = Order::orderTotalAmount($orderNumberArr);
        $memberUserRmb = MemberUser::rmb($memberUser->id);
        if ($memberUserRmb < $orderTotalAmount) {
            return statusFailure('余额不足支付，请先充值');
        }
        foreach ($orderNumberArr as $val) {
            $order = Order::detail($val);
            if (!$order) {
                throw new Exception('未找到订单');
            }
            if ($order->is_pay || $order->pay_time) {
                return statusFailure('订单已经支付过了，请勿重复支付');
            }
            if (!$order->pay_type) {
                return statusFailure('请选择支付方式');
            }
            if ($order->pay_type != 1) {
                continue;
            }
            if ($order->member_user_id != $memberUser->id) {
                continue;
            }
            $success = MemberPayRecord::reduce($memberUser->id, $order->total_amount, 'balance', $order->order_number);
            if (!$success['status']) {
                continue;
            }
            $order->fill([
                'pay_amount' => $order->total_amount,
                'is_pay' => 1,
                'pay_time' => time(),
            ]);
            $order->save();
        }
        return statusSuccess('支付成功');
    }

    /**
     * orderCheckAddress
     * @param type $orderNumberArr
     * @return boolean
     * @throws Exception
     */
    public static function orderCheckAddress($orderNumberArr) {
        if (!is_array($orderNumberArr)) {
            $orderNumberArr = explode(',', $orderNumberArr);
        }
        if (!count($orderNumberArr)) {
            return false;
        }
        foreach ($orderNumberArr as $val) {
            $order = Order::detail($val);
            if (!$order) {
                return false;
            }
            if (!$order->member_address_id) {
                return false;
            }
        }
        return true;
    }

    /**
     * listsMemberOrder
     * @param Request $request
     * @param type $memberUserId
     * @return Paginate
     */
    public static function listsMemberOrder(Request $request, $memberUserId) {
        $where = " member_user_id={$memberUserId} ";
        $type = $request->get('type', 'all');
        if ($type == 'waitPay') {
            $where .= " and is_pay=0 ";
        }
        if ($type == 'waitDelivery') {
            $where .= " and is_pay=1 and is_delivery=0 ";
        }
        if ($type == 'finash') {
            $where .= " and is_pay=1 and is_delivery=1 ";
        }
        $results = new Paginate($request, Order::db(), 6);
        $results->setSelectCount("select count(id) as count from {table} where {$where} order by create_time desc,id desc;");
        $results->setSelectResults("select * from {table} where {$where} order by create_time desc,id desc;");
        $results->appends([]);
        if (count($results->results)) {
            foreach ($results->results as &$val) {
                $shopInfo = ShopInfo::detailShopUserId($val->shop_user_id);
                $val->shopName = $shopInfo ? $shopInfo->name : '-';
                $val->listsOrderGoods = OrderGoods::listsOrderGoods($val->order_number);
                $memberAddress = MemberAddress::detail($val->member_address_id);
                $val->memberAddress = $memberAddress ? $memberAddress->name : '-';
                $val->strStatus = Order::strStatus($val);
                $val->strCreateTime = Order::strCreateTime($val);
                $val->strGoodsCount = 0;
                foreach ($val->listsOrderGoods as $_val) {
                    $val->strGoodsCount += $_val->number;
                }
            }
        }
        return $results;
    }

    /**
     * h5ListsMemberOrder
     * @param Request $request
     * @param type $memberUserId
     * @return Paginate
     */
    public static function h5ListsMemberOrder(Request $request, $memberUserId) {
        $where = " member_user_id={$memberUserId} ";
        $type = $request->get('type', 'all');
        if ($type == 'waitPay') {
            $where .= " and is_pay=0 ";
        }
        if ($type == 'waitDelivery') {
            $where .= " and is_pay=1 and is_delivery=0 ";
        }
        if ($type == 'finash') {
            $where .= " and is_pay=1 and is_delivery=1 ";
        }
        $results = new Paginate($request, Order::db(), 6);
        $results->setSelectCount("select count(id) as count from {table} where {$where} order by create_time desc,id desc;");
        $results->setSelectResults("select * from {table} where {$where} order by create_time desc,id desc;");
        $results->appends([]);
        if (count($results->results)) {
            foreach ($results->results as &$val) {
                $shopInfo = ShopInfo::detailShopUserId($val->shop_user_id);
                $val->shopName = $shopInfo ? $shopInfo->name : '-';
                $val->listsOrderGoods = OrderGoods::h5ListsOrderGoods($val->order_number);
                $memberAddress = MemberAddress::detail($val->member_address_id);
                $val->memberAddress = $memberAddress ? $memberAddress->name : '-';
                $val->strStatus = Order::strStatus($val);
                $val->strCreateTime = Order::strCreateTime($val);
                $val->strGoodsCount = 0;
                foreach ($val->listsOrderGoods as $_val) {
                    $val->strGoodsCount += $_val->number;
                }
            }
        }
        return $results;
    }

    /**
     * listsMerchantOrder
     * @param Request $request
     * @param type $shopUserId
     * @return Paginate
     */
    public static function listsMerchantOrder(Request $request, $shopUserId) {
        $where = " shop_user_id={$shopUserId} ";
        $results = new Paginate($request, Order::db(), 8);
        $results->setSelectCount("select count(id) as count from {table} where {$where} order by create_time desc,id desc;");
        $results->setSelectResults("select * from {table} where {$where} order by create_time desc,id desc;");
        $results->appends([]);
        if (count($results->results)) {
            foreach ($results->results as &$val) {
                $shopInfo = ShopInfo::detailShopUserId($val->shop_user_id);
                $val->shopName = $shopInfo ? $shopInfo->name : '-';
                $val->listsOrderGoods = OrderGoods::listsOrderGoods($val->order_number);
                $memberAddress = MemberAddress::detail($val->member_address_id);
                $val->memberAddress = $memberAddress ? $memberAddress->name : '-';
                $val->strStatus = Order::strStatus($val);
                $val->strCreateTime = Order::strCreateTime($val);
            }
        }
        return $results;
    }

    /**
     * shipments
     * @param Request $request
     * @param type $orderNumber
     * @param ShopUser $shopUser
     * @return type
     */
    public static function shipments(Request $request, $orderNumber, ShopUser $shopUser) {
        if (!$request->post('express_id', 0)) {
            return statusFailure('请选择送货方式');
        }
        $order = Order::detail($orderNumber);
        if (!$order) {
            return statusFailure('未找到订单，请重试');
        }
        if ($order->shop_user_id != $shopUser->id) {
            return statusFailure('您没有权限');
        }
        $order->fill([
            'is_shipments' => 1,
            'express_id' => intval($request->post('express_id', 0)),
            'express_no' => $request->post('express_no', ''),
            'express_shipments_time' => time(),
        ]);
        if (!$order->save()) {
            return statusFailure('发货失败，请重试');
        }
        return statusSuccess('发货成功');
    }

    /**
     * delivery
     * @param Request $reuqest
     * @param type $orderNumber
     * @param MemberUser $memberUser
     * @return type
     */
    public static function delivery(Request $reuqest, $orderNumber, MemberUser $memberUser) {
        $order = Order::detail($orderNumber);
        if (!$order) {
            return statusFailure('未找到订单，请重试');
        }
        if ($order->member_user_id != $memberUser->id) {
            return statusFailure('您没有权限');
        }
        if (!$order->is_pay) {
            return statusFailure('订单还未支付，请联系客服');
        }
        if ($order->is_delivery || $order->delivery_time) {
            return statusFailure('您已经确认收货过了');
        }
        $success = ShopPayRecord::add($order->shop_user_id, $order->pay_amount, $order->order_number);
        if (!$success['status']) {
            return $success;
        }
        $order->fill([
            'is_delivery' => 1,
            'delivery_time' => time()
        ]);
        if (!$order->save()) {
            return statusFailure('确认收货失败，请重试或联系客服');
        }
        return statusSuccess('确认收货成功');
    }

    /**
     * listsAdmin
     * @param Request $request
     * @return type
     */
    public static function listsAdmin(Request $request) {
        $where = " 1=1 and status=1 ";
        $results = new Paginate($request, Order::db(), 10);
        $results->setSelectCount("select count(id) as count from {table} where {$where} order by create_time desc,id desc;");
        $results->setSelectResults("select * from {table} where {$where} order by create_time desc,id desc;");
        $results->appends([
                //'type' => $request->get('type', 0),
        ]);
        if (count($results->results)) {
            foreach ($results->results as &$val) {
                $shopUser = ShopUser::detail($val->shop_user_id);
                $val->strShopUser = $shopUser ? $shopUser->user : '-';
                $val->strShopUserName = $shopUser ? $shopUser->name : '-';
                $memberUser = MemberUser::detail($val->member_user_id);
                $val->strMemberUser = $memberUser ? $memberUser->user : '-';
                $val->strMemberUserName = $memberUser ? $memberUser->name : '-';
                $val->strCreateTime = Order::strCreateTime($val->create_time);
                $val->strStatus = Order::strStatus($val->status);
                $val->strIsPay = Order::strIsPay($val->is_pay);
                $val->strIsShipments = Order::strIsShipments($val->is_shipments);
                $val->strIsDelivery = Order::strIsDelivery($val->is_delivery);
                $val->isOrderExcepton = Order::isOrderExcepton($val);
                $val->strIsOrderExcepton = Order::strIsOrderExcepton($val);
                $val->countOrderGoodsNumber = Order::countOrderGoodsNumber($val->order_number);
            }
        }
        return $results;
    }

    public static function countOrderGoodsNumber($shopNumber) {
        return OrderGoods::countOrderGoodsNumber($shopNumber);
    }

    /**
     * strCreateTime
     * @param type $createTime
     * @return type
     */
    public static function strCreateTime($createTime) {
        return date('Y/m/d H:i:s', $createTime);
    }

    /**
     * isOrderExcepton
     * @param type $order
     * @return int
     */
    public static function isOrderExcepton($order) {
        if ($order->is_evaluate && (!$order->is_pay || !$order->is_shipments || !$order->is_delivery)) {
            return 4;
        }
        if ($order->is_delivery && (!$order->is_pay || !$order->is_shipments)) {
            return 3;
        }
        if ($order->is_shipments && (!$order->is_pay)) {
            return 2;
        }
        return 1;
    }

    /**
     * strIsOrderExcepton
     * @param type $order
     * @return string
     */
    public static function strIsOrderExcepton($order) {
        $isOrderExcepton = Order::isOrderExcepton($order);
        switch ($isOrderExcepton) {
            case 1:
                return '正常';
            case 2:
                return '发货异常';
            case 3:
                return '收货异常';
            case 4:
                return '评价异常';
        }
        return '未知';
    }

    /**
     * strIsPay
     * @param type $isPay
     * @return string
     */
    public static function strIsPay($isPay) {
        if ($isPay == 0) {
            return '待支付';
        }
        if ($isPay == 1) {
            return '已支付';
        }
        return '-';
    }

    /**
     * strIsShipments
     * @param type $isShipments
     * @return string
     */
    public static function strIsShipments($isShipments) {
        if ($isShipments == 0) {
            return '待发货';
        }
        if ($isShipments == 1) {
            return '已发货';
        }
        return '-';
    }

    /**
     * strIsDelivery
     * @param type $isDelivery
     * @return string
     */
    public static function strIsDelivery($isDelivery) {
        if ($isDelivery == 0) {
            return '待收货';
        }
        if ($isDelivery == 1) {
            return '已收货';
        }
        return '-';
    }

    /**
     * strStatus
     * @param type $order
     * @return string
     */
    public static function strStatus($order) {
        if (!$order->is_pay) {
            return '待支付';
        }
        if ($order->is_delivery && !$order->is_shipments) {
            return '已完成';
        }
        if (!$order->is_shipments) {
            return '待发货';
        }
        if (!$order->is_delivery) {
            return '待收货';
        }
        if (!$order->is_evaluate) {
            return '待评价';
        }
        return '已完成';
    }

    /**
     * countMemberUserOrder
     * @param type $memberUserId
     * @return type
     */
    public static function countMemberUserOrder($memberUserId) {
        return Order::count("select count(id) as count from {table} where member_user_id={$memberUserId};");
    }

}
