<?php
/**
 * 微信支付
 */

include_once ("WxException.php");
include_once ("WxPayConf.php");
include_once ("WxPayBase.php");


/**
 * 请求型接口的基类
 */
class WxpayClient extends WxPayBase {
	var $parameters; // 请求参数，类型为关联数组
	public $response; // 微信返回的响应
	public $result; // 返回参数，类型为关联数组
	var $url; // 接口链接
	var $curl_timeout; // curl超时时间
	
	/**
	 * 设置请求参数
	 */
	function setParameter($parameter, $parameterValue) {
		$this->parameters [$this->trimString ( $parameter )] = $this->trimString ( $parameterValue );
	}
	
	/**
	 * 设置标配的请求参数，生成签名，生成接口参数xml
	 */
	function createXml() {
		$this->parameters ["appid"] = WxPayConf::$APPID; // 公众账号ID
		$this->parameters ["mch_id"] = WxPayConf::$MCHID; // 商户号
		$this->parameters ["nonce_str"] = $this->createNoncestr (); // 随机字符串
		$this->parameters ["sign"] = $this->getSign ( $this->parameters ); // 签名
		return $this->arrayToXml ( $this->parameters );
	}
	
	/**
	 * post请求xml
	 */
	function postXml() {
		$xml = $this->createXml ();
		$this->response = $this->postXmlCurl ( $xml, $this->url, $this->curl_timeout );
		return $this->response;
	}
	
	/**
	 * 使用证书post请求xml
	 */
	function postXmlSSL() {
		$xml = $this->createXml ();
		$this->response = $this->postXmlSSLCurl ( $xml, $this->url, $this->curl_timeout );
		return $this->response;
	}
	
	/**
	 * 获取结果，默认不使用证书
	 */
	function getResult() {
		$this->postXml ();
		$this->result = $this->xmlToArray ( $this->response );
		return $this->result;
	}
}

/**
 * 统一支付接口类
 */
class UnifiedOrder extends WxpayClient {
	function __construct() {
		// 设置接口链接
		$this->url = "https://api.mch.weixin.qq.com/pay/unifiedorder";
		// 设置curl超时时间
		$this->curl_timeout = WxPayConf::$CURL_TIMEOUT;
	}
	
	/**
	 * 生成接口参数xml
	 */
	function createXml() {
		try {
			// 检测必填参数
			if ($this->parameters ["out_trade_no"] == null) {
				throw new WxException ( "缺少统一支付接口必填参数out_trade_no！" . "<br>" );
			} elseif ($this->parameters ["body"] == null) {
				throw new WxException ( "缺少统一支付接口必填参数body！" . "<br>" );
			} elseif ($this->parameters ["total_fee"] == null) {
				throw new WxException ( "缺少统一支付接口必填参数total_fee！" . "<br>" );
			} elseif ($this->parameters ["notify_url"] == null) {
				throw new WxException ( "缺少统一支付接口必填参数notify_url！" . "<br>" );
			} elseif ($this->parameters ["trade_type"] == null) {
				throw new WxException ( "缺少统一支付接口必填参数trade_type！" . "<br>" );
			} elseif ($this->parameters ["trade_type"] == "JSAPI" && $this->parameters ["openid"] == NULL) {
				throw new WxException ( "统一支付接口中，缺少必填参数openid！trade_type为JSAPI时，openid为必填参数！" . "<br>" );
			}
			$this->parameters ["appid"] = WxPayConf::$APPID; // 公众账号ID
			$this->parameters ["mch_id"] = WxPayConf::$MCHID; // 商户号
			$this->parameters ["spbill_create_ip"] = $this->getClientIp(); // 终端ip
			$this->parameters ["nonce_str"] = $this->createNoncestr (); // 随机字符串
			$this->parameters ["sign"] = $this->getSign ( $this->parameters ); // 签名
			return $this->arrayToXml ( $this->parameters );
		} catch ( WxException $e ) {
			die ( $e->errorMessage () );
		}
	}
	
	function getClientIp(){
	    $unknown = 'unknown';
		if ( isset($_SERVER['HTTP_X_FORWARDED_FOR']) && $_SERVER['HTTP_X_FORWARDED_FOR'] && strcasecmp($_SERVER['HTTP_X_FORWARDED_FOR'], $unknown) ) {
			$ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
		} elseif ( isset($_SERVER['REMOTE_ADDR']) && $_SERVER['REMOTE_ADDR'] && strcasecmp($_SERVER['REMOTE_ADDR'], $unknown) ) {
			$ip = $_SERVER['REMOTE_ADDR'];
		}
		if (false !== strpos($ip, ',')){
			$ip = reset(explode(',', $ip));
		}
		return $ip;
	}


	
	/**
	 * 设置jsapi的参数
	 */
	public function getParameters($obj) {
		$appObj ["appid"] = WxPayConf::$APPID;
		$appObj ["partnerid"] = WxPayConf::$MCHID; // 商户号
		$appObj ["prepayid"] = $obj["prepayid"];
		$appObj ["package"] = "Sign=WXPay";
		$appObj ["noncestr"] = $this->createNoncestr ();
		$timeStamp = time ();
		$appObj ["timestamp"] =  (string)$timeStamp;
		$appObj ["sign"] = $this->getSign ( $appObj );
		return $appObj;
	}
	
	/**
	 * 获取prepay_id
	 */
	function getPrepayId() {
		$this->postXml ();
		$this->result = $this->xmlToArray ( $this->response );
		
		$prepay_id = $this->result ["prepay_id"];
		
		return $prepay_id;
	}
}

/**
 * 订单查询接口
 */
class OrderQuery extends WxpayClient {
	function __construct() {
		// 设置接口链接
		$this->url = "https://api.mch.weixin.qq.com/pay/orderquery";
		// 设置curl超时时间
		$this->curl_timeout = WxPayConf::$CURL_TIMEOUT;
	}
	
	/**
	 * 生成接口参数xml
	 */
	function createXml() {
		try {
			// 检测必填参数
			if ($this->parameters ["out_trade_no"] == null && $this->parameters ["transaction_id"] == null) {
				throw new WxException ( "订单查询接口中，out_trade_no、transaction_id至少填一个！" . "<br>" );
			}
			$this->parameters ["appid"] = WxPayConf::$APPID; // 公众账号ID
			$this->parameters ["mch_id"] = WxPayConf::$MCHID; // 商户号
			$this->parameters ["nonce_str"] = $this->createNoncestr (); // 随机字符串
			$this->parameters ["sign"] = $this->getSign ( $this->parameters ); // 签名
			return $this->arrayToXml ( $this->parameters );
		} catch ( WxException $e ) {
			die ( $e->errorMessage () );
		}
	}
}


/**
 * 企业付款接口
 */
class Transfers extends WxpayClient{
	function __construct() {
		// 设置接口链接
		$this->url = "https://api.mch.weixin.qq.com/mmpaymkttransfers/promotion/transfers";
		// 设置curl超时时间
		$this->curl_timeout = WxPayConf::$CURL_TIMEOUT;
	}

	/**
	 * 生成接口参数xml
	 */
	function createXml() {
		try {
			// 检测必填参数
			if ($this->parameters ["partner_trade_no"] == null) {
				throw new WxException ( "提现申请接口中,商户订单号partner_trade_no为必填参数！" . "<br>" );
			} elseif ($this->parameters ["desc"] == null) {
				throw new WxException ( "提现申请接口中,企业付款描述信息desc为必填参数！" . "<br>" );
			} elseif ($this->parameters ["amount"] == null) {
				throw new WxException ( "缺少统一支付接口必填参数amount！" . "<br>" );
			}elseif ($this->parameters ["openid"] == NULL) {
				throw new WxException ( "提现申请接口中，缺少必填参数openid为必填参数！" . "<br>" );
			}
			$this->parameters ["appid"] = WxPayConf::$APPID; // 公众账号ID
			$this->parameters ["mch_id"] = WxPayConf::$MCHID; // 商户号
			$this->parameters ["spbill_create_ip"] = $this->getClientIp(); // 终端ip
			$this->parameters ["nonce_str"] = $this->createNoncestr (); // 随机字符串
			$this->parameters ["sign"] = $this->getSign ( $this->parameters ); // 签名
			return $this->arrayToXml ( $this->parameters );
		} catch ( WxException $e ) {
			die ( $e->errorMessage () );
		}
	}
	/**
	 * 作用：获取结果，使用证书通信
	 */
	function getResult() {
		$this->postXmlSSL ();
		$this->result = $this->xmlToArray ( $this->response );
		return $this->result;
	}

}


/**
 * 退款申请接口
 */
class Refund extends WxpayClient {
	function __construct() {
		// 设置接口链接
		$this->url = "https://api.mch.weixin.qq.com/secapi/pay/refund";
		// 设置curl超时时间
		$this->curl_timeout = WxPayConf::$CURL_TIMEOUT;
	}
	
	/**
	 * 生成接口参数xml
	 */
	function createXml() {
		try {
			// 检测必填参数
			if ((isset($this->parameters ["out_trade_no"]) && $this->parameters ["out_trade_no"] == null) && $this->parameters ["transaction_id"] == null) {
				throw new WxException ( "退款申请接口中，out_trade_no、transaction_id至少填一个！" . "<br>" );
			} elseif ($this->parameters ["out_refund_no"] == null) {
				throw new WxException ( "退款申请接口中，缺少必填参数out_refund_no！" . "<br>" );
			} elseif ($this->parameters ["total_fee"] == null) {
				throw new WxException ( "退款申请接口中，缺少必填参数total_fee！" . "<br>" );
			} elseif ($this->parameters ["refund_fee"] == null) {
				throw new WxException ( "退款申请接口中，缺少必填参数refund_fee！" . "<br>" );
			}
			$this->parameters ["appid"] = WxPayConf::$APPID; // 公众账号ID
			$this->parameters ["mch_id"] = WxPayConf::$MCHID; // 商户号
			$this->parameters ["nonce_str"] = $this->createNoncestr (); // 随机字符串
			$this->parameters ["sign"] = $this->getSign ( $this->parameters ); // 签名
			return $this->arrayToXml ( $this->parameters );
		} catch ( SDKRuntimeException $e ) {
			die ( $e->errorMessage () );
		}
	}
	/**
	 * 作用：获取结果，使用证书通信
	 */
	function getResult() {
		$this->postXmlSSL ();
		$this->result = $this->xmlToArray ( $this->response );
		return $this->result;
	}
}

/**
 * 退款查询接口
 */
class RefundQuery extends WxpayClient {
	function __construct() {
		// 设置接口链接
		$this->url = "https://api.mch.weixin.qq.com/pay/refundquery";
		// 设置curl超时时间
		$this->curl_timeout = WxPayConf_pub::$CURL_TIMEOUT;
	}
	
	/**
	 * 生成接口参数xml
	 */
	function createXml() {
		try {
			if ($this->parameters ["out_refund_no"] == null && $this->parameters ["out_trade_no"] == null && $this->parameters ["transaction_id"] == null && $this->parameters ["refund_id "] == null) {
				throw new SDKRuntimeException ( "退款查询接口中，out_refund_no、out_trade_no、transaction_id、refund_id四个参数必填一个！" . "<br>" );
			}
			$this->parameters ["appid"] = WxPayConf_pub::$APPID; // 公众账号ID
			$this->parameters ["mch_id"] = WxPayConf_pub::$MCHID; // 商户号
			$this->parameters ["nonce_str"] = $this->createNoncestr (); // 随机字符串
			$this->parameters ["sign"] = $this->getSign ( $this->parameters ); // 签名
			return $this->arrayToXml ( $this->parameters );
		} catch ( SDKRuntimeException $e ) {
			die ( $e->errorMessage () );
		}
	}
	
	/**
	 * 作用：获取结果，使用证书通信
	 */
	function getResult() {
		$this->postXmlSSL ();
		$this->result = $this->xmlToArray ( $this->response );
		return $this->result;
	}
}


/**
 * 响应型接口基类
 */
class WxpayServer extends WxPayBase {
	public $data; // 接收到的数据，类型为关联数组
	var $returnParameters; // 返回参数，类型为关联数组
	
	/**
	 * 将微信的请求xml转换成关联数组，以方便数据处理
	 */
	function saveData($xml) {
		$this->data = $this->xmlToArray ( $xml );
	}
	function checkSign() {
		$tmpData = $this->data;
		unset ( $tmpData ['sign'] );
		$sign = $this->getSign ( $tmpData ); // 本地签名
		if ($this->data ['sign'] == $sign) {
			return TRUE;
		}
		return FALSE;
	}
	
	/**
	 * 获取微信的请求数据
	 */
	function getData() {
		return $this->data;
	}
	
	/**
	 * 设置返回微信的xml数据
	 */
	function setReturnParameter($parameter, $parameterValue) {
		$this->returnParameters [$this->trimString ( $parameter )] = $this->trimString ( $parameterValue );
	}
	
	/**
	 * 生成接口参数xml
	 */
	function createXml() {
		return $this->arrayToXml ( $this->returnParameters );
	}
	
	/**
	 * 将xml数据返回微信
	 */
	function returnXml() {
		$returnXml = $this->createXml ();
		return $returnXml;
	}

	function decryptReqinfo($req_info){
		$md5Key = md5(WxPayConf::$KEY);
   		$decrypt = base64_decode($req_info,true);
       	$xml = openssl_decrypt($decrypt , 'aes-256-ecb', $md5Key, OPENSSL_RAW_DATA);
       	$reqInfo = $this->xmlToArray ( $xml );
       	return $reqInfo;
	}
}

/**
 * 通用通知接口
 */
class Notify extends WxpayServer {
}


/**
 * 请求商家获取商品信息接口
 */
class NativeCall extends WxpayServer {
	/**
	 * 生成接口参数xml
	 */
	function createXml() {
		if ($this->returnParameters ["return_code"] == "SUCCESS") {
			$this->returnParameters ["appid"] = WxPayConf::$APPID; // 公众账号ID
			$this->returnParameters ["mch_id"] = WxPayConf::$MCHID; // 商户号
			$this->returnParameters ["nonce_str"] = $this->createNoncestr (); // 随机字符串
			$this->returnParameters ["sign"] = $this->getSign ( $this->returnParameters ); // 签名
		}
		return $this->arrayToXml ( $this->returnParameters );
	}
	
	/**
	 * 获取product_id
	 */
	function getProductId() {
		$product_id = $this->data ["product_id"];
		return $product_id;
	}
}

/**
 * 静态链接二维码
 */
class NativeLink extends WxPayBase {
	var $parameters; // 静态链接参数
	var $url; // 静态链接
	function __construct() {
	}
	
	/**
	 * 设置参数
	 */
	function setParameter($parameter, $parameterValue) {
		$this->parameters [$this->trimString ( $parameter )] = $this->trimString ( $parameterValue );
	}
	
	/**
	 * 生成Native支付链接二维码
	 */
	function createLink() {
		try {
			if ($this->parameters ["product_id"] == null) {
				throw new WxException ( "缺少Native支付二维码链接必填参数product_id！" . "<br>" );
			}
			$this->parameters ["appid"] = WxPayConf::$APPID; // 公众账号ID
			$this->parameters ["mch_id"] = WxPayConf::$MCHID; // 商户号
			$time_stamp = time ();
			$this->parameters ["time_stamp"] = "$time_stamp"; // 时间戳
			$this->parameters ["nonce_str"] = $this->createNoncestr (); // 随机字符串
			$this->parameters ["sign"] = $this->getSign ( $this->parameters ); // 签名
			$bizString = $this->formatBizQueryParaMap ( $this->parameters, false );
			$this->url = "weixin://wxpay/bizpayurl?" . $bizString;
		} catch ( WxException $e ) {
			die ( $e->errorMessage () );
		}
	}
	
	/**
	 * 返回链接
	 */
	function getUrl() {
		$this->createLink ();
		return $this->url;
	}
}

/**
 * JSAPI支付——H5网页端调起支付接口
 */
class JsApi extends WxPayBase {
	var $code; // code码，用以获取openid
	var $openid; // 用户的openid
	var $parameters; // jsapi参数，格式为json
	var $prepay_id; // 使用统一支付接口得到的预支付id
	var $curl_timeout; // curl超时时间
	function __construct() {
		// 设置curl超时时间
		$this->curl_timeout = WxPayConf::$CURL_TIMEOUT;
	}
	
	/**
	 * 生成可以获得code的url
	 */
	function createOauthUrlForCode($redirectUrl) {
		$urlObj ["appid"] = WxPayConf::$APPID;
		$urlObj ["redirect_uri"] = "$redirectUrl";
		$urlObj ["response_type"] = "code";
		$urlObj ["scope"] = "snsapi_base";
		$urlObj ["state"] = "STATE" . "#wechat_redirect";
		$bizString = $this->formatBizQueryParaMap ( $urlObj, false );
		return "https://open.weixin.qq.com/connect/oauth2/authorize?" . $bizString;
	}
	
	/**
	 * 生成可以获得openid的url
	 */
	function createOauthUrlForOpenid() {
		$urlObj ["appid"] = WxPayConf::$APPID;
		$urlObj ["secret"] = WxPayConf::$APPSECRET;
		$urlObj ["code"] = $this->code;
		$urlObj ["grant_type"] = "authorization_code";
		$bizString = $this->formatBizQueryParaMap ( $urlObj, false );
		return "https://api.weixin.qq.com/sns/oauth2/access_token?" . $bizString;
	}
	
	/**
	 * 通过curl向微信提交code，以获取openid
	 */
	function getOpenid() {
		$url = $this->createOauthUrlForOpenid ();
		// 初始化curl
		$ch = curl_init ();
		// 设置超时
		curl_setopt ( $ch, CURLOPT_TIMEOUT, $this->curl_timeout );
		curl_setopt ( $ch, CURLOPT_URL, $url );
		curl_setopt ( $ch, CURLOPT_SSL_VERIFYPEER, FALSE );
		curl_setopt ( $ch, CURLOPT_SSL_VERIFYHOST, FALSE );
		curl_setopt ( $ch, CURLOPT_HEADER, FALSE );
		curl_setopt ( $ch, CURLOPT_RETURNTRANSFER, TRUE );
		// 运行curl，结果以jason形式返回
		$res = curl_exec ( $ch );
		curl_close ( $ch );
		// 取出openid
		$data = json_decode ( $res, true );
		$this->openid = $data ['openid'];
		return $this->openid;
	}
	
	/**
	 * 设置prepay_id
	 */
	function setPrepayId($prepayId) {
		$this->prepay_id = $prepayId;
	}
	
	/**
	 * 设置code
	 */
	function setCode($code_) {
		$this->code = $code_;
	}
	
	/**
	 * 设置jsapi的参数
	 */
	public function getParameters() {
		$jsApiObj ["appId"] = WxPayConf::$APPID;
		$timeStamp = time ();
		$jsApiObj ["timeStamp"] =  (string)$timeStamp;
		$jsApiObj ["nonceStr"] = $this->createNoncestr ();
		$jsApiObj ["package"] = "prepay_id=$this->prepay_id";
		$jsApiObj ["signType"] = "MD5";
		$jsApiObj ["paySign"] = $this->getSign ( $jsApiObj );
		$this->parameters = json_encode ( $jsApiObj );
		
		return $this->parameters;
	}
}


?>
