﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Web;

namespace ynhtm.BLL
{
    using Common;
    using DBUtility;

    /// <summary>
    /// 职位操作。
    /// </summary>
    public class Job
    {
        public const string SQL_ORDER = "ORDER BY " + DAL.Job.JOBID + " DESC";

        public static readonly IDictionary<int, string> TYPES;
        public static readonly IDictionary<int, string> SEXS;
        public static readonly IDictionary<int, string> DEGREES;
        public static readonly IDictionary<int, string> EXPERIENCES;

        static Job()
        {
            TYPES = new Dictionary<int, string>();
            TYPES.Add(1, "社会招聘");
            TYPES.Add(2, "校园招聘");
            TYPES.Add(3, "高级人才");

            SEXS = new Dictionary<int, string>();
            SEXS.Add(1, "男");
            SEXS.Add(2, "女");

            DEGREES = new Dictionary<int, string>();
            DEGREES.Add(1, "初中");
            DEGREES.Add(2, "高中");
            DEGREES.Add(3, "中技");
            DEGREES.Add(4, "中专");
            DEGREES.Add(5, "大专");
            DEGREES.Add(6, "本科");
            DEGREES.Add(7, "硕士");
            DEGREES.Add(8, "博士");

            EXPERIENCES = new Dictionary<int, string>();
            EXPERIENCES.Add(1, "在读学生");
            EXPERIENCES.Add(2, "应届毕业生");
            EXPERIENCES.Add(3, "一年以上");
            EXPERIENCES.Add(4, "二年以上");
            EXPERIENCES.Add(5, "三年以上");
            EXPERIENCES.Add(6, "五年以上");
            EXPERIENCES.Add(7, "八年以上");
            EXPERIENCES.Add(8, "十年以上");
        }

        /// <summary>
        /// 插入职位实例到数据库。
        /// </summary>
        /// <param name="item">职位实例。</param>
        public void Insert(Model.Job item)
        {
            item.Url = item.Rewrite;
            DAL.Job.Insert(item);
            //没有重写地址，更新地址
            if (item.Url.IsNullOrWhiteSpace())
            {
                item.Url = "/Job/" + item.JobID;
                using (var db = DbBase.GetDb())
                    db.ExecuteNonQuery("UPDATE " + DAL.Job.SQL_TABLE + " SET " + DAL.Job.URL + " = @" + DAL.Job.URL + " WHERE " + DAL.Job.JOBID + " = @" + DAL.Job.JOBID, db.CreateParameter(DAL.Job.JOBID, item.JobID), db.CreateParameter(DAL.Job.URL, item.Url));
            }
        }

        #region JobID

        /// <summary>
        /// 取得职位实例。
        /// </summary>
        /// <param name="jobID">职位编号。</param>
        /// <returns>返回职位实例，没有将返回 null 。</returns>
        public Model.Job GetByJobID(int jobID)
        {
            return DAL.Job.GetByJobID(jobID);
        }

        /// <summary>
        /// 更新职位。
        /// </summary>
        /// <param name="item">职位实例。</param>
        /// <returns>返回受到影响的行数。</returns>
        public int Update(Model.Job item)
        {
            //没有重写地址，更新地址
            item.Url = item.Rewrite.IifNullOrWhiteSpace("/Job/" + item.JobID);
            return DAL.Job.Update(item);
        }

        /// <summary>
        /// 删除职位。
        /// </summary>
        /// <param name="jobID">职位编号。</param>
        /// <returns>返回受到影响的行数。</returns>
        public int Delete(int jobID)
        {
            return DAL.Job.Delete(jobID);
        }

        #endregion

        /// <summary>
        /// 取得职位分页集合。
        /// </summary>
        /// <param name="recordCount">记录总数。</param>
        /// <param name="pageSize">分页大小。</param>
        /// <param name="pageIndex">要返回的页数。</param>
        /// <param name="query">查询条件。</param>
        /// <param name="order">排序方式。</param>
        /// <returns>返回分页数据。</returns>
        public DataTable GetList(ref int recordCount, int pageSize, int pageIndex = 1, IQuery query = null, string order = SQL_ORDER)
        {
            using (var db = DbBase.GetDb())
                return db.GetPage(ref recordCount, DAL.Job.SQL_TABLE, order, pageSize, pageIndex, query, DAL.Job.SQL_FIELD);
        }

        /// <summary>
        /// 取得职位分页集合。
        /// </summary>
        /// <param name="pageSize">分页大小。</param>
        /// <param name="pageIndex">要返回的页数。</param>
        /// <param name="query">查询条件。</param>
        /// <param name="order">排序方式。</param>
        /// <returns>返回分页数据。</returns>
        public DataTable GetList(int pageSize, int pageIndex = 1, IQuery query = null, string order = SQL_ORDER)
        {
            using (var db = DbBase.GetDb())
                return db.GetPage(DAL.Job.SQL_TABLE, order, pageSize, pageIndex, query, DAL.Job.SQL_FIELD);
        }

        /// <summary>
        /// 取得职位类型文字描述。
        /// </summary>
        /// <param name="type">数据类型。</param>
        /// <returns>返回类型文字描述，没有符合的描述将返回 null。</returns>
        public string GetType(int? type)
        {
            if (type == null) return null;
            return TYPES[type.Value].ToString();
        }

        /// <summary>
        /// 取得职位性别要求文字描述。
        /// </summary>
        /// <param name="type">性别的数据类型。</param>
        /// <returns>返回性别文字描述，没有符合的描述将返回 null。</returns>
        public string GetSex(int? sex)
        {
            if (sex == null) return null;
            return SEXS[sex.Value].ToString();
        }

        /// <summary>
        /// 取得职位要求学历文字描述。
        /// </summary>
        /// <param name="type">学历的数据类型。</param>
        /// <returns>返回学历文字描述，没有符合的描述将返回 null。</returns>
        public string GetDegree(int? degree)
        {
            if (degree == null) return null;
            return DEGREES[degree.Value].ToString();
        }

        /// <summary>
        /// 取得职位工作经验文字描述。
        /// </summary>
        /// <param name="type">工作经验的数据类型。</param>
        /// <returns>返回工作经验文字描述，没有符合的描述将返回 null。</returns>
        public string GetExperience(int? experience)
        {
            if (experience == null) return null;
            return EXPERIENCES[experience.Value].ToString();
        }

        /// <summary>
        /// 增加职位查看次数。
        /// </summary>
        /// <param name="jobID">职位编号。</param>
        /// <param name="request">客户端 Web 请求。</param>
        /// <param name="response">客户端 HTTP 响应的信息。</param>
        /// <returns>返回受到影响的行数。</returns>
        public int IncViewCount(int jobID, HttpRequest request, HttpResponse response)
        {
            var setting = new BLL.Setting();
            var settingItem = setting.Get();
            var name = "ynhtm.Job." + jobID;
            var key = "View";
            var cookie = request.Cookies[name] ?? new HttpCookie(name) { Expires = DateTime.Parse(DateTime.Now.ToString("yyy-MM-dd 23:59:59")) };
            if (settingItem.TotalType == 2 && !cookie[key].IsNullOrEmpty()) return 0;
            cookie[key] = "1";
            response.AppendCookie(cookie);
            using (var db = DbBase.GetDb())
                return db.ExecuteNonQuery("UPDATE " + DAL.Job.SQL_TABLE + " SET " + DAL.Job.VIEWCOUNT + " = " + DAL.Job.VIEWCOUNT + " + 1 WHERE " + DAL.Job.JOBID + " = @" + DAL.Job.JOBID, db.CreateParameter(DAL.Job.JOBID, jobID));
        }

        /// <summary>
        /// 取得职位的查看次数。
        /// </summary>
        /// <param name="jobID">职位编号。</param>
        /// <returns>返回职位的查看次数。</returns>
        public int GetViewCount(int jobID)
        {
            using (var db = DbBase.GetDb())
                return Convert.ToInt32(db.ExecuteScalar("SELECT " + DAL.Job.VIEWCOUNT + " FROM " + DAL.Job.SQL_TABLE + " WHERE " + DAL.Job.JOBID + " = @" + DAL.Job.JOBID, db.CreateParameter(DAL.Job.JOBID, jobID)));
        }

        /// <summary>
        /// 切换职位状态。
        /// </summary>
        /// <param name="jobID">职位编号。</param>
        /// <returns>返回受到影响的行数。</returns>
        public int EnabledToggle(int jobID)
        {
            using (var db = DbBase.GetDb())
                return db.ExecuteNonQuery("UPDATE " + DAL.Job.SQL_TABLE + " SET " + DAL.Job.ENABLED + " = ABS(" + DAL.Job.ENABLED + " - 1) WHERE " + DAL.Job.JOBID + " = @" + DAL.Job.JOBID, db.CreateParameter(DAL.Job.JOBID, jobID));
        }

        /// <summary>
        /// 统计职位数据。
        /// </summary>
        /// <returns>返回受到影响的行数。</returns>
        public int Sumup()
        {
            using (var db = DbBase.GetDb())
                return db.ExecuteNonQuery("UPDATE " + DAL.Job.SQL_TABLE + " SET " + DAL.Job.TRYCOUNT + " = (SELECT COUNT(0) FROM " + DAL.JobTry.SQL_TABLE + " WHERE " + DAL.JobTry.JOBID + " = " + DAL.Job.SQL_TABLE + "." + DAL.Job.JOBID + ")");
        }
    }
}
