﻿using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Text.RegularExpressions;
using System.Web;
using System.Web.Mvc;
using System.Web.Script.Serialization;

namespace ynhtm.Theme
{
    using Common;

    /// <summary>
    /// 模板助手。
    /// </summary>
    public static class Helper
    {
        /// <summary>
        /// 取得缩略图地址。
        /// </summary>
        /// <param name="url">原始图片地址。</param>
        /// <param name="width">宽度。</param>
        /// <param name="height">高度。</param>
        /// <param name="mode">模式。</param>
        /// <param name="quality">品质。</param>
        /// <returns>返回缩略图地址。</returns>
        public static string Thumb(string url, int width, int height, ImageThumbMode mode = ImageThumbMode.Auto, long quality = 80)
        {
            return Utils.Thumb(url, width, height, mode, quality);
        }

        /// <summary>
        /// 取得缩略图地址。
        /// </summary>
        /// <param name="url">原始图片地址。</param>
        /// <param name="size">要显示的图像大小。</param>
        /// <param name="mode">模式。</param>
        /// <param name="quality">品质。</param>
        /// <returns>返回缩略图地址。</returns>
        public static string Thumb(string url, Size size, ImageThumbMode mode = ImageThumbMode.Auto, long quality = 80)
        {
            return Thumb(url, size.Width, size.Height, mode, quality);
        }

        /// <summary>
        /// 转换WAP文本。
        /// </summary>
        /// <param name="s">要转换的文本。</param>
        /// <returns>返回转换后的文本。</returns>
        public static string WapText(string s)
        {
            var setting = new BLL.Setting();
            var settingItem = setting.Get();
            if (s.IsNullOrEmpty()) return s;
            s = Regex.Replace(s, @"<\s*/?\s*(?<tag>\w*)[^>]*>", delegate(Match a) { return string.Equals(a.Result("${tag}"), "img", StringComparison.OrdinalIgnoreCase) ? a.Value : ""; }, RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.Compiled);
            s = Regex.Replace(s, @"<img(.*?)>", delegate(Match a)
            {
                var b = Regex.Match(a.Result("$1"), @"\s+src=[""'](?<src>[^""'>]+)", RegexOptions.IgnoreCase | RegexOptions.Compiled);
                if (!b.Success) return "";
                return "<p align=\"center\"><img src=\"" + Thumb(b.Result("${src}"), 167, 173, ImageThumbMode.Auto, 70L).AttributeEncode() + "\" alt=\"\"/></p>";
            }, RegexOptions.IgnoreCase | RegexOptions.Compiled);
            s = Regex.Replace(s, @"[\s\n\r\t]+<p", @"<p", RegexOptions.IgnoreCase | RegexOptions.Compiled);
            s = Regex.Replace(s, @"</p>[\s\n\r\t]+", @"</p>", RegexOptions.IgnoreCase | RegexOptions.Compiled);
            s = Regex.Replace(s, @"(^|>)([^<]+)(<|$)", delegate(Match a)
            {
                return a.Result("$1") + a.Result("$2").HtmlDecode().Replace("&", "&amp;").Replace("'", "&apos;").Replace(" ", "&nbsp;").Replace("<", "&lt;").Replace(">", "&gt;").Replace("\"", "&quot;").Replace("\r", "").Replace("\t", "").Replace("\n", "<br />") + a.Result("$3");
            }, RegexOptions.Compiled);
            return s;
        }

        /// <summary>
        /// 将对象序列化为 JSON 字符。 
        /// </summary>
        /// <param name="obj">要序列化的对象。</param>
        /// <returns>返回序列化后的 JSON 字符。</returns>
        public static string JSerialize(object obj)
        {
            return new JavaScriptSerializer().Serialize(obj);
        }

        static HtmlString ToHtmlString(TagBuilder builder, TagRenderMode renderMode)
        {
            return new HtmlString(builder.ToString(renderMode));
        }

        static void MergeAttributes(TagBuilder builder, object attributes)
        {
            Action<string, object> MergeAttribute = (string name, object value) =>
            {
                if (value == Text.Null) return;
                if (string.Equals(name, "classname", StringComparison.OrdinalIgnoreCase))
                    name = "class";
                builder.MergeAttribute(name, value == null ? "" : value.ToString(), true);
            };
            if (attributes is IDictionary)
            {
                foreach (DictionaryEntry i in (IDictionary)attributes)
                    MergeAttribute(i.Key.ToString(), i.Value);
            }
            else
            {
                foreach (PropertyDescriptor i in TypeDescriptor.GetProperties(attributes))
                    MergeAttribute(i.Name, i.GetValue(attributes));
            }
        }

        /// <summary>
        /// Html 标签开始。
        /// </summary>
        /// <param name="tagName">标签名。</param>
        /// <param name="attributes">属性参数。</param>
        /// <returns>返回标签开始。</returns>
        public static HtmlString BeginTag(string tagName, object attributes = null)
        {
            var builder = new TagBuilder(tagName);
            if (attributes != null) MergeAttributes(builder, attributes);
            return ToHtmlString(builder, TagRenderMode.StartTag);
        }

        /// <summary>
        /// Html 标签结束。
        /// </summary>
        /// <param name="tagName">标签名。</param>
        /// <returns>返回标签结束。</returns>
        public static HtmlString EndTag(string tagName)
        {
            return ToHtmlString(new TagBuilder(tagName), TagRenderMode.EndTag);
        }

        /// <summary>
        /// Html 包裹标签。
        /// </summary>
        /// <param name="tagName">标签名。</param>
        /// <param name="innerHtml">标签内部文本。</param>
        /// <param name="attributes">属性参数。</param>
        /// <returns>返回包裹标签。</returns>
        public static HtmlString WrapTag(string tagName, string innerHtml = null, object attributes = null)
        {
            var builder = new TagBuilder(tagName);
            if (attributes != null) MergeAttributes(builder, attributes);
            if (!innerHtml.IsNullOrEmpty()) builder.InnerHtml = innerHtml;
            return ToHtmlString(builder, TagRenderMode.Normal);
        }

        /// <summary>
        /// Html INPUT 标签。
        /// </summary>
        /// <param name="type">类型。</param>
        /// <param name="value">值。</param>
        /// <param name="attributes">属性参数。</param>
        /// <returns>返回 INPUT 标签。</returns>
        public static HtmlString Input(string type, string value = null, object attributes = null)
        {
            var builder = new TagBuilder("input");
            builder.MergeAttribute("type", type);
            if (!value.IsNullOrEmpty()) builder.MergeAttribute("value", value);
            if (attributes != null) MergeAttributes(builder, attributes);
            return ToHtmlString(builder, TagRenderMode.SelfClosing);
        }

        /// <summary>
        /// Html A 标签。
        /// </summary>
        /// <param name="href">链接地址。</param>
        /// <param name="innerHtml">标签内部文本。</param>
        /// <param name="attributes">属性参数。</param>
        /// <returns>返回 A 标签。</returns>
        public static HtmlString A(string href, string innerHtml = null, object attributes = null)
        {
            var builder = new TagBuilder("a");
            builder.MergeAttribute("href", href);
            if (attributes != null) MergeAttributes(builder, attributes);
            if (!innerHtml.IsNullOrEmpty()) builder.InnerHtml = innerHtml;
            return ToHtmlString(builder, TagRenderMode.Normal);
        }

        /// <summary>
        /// Html IMG 标签。
        /// </summary>
        /// <param name="src">图片地址。</param>
        /// <param name="attributes">属性参数。</param>
        /// <returns>返回 IMG 标签。</returns>
        public static HtmlString Img(string src, object attributes = null)
        {
            var builder = new TagBuilder("img");
            builder.MergeAttribute("src", src);
            if (attributes != null) MergeAttributes(builder, attributes);
            return ToHtmlString(builder, TagRenderMode.SelfClosing);
        }

        /// <summary>
        /// 取得带有样式的文章标题。
        /// </summary>
        /// <param name="item">文章实例。</param>
        /// <returns>返回带有样式的文章标题。</returns>
        public static string GetStyleTitle(Model.Post item)
        {
            return new BLL.Post().GetStyleTitle(item);
        }
    }
}