﻿/*
 * 
 * 数据库操作类
 * 
 */ 
using System;
using System.Collections;
using System.Data;

using taowaw.Common;

namespace taowaw.Data
{
    /// <summary>
    /// 数据库访问类(外部直接调用)
    /// 通用
    /// 2009-07-12
    /// </summary>
    public abstract class DbHelper
    {
        // 数据库连接字符串
        public static readonly string connstring = Utils.getConnectionStrings("SqliteConnString").Replace("@", System.Web.HttpContext.Current.Request.MapPath("/"));       
        // 哈希表用来存储缓存的参数
        private static Hashtable parmCache = Hashtable.Synchronized(new Hashtable());    
        // 查询次数
        private static int m_querycount = 0;

        #region 使用接口

        /// <summary>
        /// 返回数据库连接对象
        /// </summary>
        /// <returns></returns>
        public static IDbConnection dbconn()
        {
            return DatabaseProvider.GetInstance().CreateConn();
        }

        public static IDbConnection dbconn(string _connstring)
        {
            return DatabaseProvider.GetInstance().CreateConn(_connstring);
        }

        /// <summary>
        ///执行一个不需要返回值的SqlCommand命令，通过指定专用的连接字符串。
        /// 使用参数数组形式提供参数列表 
        /// </summary>
        /// <remarks>
        /// 使用示例：
        ///  int result = ExecuteNonQuery(connString, CommandType.StoredProcedure, "PublishOrders", new SqlParameter("@prodid", 24));
        /// </remarks>
        /// <param name="connectionString">一个有效的数据库连接字符串</param>
        /// <param name="commandType">SqlCommand命令类型 (存储过程， T-SQL语句， 等等。)</param>
        /// <param name="commandText">存储过程的名字或者 T-SQL 语句</param>
        /// <param name="commandParameters">以数组形式提供SqlCommand命令中用到的参数列表</param>
        /// <returns>返回一个数值表示此SqlCommand命令执行后影响的行数</returns>
        public static int ExecuteNonQuery(string connectionString, CommandType cmdType, string cmdText, params IDataParameter[] commandParameters)
        {

            IDbCommand cmd = DatabaseProvider.GetInstance().CreateCommand();

            using (IDbConnection connection = DatabaseProvider.GetInstance().CreateConn(connectionString))
            {
                //通过PrePareCommand方法将参数逐个加入到SqlCommand的参数集合中
                PrepareCommand(cmd,connection, null, cmdType, cmdText, commandParameters);
                int val = cmd.ExecuteNonQuery();

                //清空SqlCommand中的参数列表
                cmd.Parameters.Clear();
                cmd.Dispose();
                m_querycount++;

                return val;
            }
        }

        /// <summary>
        ///执行一条不返回结果的SqlCommand，通过一个已经存在的数据库连接 
        /// 使用参数数组提供参数
        /// </summary>
        /// <remarks>
        /// 使用示例：  
        ///  int result = ExecuteNonQuery(conn, CommandType.StoredProcedure, "PublishOrders", new SqlParameter("@prodid", 24));
        /// </remarks>
        /// <param name="conn">一个现有的数据库连接</param>
        /// <param name="commandType">SqlCommand命令类型 (存储过程， T-SQL语句， 等等。)</param>
        /// <param name="commandText">存储过程的名字或者 T-SQL 语句</param>
        /// <param name="isClose">是否关闭数据库连接</param>
        /// <param name="commandParameters">以数组形式提供SqlCommand命令中用到的参数列表</param>
        /// <returns>返回一个数值表示此SqlCommand命令执行后影响的行数</returns>
        public static int ExecuteNonQuery(IDbConnection connection, CommandType cmdType, string cmdText, bool isClose, params IDataParameter[] commandParameters)
        {

            IDbCommand cmd = DatabaseProvider.GetInstance().CreateCommand();

            PrepareCommand(cmd, connection, null, cmdType, cmdText, commandParameters);
            int val = cmd.ExecuteNonQuery();
            cmd.Parameters.Clear();
            cmd.Dispose();
            if (isClose)
            {
                connection.Close();
                connection.Dispose();
            }
            m_querycount++;

            return val;
        }


        public static int ExecuteNonQuery(IDbTransaction trans,IDbConnection connection, CommandType cmdType, string cmdText, bool isClose, params IDataParameter[] commandParameters)
        {

            IDbCommand cmd = DatabaseProvider.GetInstance().CreateCommand();

            PrepareCommand(cmd, connection, trans, cmdType, cmdText, commandParameters);
            int val = cmd.ExecuteNonQuery();
            cmd.Parameters.Clear();
            cmd.Dispose();
            if (isClose)
            {
                connection.Close();
                connection.Dispose();
            }
            m_querycount++;

            return val;
        }

        /// <summary>
        /// 执行一条不返回结果的SqlCommand，通过一个已经存在的数据库事物处理 
        /// 使用参数数组提供参数
        /// </summary>
        /// <remarks>
        /// 使用示例： 
        ///  int result = ExecuteNonQuery(trans, CommandType.StoredProcedure, "PublishOrders", new SqlParameter("@prodid", 24));
        /// </remarks>
        /// <param name="trans">一个存在的 sql 事物处理</param>
        /// <param name="commandType">SqlCommand命令类型 (存储过程， T-SQL语句， 等等。)</param>
        /// <param name="commandText">存储过程的名字或者 T-SQL 语句</param>
        /// <param name="commandParameters">以数组形式提供SqlCommand命令中用到的参数列表</param>
        /// <returns>返回一个数值表示此SqlCommand命令执行后影响的行数</returns>
        public static int ExecuteNonQuery(IDbTransaction trans, CommandType cmdType, string cmdText, params IDataParameter[] commandParameters)
        {
            IDbCommand cmd = DatabaseProvider.GetInstance().CreateCommand();
            PrepareCommand(cmd, trans.Connection, trans, cmdType, cmdText, commandParameters);
            int val = cmd.ExecuteNonQuery();
            cmd.Parameters.Clear();
            cmd.Dispose();
            m_querycount++;

            return val;
        }

        /// <summary>
        /// 执行一条返回结果集的SqlCommand命令，通过专用的连接字符串。
        /// 使用参数数组提供参数
        /// </summary>
        /// <remarks>
        /// 使用示例：  
        ///  SqlDataReader r = ExecuteReader(connString, CommandType.StoredProcedure, "PublishOrders", new SqlParameter("@prodid", 24));
        /// </remarks>
        /// <param name="connectionString">一个有效的数据库连接字符串</param>
        /// <param name="commandType">SqlCommand命令类型 (存储过程， T-SQL语句， 等等。)</param>
        /// <param name="commandText">存储过程的名字或者 T-SQL 语句</param>
        /// <param name="commandParameters">以数组形式提供SqlCommand命令中用到的参数列表</param>
        /// <returns>返回一个包含结果的SqlDataReader</returns>
        public static IDataReader ExecuteReader(string connectionString, CommandType cmdType, string cmdText, params IDataParameter[] commandParameters)
        {
            IDbCommand cmd = DatabaseProvider.GetInstance().CreateCommand();
            IDbConnection conn = DatabaseProvider.GetInstance().CreateConn(connectionString);

            // 在这里使用try/catch处理是因为如果方法出现异常，则SqlDataReader就不存在，
            //CommandBehavior.CloseConnection的语句就不会执行，触发的异常由catch捕获。
            //关闭数据库连接，并通过throw再次引发捕捉到的异常。
            try
            {
                PrepareCommand(cmd, conn, null, cmdType, cmdText, commandParameters);
                IDataReader idr = cmd.ExecuteReader(CommandBehavior.CloseConnection);
                cmd.Parameters.Clear();
                m_querycount++;

                return idr;
            }
            catch
            {
                throw;
            }
            finally
            {
                conn.Close();
                conn.Dispose();
            }
        }


        public static object ExecuteScalar(string connectionString, CommandType cmdType, string cmdText, params IDataParameter[] commandParameters)
        {
            IDbCommand cmd = DatabaseProvider.GetInstance().CreateCommand();
            using (IDbConnection connection=DatabaseProvider.GetInstance().CreateConn(connectionString))
            {
                PrepareCommand(cmd, connection, null, cmdType, cmdText, commandParameters);
                object val = cmd.ExecuteScalar();
                cmd.Parameters.Clear();
                cmd.Dispose();
                m_querycount++;

                return val;
            }
        }

        /// <summary>
        /// 执行一条返回第一条记录第一列的SqlCommand命令，通过已经存在的数据库连接。
        /// 使用参数数组提供参数
        /// </summary>
        /// <remarks>
        /// 使用示例： 
        ///  Object obj = ExecuteScalar(connString, CommandType.StoredProcedure, "PublishOrders", new SqlParameter("@prodid", 24));
        /// </remarks>
        /// <param name="conn">一个已经存在的数据库连接</param>
        /// <param name="commandType">SqlCommand命令类型 (存储过程， T-SQL语句， 等等。)</param>
        /// <param name="commandText">存储过程的名字或者 T-SQL 语句</param>
        /// <param name="isClose">是否关闭数据库连接</param>
        /// <param name="commandParameters">以数组形式提供SqlCommand命令中用到的参数列表</param>
        /// <returns>返回一个object类型的数据，可以通过 Convert.To{Type}方法转换类型</returns>
        public static object ExecuteScalar(IDbConnection connection, CommandType cmdType, string cmdText, bool isClose, params IDataParameter[] commandParameters)
        {

            IDbCommand cmd = DatabaseProvider.GetInstance().CreateCommand();
            Isconnection(connection);
            PrepareCommand(cmd, connection, null, cmdType, cmdText, commandParameters);
            object val = cmd.ExecuteScalar();
            cmd.Parameters.Clear();
            cmd.Dispose();
            if (isClose)
            {
                connection.Close();
                connection.Dispose();
            }
            m_querycount++;

            return val;
        }

        public static object ExecuteScalar(IDbTransaction trans, IDbConnection connection, CommandType cmdType, string cmdText, bool isClose, params IDataParameter[] commandParameters)
        {

            IDbCommand cmd = DatabaseProvider.GetInstance().CreateCommand();
            Isconnection(connection);
            PrepareCommand(cmd, connection, trans, cmdType, cmdText, commandParameters);
            object val = cmd.ExecuteScalar();
            cmd.Parameters.Clear();
            cmd.Dispose();
            if (isClose)
            {
                connection.Close();
                connection.Dispose();
            }
            m_querycount++;

            return val;
        }

        public static DataTable ExecuteDataTable(IDbTransaction trans,IDbConnection connection, CommandType cmdType, string cmdText, bool close, params IDataParameter[] commandParameters)
        {
            if (connection == null) throw new ArgumentNullException("connection");

            // 预处理
            IDbCommand cmd = DatabaseProvider.GetInstance().CreateCommand();
            Isconnection(connection);
            PrepareCommand(cmd, connection, trans, cmdType, cmdText, commandParameters);
            // 创建DbDataAdapter和DataSet
            DataSet ds=new DataSet();
            IDbDataAdapter da = DatabaseProvider.GetInstance().CreateDataAdapter();
            da.SelectCommand = cmd;
            da.Fill(ds);
            cmd.Dispose();
            if (close)
            {
                connection.Close();
                connection.Dispose();
            }
            m_querycount++;

            return ds.Tables[0];
        }

        public static DataTable ExecuteDataTable(IDbConnection connection, CommandType cmdType, string cmdText, bool close, params IDataParameter[] commandParameters)
        {
            if (connection == null) throw new ArgumentNullException("connection");

            // 预处理
            IDbCommand cmd = DatabaseProvider.GetInstance().CreateCommand();
            Isconnection(connection);
            PrepareCommand(cmd, connection, null, cmdType, cmdText, commandParameters);
            // 创建DbDataAdapter和DataSet
            DataSet ds = new DataSet();
            IDbDataAdapter da = DatabaseProvider.GetInstance().CreateDataAdapter();
            da.SelectCommand = cmd;
            da.Fill(ds);
            cmd.Dispose();
            if (close)
            {
                connection.Close();
                connection.Dispose();
            }
            m_querycount++;

            return ds.Tables[0];
        }

        /// <summary>
        /// 执行指定数据库连接对象的命令,指定存储过程参数,返回DataSet.
        /// </summary>
        /// <remarks>
        /// 示例:  
        ///  DataSet ds = ExecuteDataset(conn, CommandType.StoredProcedure, "GetOrders", new DbParameter("@prodid", 24));
        /// </remarks>
        /// <param name="connection">一个有效的数据库连接对象</param>
        /// <param name="commandType">命令类型 (存储过程,命令文本或其它)</param>
        /// <param name="commandText">存储过程名或SQL语句</param>
        /// <param name="isClose">是否关闭数据库连接</param>
        /// <param name="commandParameters">SqlParamter参数数组</param>
        /// <returns>返回一个包含结果集的DataSet</returns>
        public static DataSet ExecuteDataset(IDbConnection connection, CommandType cmdType, string cmdText, bool isClose, params IDataParameter[] commandParameters)
        {
            if (connection == null) throw new ArgumentNullException("connection");

            // 预处理
            IDbCommand cmd = DatabaseProvider.GetInstance().CreateCommand();
            Isconnection(connection);
            PrepareCommand(cmd, connection, null, cmdType, cmdText, commandParameters);
            // 创建DbDataAdapter和DataSet.
            DataSet ds = new DataSet();
            IDbDataAdapter da = DatabaseProvider.GetInstance().CreateDataAdapter();
            da.SelectCommand = cmd;
            da.Fill(ds);
            cmd.Parameters.Clear();
            cmd.Dispose();
            if (isClose)
            {
                connection.Close();
                connection.Dispose();
            }
            m_querycount++;

            return ds;
        }

        /// <summary>
        /// 缓存参数数组
        /// </summary>
        /// <param name="cacheKey">参数缓存的键值</param>
        /// <param name="cmdParms">被缓存的参数列表</param>
        public static void CacheParameters(string cacheKey, params IDataParameter[] commandParameters)
        {
            parmCache[cacheKey] = commandParameters;
        }

        /// <summary>
        /// 获取被缓存的参数
        /// </summary>
        /// <param name="cacheKey">用于查找参数的KEY值</param>
        /// <returns>返回缓存的参数数组</returns>
        public static IDataParameter[] GetCachedParameters(string cacheKey)
        {
            IDataParameter[] cachedParms = (IDataParameter[])parmCache[cacheKey];

            if (cachedParms == null)
                return null;

            //新建一个参数的克隆列表
            IDataParameter[] clonedParms = new IDataParameter[cachedParms.Length];

            //通过循环为克隆参数列表赋值
            for (int i = 0, j = cachedParms.Length; i < j; i++)
                //使用clone方法复制参数列表中的参数
                clonedParms[i] = (IDataParameter)((ICloneable)cachedParms[i]).Clone();

            return clonedParms;
        }

        #region PrepareCommand：参数
        private static void PrepareCommand(IDbCommand cmd, IDbConnection conn, IDbTransaction trans, CommandType cmdType, string cmdText, IDataParameter[] cmdParms)
        {

            //判断数据库连接状态
            if (conn.State != ConnectionState.Open)
                conn.Open();

            cmd.Connection = conn;
            cmd.CommandText = cmdText;

            //判断是否需要事物处理
            if (trans != null)
                cmd.Transaction = trans;

            cmd.CommandType = cmdType;

            if (cmdParms != null)
            {
                foreach (IDataParameter parm in cmdParms)
                    cmd.Parameters.Add(parm);
            }
        } 
        #endregion

        #region Isconnection：检查连接
        private static void Isconnection(IDbConnection connection)
        {
            if (connection.State != ConnectionState.Open)
            {
                connection.ConnectionString = connstring;
                connection.Open();
            }
        } 
        #endregion

        #region Close：关闭连接
        public static void Close(IDbConnection connection)
        {
            if (connection.State == ConnectionState.Open)
            {
                connection.Close();
                connection.Dispose();
            }
        } 
        #endregion

        #endregion
    }
}
