package wiki.xsx.core.pdf.component.line;

import org.apache.pdfbox.pdmodel.font.PDFont;
import wiki.xsx.core.pdf.doc.XEasyPdfDocument;
import wiki.xsx.core.pdf.page.XEasyPdfPage;
import wiki.xsx.core.pdf.util.XEasyPdfFontUtil;

import java.awt.*;
import java.io.IOException;

/**
 * 虚线分割线组件
 * @author xsx
 * @date 2020/3/4
 * @since 1.8
 * <p>
 * Copyright (c) 2020 xsx All Rights Reserved.
 * x-easypdf is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 * http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 * </p>
 */
public class XEasyPdfDottedSplitLine implements XEasyPdfLine {

    /**
     * 分割线参数
     */
    private final XEasyPdfLineParam param = new XEasyPdfLineParam();
    /**
     * 点线长度
     */
    private Float lineLength = 10F;
    /**
     * 点线间隔
     */
    private Float lineSpace = 10F;

    /**
     * 无参构造
     */
    public XEasyPdfDottedSplitLine() {}

    /**
     * 设置字体路径
     * @param fontPath 字体路径
     * @return 返回虚线分割线组件
     */
    @Override
    public XEasyPdfDottedSplitLine setFontPath(String fontPath) {
        this.param.setFontPath(fontPath);
        return this;
    }

    /**
     * 设置字体
     * @param font pdfBox字体
     * @return 返回虚线分割线组件
     */
    @Override
    public XEasyPdfDottedSplitLine setFont(PDFont font) {
        this.param.setFont(font);
        this.param.setFontPath("");
        return this;
    }

    /**
     * 设置边距（上下左右）
     * @param margin 边距
     * @return 返回虚线分割线组件
     */
    public XEasyPdfDottedSplitLine setMargin(float margin) {
        this.param.setMarginLeft(margin).setMarginRight(margin).setMarginTop(margin).setMarginBottom(margin);
        return this;
    }

    /**
     * 设置左边距
     * @param margin 边距
     * @return 返回虚线分割线组件
     */
    @Override
    public XEasyPdfDottedSplitLine setMarginLeft(float margin) {
        this.param.setMarginLeft(margin);
        return this;
    }

    /**
     * 设置右边距
     * @param margin 边距
     * @return 返回虚线分割线组件
     */
    @Override
    public XEasyPdfDottedSplitLine setMarginRight(float margin) {
        this.param.setMarginRight(margin);
        return this;
    }

    /**
     * 设置上边距
     * @param margin 边距
     * @return 返回虚线分割线组件
     */
    public XEasyPdfDottedSplitLine setMarginTop(float margin) {
        this.param.setMarginTop(margin);
        return this;
    }

    /**
     * 设置下边距
     * @param margin 边距
     * @return 返回虚线分割线组件
     */
    public XEasyPdfDottedSplitLine setMarginBottom(float margin) {
        this.param.setMarginBottom(margin);
        return this;
    }

    /**
     * 设置分割线宽度
     * @param lineWidth 分割线宽度
     * @return 返回虚线分割线组件
     */
    @Override
    public XEasyPdfDottedSplitLine setLineWidth(float lineWidth) {
        this.param.setLineWidth(lineWidth);
        return this;
    }

    /**
     * 设置分割线颜色
     * @param color 分割线颜色
     * @return 返回虚线分割线组件
     */
    @Override
    public XEasyPdfDottedSplitLine setColor(Color color) {
        this.param.setColor(color);
        return this;
    }

    /**
     * 设置分割线线型
     * @param lineCapStyle 分割线线型
     * @return 返回虚线分割线组件
     */
    @Override
    public XEasyPdfDottedSplitLine setLineCapStyle(XEasyPdfLineCapStyle lineCapStyle) {
        this.param.setStyle(lineCapStyle);
        return this;
    }

    /**
     * 设置点线长度
     * @param lineLength 点线长度
     * @return 返回虚线分割线组件
     */
    public XEasyPdfDottedSplitLine setLineLength(float lineLength) {
        this.lineLength = lineLength;
        return this;
    }

    /**
     * 设置点线间隔
     * @param lineSpace 点线间隔
     * @return 返回虚线分割线组件
     */
    public XEasyPdfDottedSplitLine setLineSpace(float lineSpace) {
        this.lineSpace = lineSpace;
        return this;
    }

    /**
     * 设置坐标
     * @param beginX X轴起始坐标
     * @param beginY Y轴起始坐标
     * @return 返回虚线分割线组件
     */
    @Override
    public XEasyPdfDottedSplitLine setPosition(float beginX, float beginY) {
        return this;
    }

    /**
     * 设置宽度
     * @param width 宽度
     * @return 返回虚线分割线组件
     */
    @Override
    public XEasyPdfDottedSplitLine setWidth(float width) {
        return this;
    }

    /**
     * 设置高度
     * @param height 高度
     * @return 返回虚线分割线组件
     */
    @Override
    public XEasyPdfDottedSplitLine setHeight(float height) {
        return this;
    }

    /**
     * 设置内容模式
     * @param mode 内容模式
     * @return 返回虚线分割线组件
     */
    @Override
    public XEasyPdfDottedSplitLine setContentMode(ContentMode mode) {
        this.param.setContentMode(mode);
        return this;
    }

    /**
     * 绘制
     * @param document pdf文档
     * @param page     pdf页面
     * @throws IOException IO异常
     */
    @Override
    public void draw(XEasyPdfDocument document, XEasyPdfPage page) throws IOException {
        // 初始化虚线分割线参数
        this.init(document, page);
        // 定义线条组件
        XEasyPdfBaseLine xpdfLine;
        // 初始化线条组件
        xpdfLine = new XEasyPdfBaseLine(this.param);
        // 执行画图
        xpdfLine.draw(document, page);
        // 计算点线数量，点线数量 = (pdfBox最新页面宽度 - 左边距 - 右边距) / (点线长度 + 点线间隔)
        int count = (int) Math.floor(
                (page.getLastPage().getMediaBox().getWidth() - this.param.getMarginLeft() - this.param.getMarginRight())
                /
                (this.lineLength + this.lineSpace)
        );
        // 循环点线数量进行画图
        for (int j = 1; j <= count; j++) {
            // 设置页面X轴起始坐标，起始坐标 = 结束坐标 + 点线间隔
            this.param.setBeginX(
                    this.param.getEndX() + this.lineSpace
            // 设置页面X轴结束坐标，结束坐标 = 起始坐标 + 点线长度
            ).setEndX(
                    this.param.getBeginX() + this.lineLength
            );
            // 重新初始化线条组件
            xpdfLine = new XEasyPdfBaseLine(this.param);
            // 执行画图
            xpdfLine.draw(document, page);
        }
        if (page.getParam().isAllowResetPosition()) {
            // 设置pdf页面Y轴起始坐标，起始坐标 = 起始坐标 - 线宽 / 2
            page.getParam().setPageY(this.param.getBeginY() - this.param.getLineWidth() / 2);
        }
        // 完成标记
        this.param.setDraw(true);
        // 字体路径不为空，说明该组件设置字体，则直接进行字体关联
        if (this.param.getFontPath()!=null&&this.param.getFontPath().length()>0) {
            // 关联字体
            this.param.getFont().subset();
            // 重置字体为null
            this.param.setFont(null);
        }
    }

    /**
     * 是否完成绘制
     * @return 返回布尔值，完成为true，未完成为false
     */
    @Override
    public boolean isDraw() {
        return this.param.isDraw();
    }

    /**
     * 初始化参数
     * @param document pdf文档
     * @param page pdf页面
     * @throws IOException IO异常
     */
    private void init(XEasyPdfDocument document, XEasyPdfPage page) throws IOException {
        // 分页检查
        this.param.checkPage(document, page);
        // 定义线宽
        float lineWidth = this.param.getLineWidth() / 2;
        // 设置X轴Y轴起始结束坐标
        this.param.setBeginX(
            // 左边距
            this.param.getMarginLeft()
        ).setBeginY(
            // 如果当前页面Y轴坐标为空，则起始坐标 = pdfBox最新页面高度 - 上边距 - 线宽，否则起始坐标 = 当前页面Y轴坐标 - 上边距 - 线宽
            page.getParam().getPageY()==null?
            // pdfBox最新页面高度 - 上边距 - 线宽
            page.getLastPage().getMediaBox().getHeight() - this.param.getMarginTop() - lineWidth:
            // 当前页面Y轴坐标 - 上边距 - 线宽
            page.getParam().getPageY() - this.param.getMarginTop() - lineWidth
        ).setEndX(
            // X轴起始坐标 + 点线长度
            this.param.getBeginX() + this.lineLength
        ).setEndY(
            // Y轴起始坐标
            this.param.getBeginY()
        );
        // 如果字体为空，则加载全局字体
        if (this.param.getFont()==null) {
            // 设置全局字体
            this.param.setFont(XEasyPdfFontUtil.loadFont(document, page, this.param.getFontPath()));
        }
    }
}
