package com.alibaba.fastsql.interpreter;

import com.alibaba.fastjson.serializer.SerializeConfig;
import com.alibaba.fastsql.interpreter.providers.FastjsonFieldValueProvider;
import com.alibaba.fastsql.interpreter.filters.*;
import com.alibaba.fastsql.sql.ast.SQLExpr;
import com.alibaba.fastsql.sql.ast.SQLObject;
import com.alibaba.fastsql.sql.ast.expr.*;
import com.alibaba.fastsql.sql.parser.*;

import java.util.List;

public class Filters {
    public static Filter createFilter(String sql) {
        SQLExprParser parser = SQLParserUtils.createExprParser(sql, null, SQLParserFeature.EnableSQLBinaryOpExprGroup);
        SQLExpr expr = parser.expr();

        if (parser.getLexer().token() != Token.EOF) {
            throw new ParserException("illegal sql expr : " + sql);
        }

        return createFilter(expr);
    }

    private static Filter createFilter(SQLExpr expr) {
        if (expr instanceof SQLBinaryOpExpr) {
            SQLBinaryOpExpr binaryOpExpr = (SQLBinaryOpExpr) expr;
            SQLBinaryOperator op = binaryOpExpr.getOperator();
            SQLExpr left = binaryOpExpr.getLeft();
            SQLExpr right = binaryOpExpr.getRight();

            if (left instanceof SQLIdentifierExpr) {
                String property = ((SQLIdentifierExpr) left).getName();
                FieldValueProvider fieldValueProvider = new FastjsonFieldValueProvider(property);

                if (right instanceof SQLIntegerExpr) {
                    Number number = ((SQLIntegerExpr) right).getNumber();
                    int val = (Integer) number;

                    switch (op) {
                        case Equality:
                            return new IntFilter(val, BinaryOp.Eq, fieldValueProvider);
                        case NotEqual:
                        case LessThanOrGreater:
                            return new IntFilter(val, BinaryOp.Ne, fieldValueProvider);
                        case LessThan:
                            return new IntFilter(val, BinaryOp.Lt, fieldValueProvider);
                        case LessThanOrEqual:
                            return new IntFilter(val, BinaryOp.Le, fieldValueProvider);
                        case GreaterThan:
                            return new IntFilter(val, BinaryOp.Gt, fieldValueProvider);
                        case GreaterThanOrEqual:
                            return new IntFilter(val, BinaryOp.Ge, fieldValueProvider);
                        default:
                            break;
                    }
                } else if (right instanceof SQLCharExpr) {
                    String val = ((SQLCharExpr) right).getText();

                    switch (op) {
                        case Equality:
                            return new StringFilter(val, BinaryOp.Eq, fieldValueProvider);
                        case NotEqual:
                        case LessThanOrGreater:
                            return new StringFilter(val, BinaryOp.Ne, fieldValueProvider);
                        case LessThan:
                            return new StringFilter(val, BinaryOp.Lt, fieldValueProvider);
                        case LessThanOrEqual:
                            return new StringFilter(val, BinaryOp.Le, fieldValueProvider);
                        case GreaterThan:
                            return new StringFilter(val, BinaryOp.Gt, fieldValueProvider);
                        case GreaterThanOrEqual:
                            return new StringFilter(val, BinaryOp.Ge, fieldValueProvider);
                        default:
                            break;
                    }
                }
            }

            if (op == SQLBinaryOperator.BooleanOr) {
                Filter leftFilter = createFilter(left);
                Filter rightFilter = createFilter(right);

                return new OrFilter(leftFilter, rightFilter);
            }

            if (op == SQLBinaryOperator.BooleanAnd) {
                Filter leftFilter = createFilter(left);
                Filter rightFilter = createFilter(right);

                return new AndFilter(leftFilter, rightFilter);
            }
        }

        if (expr instanceof SQLInListExpr) {
            SQLInListExpr inListExpr = (SQLInListExpr) expr;

            SQLExpr left = inListExpr.getExpr();
            List<SQLExpr> targetList = inListExpr.getTargetList();
            if (left instanceof SQLIdentifierExpr) {
                String property = ((SQLIdentifierExpr) left).getName();

                FieldValueProvider fieldValueProvider = new FastjsonFieldValueProvider(property);

                Object[] values = new Object[targetList.size()];
                for (int i = 0; i < targetList.size(); i++) {
                    SQLObject item = targetList.get(i);
                    if (item instanceof SQLValuableExpr) {
                        Object value = ((SQLValuableExpr) item).getValue();
                        values[i] = value;
                    }
                }

                int intCount = 0, strCount = 0, longCount = 0;
                for (int i = 0; i < values.length; i++) {
                    Object value = values[i];
                    if (value instanceof Integer) {
                        intCount++;
                    } else if (value instanceof String) {
                        strCount++;
                    } else if (value instanceof Long) {
                        longCount++;
                    }
                }

                if (intCount == values.length) {
                    int[] array = new int[values.length];
                    for (int i = 0; i < values.length; i++) {
                        array[i] = (Integer) values[i];
                    }

                    if (inListExpr.isNot()) {
                        return new ListIntFilter(fieldValueProvider, ListOp.NOT_IN, array);
                    } else {
                        return new ListIntFilter(fieldValueProvider, ListOp.IN, array);
                    }
                } else if (strCount == values.length) {
                    String[] array = new String[values.length];
                    for (int i = 0; i < values.length; i++) {
                        array[i] = (String) values[i];
                    }

                    if (inListExpr.isNot()) {
                        return new ListStringFilter(fieldValueProvider, ListOp.NOT_IN, array);
                    } else {
                        return new ListStringFilter(fieldValueProvider, ListOp.IN, array);
                    }
                }
            }
        }

        throw new UnsupportedOperationException();
    }

}
