/*
 * Copyright 1999-2017 Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.fastsql.sql.ast.expr;

import com.alibaba.fastsql.DbType;
import com.alibaba.fastsql.sql.ast.*;
import com.alibaba.fastsql.sql.ast.statement.SQLCharacterDataType;
import com.alibaba.fastsql.sql.visitor.SQLASTVisitor;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;

public class SQLDateExpr extends SQLExprImpl implements SQLLiteralExpr, SQLValuableExpr {
    public static final SQLDataType DEFAULT_DATA_TYPE = new SQLDataTypeImpl("date");

    private SQLExpr literal;

    public SQLDateExpr(){

    }

    public SQLDateExpr(String literal) {
        this.setLiteral(literal);
    }

    public SQLExpr getLiteral() {
        return literal;
    }

    public void setLiteral(String literal) {
        setLiteral(new SQLCharExpr(literal));
    }

    public void setLiteral(SQLExpr x) {
        if (x != null) {
            x.setParent(this);
        }
        this.literal = x;
    }

    public String getValue() {
        if (literal instanceof SQLCharExpr) {
            return ((SQLCharExpr) literal).getText();
        }
        return null;
    }

    public Date getDate() {
        if (literal instanceof SQLCharExpr) {
            String chars = ((SQLCharExpr) literal).getText();
            SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
            try {
                return format.parse(chars);
            } catch (ParseException e) {
                // skip
            }
        }
        return null;
    }

    public boolean addDay(int delta) {
        if (literal == null) {
            return false;
        }

        if (literal instanceof SQLCharExpr) {
            String chars = ((SQLCharExpr) literal).getText();
            SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
            try {
                Date date = format.parse(chars);
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(date);
                calendar.add(Calendar.DAY_OF_MONTH, delta);
                String result_chars = format.format(calendar.getTime());
                setLiteral(result_chars);
                return true;
            } catch (ParseException e) {
                // skip
            }
        }

        return false;
    }

    @Override
    protected void accept0(SQLASTVisitor visitor) {
        visitor.visit(this);
        visitor.endVisit(this);
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((literal == null) ? 0 : literal.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        SQLDateExpr other = (SQLDateExpr) obj;
        if (literal == null) {
            if (other.literal != null) {
                return false;
            }
        } else if (!literal.equals(other.literal)) {
            return false;
        }
        return true;
    }

    public SQLDateExpr clone() {
        SQLDateExpr x = new SQLDateExpr();

        if (this.literal != null) {
            x.setLiteral(literal.clone());
        }

        return x;
    }

    @Override
    public List<SQLObject> getChildren() {
        return Collections.emptyList();
    }

    public static long supportDbTypes = DbType.of(DbType.mysql, DbType.oracle, DbType.presto, DbType.postgresql, DbType.ads);

    public static boolean isSupport(DbType dbType) {
        return (dbType.mask & supportDbTypes) != 0;
    }
}
