/*
 * Copyright 1999-2017 Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.fastsql.sql.ast.expr;

import com.alibaba.fastsql.sql.SQLUtils;
import com.alibaba.fastsql.sql.ast.*;
import com.alibaba.fastsql.sql.dialect.oracle.visitor.OracleASTVisitor;
import com.alibaba.fastsql.sql.visitor.SQLASTVisitor;
import com.alibaba.fastsql.util.FnvHash;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

public class SQLMethodInvokeExpr extends SQLExprImpl implements SQLReplaceable, Serializable {

    private static final long     serialVersionUID = 1L;

    protected String              methodName;
    protected long                methodNameHashCode64;
    protected SQLExpr             owner;
    protected final List<SQLExpr> arguments       = new ArrayList<SQLExpr>();
    private SQLExpr               from;
    private SQLExpr               using;
    private SQLExpr               _for;
    private String                trimOption;
    private transient SQLDataType resolvedReturnDataType;

    public SQLMethodInvokeExpr(){

    }

    public SQLMethodInvokeExpr(String methodName){
        this.methodName = methodName;
    }

    public SQLMethodInvokeExpr(String methodName, long methodNameHashCode64){
        this.methodName = methodName;
        this.methodNameHashCode64 = methodNameHashCode64;
    }

    public SQLMethodInvokeExpr(String methodName, SQLExpr owner){

        this.methodName = methodName;
        setOwner(owner);
    }

    public SQLMethodInvokeExpr(String methodName, SQLExpr owner, SQLExpr... params){
        this.methodName = methodName;
        setOwner(owner);
        for (SQLExpr param : params) {
            this.addArgument(param);
        }
    }

    public long methodNameHashCode64() {
        if (methodNameHashCode64 == 0
                && methodName != null) {
            methodNameHashCode64 = FnvHash.hashCode64(methodName);
        }
        return methodNameHashCode64;
    }

    public String getMethodName() {
        return this.methodName;
    }

    public void setMethodName(String methodName) {
        this.methodName = methodName;
        this.methodNameHashCode64 = 0L;
    }

    /**
     * instead of getArguments
     * @deprecated
     */
    public List<SQLExpr> getParameters() {
        return this.arguments;
    }

    public List<SQLExpr> getArguments() {
        return this.arguments;
    }

    /**
     * deprecated, instead of addArgument
     * @deprecated
     */
    public void addParameter(SQLExpr param) {
        if (param != null) {
            param.setParent(this);
        }
        this.arguments.add(param);
    }

    public void addArgument(SQLExpr arg) {
        if (arg != null) {
            arg.setParent(this);
        }
        this.arguments.add(arg);
    }

    public SQLExpr getOwner() {
        return this.owner;
    }

    public void setOwner(SQLExpr owner) {
        if (owner != null) {
            owner.setParent(this);
        }
        this.owner = owner;
    }

    public SQLExpr getFrom() {
        return from;
    }

    public void setFrom(SQLExpr x) {
        if (x != null) {
            x.setParent(this);
        }
        this.from = x;
    }



    public void output(StringBuffer buf) {
        if (this.owner != null) {
            this.owner.output(buf);
            buf.append(".");
        }

        buf.append(this.methodName);
        buf.append("(");
        for (int i = 0, size = this.arguments.size(); i < size; ++i) {
            if (i != 0) {
                buf.append(", ");
            }

            this.arguments.get(i).output(buf);
        }
        buf.append(")");
    }

    @Override
    protected void accept0(SQLASTVisitor visitor) {
        if (visitor.visit(this)) {
            acceptChild(visitor, this.owner);
            acceptChild(visitor, this.arguments);
            acceptChild(visitor, this.from);
            acceptChild(visitor, this.using);
            acceptChild(visitor, this._for);
        }

        visitor.endVisit(this);
    }

    public List getChildren() {
        if (this.owner == null) {
            return this.arguments;
        }

        List<SQLObject> children = new ArrayList<SQLObject>();
        children.add(owner);
        children.addAll(this.arguments);
        return children;
    }

    protected void accept0(OracleASTVisitor visitor) {
        if (visitor.visit(this)) {
            acceptChild(visitor, this.owner);
            acceptChild(visitor, this.arguments);
            acceptChild(visitor, this.from);
            acceptChild(visitor, this.using);
            acceptChild(visitor, this._for);
        }

        visitor.endVisit(this);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        SQLMethodInvokeExpr that = (SQLMethodInvokeExpr) o;

        if (methodName != null ? !methodName.equals(that.methodName) : that.methodName != null) return false;
        if (owner != null ? !owner.equals(that.owner) : that.owner != null) return false;
        if (arguments != null ? !arguments.equals(that.arguments) : that.arguments != null) return false;
        return from != null ? from.equals(that.from) : that.from == null;

    }

    @Override
    public int hashCode() {
        int result = methodName != null ? methodName.hashCode() : 0;
        result = 31 * result + (owner != null ? owner.hashCode() : 0);
        result = 31 * result + (arguments != null ? arguments.hashCode() : 0);
        result = 31 * result + (from != null ? from.hashCode() : 0);
        return result;
    }

    public SQLMethodInvokeExpr clone() {
        SQLMethodInvokeExpr x = new SQLMethodInvokeExpr();

        x.methodName = methodName;

        if (owner != null) {
            x.setOwner(owner.clone());
        }

        for (SQLExpr arg : arguments) {
            x.addArgument(arg.clone());
        }

        if (from != null) {
            x.setFrom(from.clone());
        }

        if (using != null) {
            x.setUsing(using.clone());
        }

        if (trimOption != null) {
            x.setTrimOption(trimOption);
        }

        return x;
    }

    @Override
    public boolean replace(SQLExpr expr, SQLExpr target) {
        if (target == null) {
            return false;
        }

        for (int i = 0; i < arguments.size(); ++i) {
            if (arguments.get(i) == expr) {
                arguments.set(i, target);
                target.setParent(this);
                return true;
            }
        }

        if (from == expr) {
            setFrom(target);
            return true;
        }

        if (using == expr) {
            setUsing(target);
            return true;
        }

        if (_for == expr) {
            setFor(target);
            return true;
        }

        return false;
    }

    public boolean match(String owner, String function) {
        if (function == null) {
            return false;
        }

        if (!SQLUtils.nameEquals(function, methodName)) {
            return false;
        }

        if (owner == null && this.owner == null) {
            return true;
        }

        if (owner == null || this.owner == null) {
            return false;
        }

        if (this.owner instanceof SQLIdentifierExpr) {
            return SQLUtils.nameEquals(((SQLIdentifierExpr) this.owner).name, owner);
        }

        return false;
    }

    public SQLDataType computeDataType() {
        if (SQLUtils.nameEquals("to_date", methodName)
                || SQLUtils.nameEquals("add_months", methodName)) {
            return SQLDateExpr.DEFAULT_DATA_TYPE;
        }

        if (arguments.size() == 1) {
            if (SQLUtils.nameEquals("trunc", methodName)) {
                return arguments.get(0).computeDataType();
            }
        } else if (arguments.size() == 2) {
            SQLExpr param0 = arguments.get(0);
            SQLExpr param1 = arguments.get(1);
            if (SQLUtils.nameEquals("nvl", methodName) || SQLUtils.nameEquals("ifnull", methodName)) {
                SQLDataType dataType = param0.computeDataType();
                if (dataType != null) {
                    return dataType;
                }

                return param1.computeDataType();
            }
        }
        return null;
    }

    public SQLExpr getUsing() {
        return using;
    }

    public void setUsing(SQLExpr x) {
        if (x != null) {
            x.setParent(this);
        }
        this.using = x;
    }

    public SQLExpr getFor() {
        return _for;
    }

    public void setFor(SQLExpr x) {
        if (x != null) {
            x.setParent(this);
        }
        this._for = x;
    }

    public String getTrimOption() {
        return trimOption;
    }

    public void setTrimOption(String trimOption) {
        this.trimOption = trimOption;
    }

    public SQLDataType getResolvedReturnDataType() {
        return resolvedReturnDataType;
    }

    public void setResolvedReturnDataType(SQLDataType resolvedReturnDataType) {
        this.resolvedReturnDataType = resolvedReturnDataType;
    }
}
