/*
 * Copyright 1999-2017 Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.fastsql.sql.dialect.hive.parser;

import com.alibaba.fastsql.DbType;
import com.alibaba.fastsql.sql.ast.SQLExpr;
import com.alibaba.fastsql.sql.ast.SQLName;
import com.alibaba.fastsql.sql.ast.SQLStatement;
import com.alibaba.fastsql.sql.ast.expr.SQLQueryExpr;
import com.alibaba.fastsql.sql.ast.statement.*;
import com.alibaba.fastsql.sql.dialect.hive.ast.HiveInsert;
import com.alibaba.fastsql.sql.dialect.hive.ast.HiveMultiInsertStatement;
import com.alibaba.fastsql.sql.dialect.hive.stmt.HiveCreateFunctionStatement;
import com.alibaba.fastsql.sql.dialect.hive.stmt.HiveLoadDataStatement;
import com.alibaba.fastsql.sql.parser.*;
import com.alibaba.fastsql.util.FnvHash;

import java.util.List;

public class HiveStatementParser extends SQLStatementParser {
    {
        dbType = DbType.hive;
    }
    public HiveStatementParser(String sql) {
        super (new HiveExprParser(sql));
    }

    public HiveStatementParser(String sql, SQLParserFeature... features) {
        super (new HiveExprParser(sql, features));
    }

    public HiveStatementParser(Lexer lexer){
        super(new HiveExprParser(lexer));
    }

    public HiveSelectParser createSQLSelectParser() {
        return new HiveSelectParser(this.exprParser, selectListCache);
    }

    public SQLCreateTableParser getSQLCreateTableParser() {
        return new HiveCreateTableParser(this.exprParser);
    }

    public SQLStatement parseInsert() {
        if (lexer.token() == Token.FROM) {
            lexer.nextToken();

            HiveMultiInsertStatement stmt = new HiveMultiInsertStatement();

            if (lexer.token() == Token.IDENTIFIER) {
                SQLName tableName = this.exprParser.name();
                SQLExprTableSource from = new SQLExprTableSource(tableName);
                stmt.setFrom(from);

                if (lexer.token() == Token.IDENTIFIER) {
                    from.setAlias(lexer.stringVal());
                    lexer.nextToken();
                }
            } else {
                accept(Token.LPAREN);

                SQLSelectParser selectParser = createSQLSelectParser();
                SQLSelect select = selectParser.select();

                accept(Token.RPAREN);

                String alias = lexer.stringVal();
                accept(Token.IDENTIFIER);

                SQLSubqueryTableSource from = new SQLSubqueryTableSource(select, alias);

                stmt.setFrom(from);
            }

            for (;;) {
                HiveInsert insert = parseHiveInsert();
                stmt.addItem(insert);

                if (lexer.token() != Token.INSERT) {
                    break;
                }
            }

            return stmt;
        }

        return parseHiveInsertStmt();
    }

    public boolean parseStatementListDialect(List<SQLStatement> statementList) {
        if (lexer.token() == Token.FROM) {
            SQLStatement stmt = this.parseInsert();
            statementList.add(stmt);
            return true;
        }

        if (lexer.identifierEquals(FnvHash.Constants.LOAD)) {
            lexer.nextToken();
            acceptIdentifier("DATA");

            HiveLoadDataStatement stmt = new HiveLoadDataStatement();
            if (lexer.identifierEquals(FnvHash.Constants.LOCAL)) {
                lexer.nextToken();
                stmt.setLocal(true);
            }

            acceptIdentifier("INPATH");

            SQLExpr inpath = this.exprParser.expr();
            stmt.setInpath(inpath);

            if (lexer.token() == Token.OVERWRITE) {
                lexer.nextToken();
                stmt.setOverwrite(true);
            }

            accept(Token.INTO);
            accept(Token.TABLE);
            SQLExpr table = this.exprParser.expr();
            stmt.setInto(table);

            if (lexer.token() == Token.PARTITION) {
                lexer.nextToken();
                accept(Token.LPAREN);
                this.exprParser.exprList(stmt.getPartition(), stmt);
                accept(Token.RPAREN);
            }

            statementList.add(stmt);
            return true;
        }

        if (lexer.identifierEquals(FnvHash.Constants.SHOW)) {
            Lexer.SavePoint savePoint = this.lexer.mark();
            lexer.nextToken();

            if (lexer.identifierEquals(FnvHash.Constants.VIEWS)) {
                lexer.nextToken();

                SQLShowViewsStatement stmt = new SQLShowViewsStatement();
                if (lexer.token() == Token.IN) {
                    lexer.nextToken();
                    SQLName db = this.exprParser.name();
                    stmt.setDatabase(db);
                }
                if (lexer.token() == Token.LIKE) {
                    lexer.nextToken();
                    SQLExpr pattern = this.exprParser.expr();
                    stmt.setLike(pattern);
                }
                statementList.add(stmt);
                return true;
            }

            if (lexer.identifierEquals(FnvHash.Constants.TABLES)) {
                lexer.reset(savePoint);
                SQLStatement stmt = this.parseShowTables();
                statementList.add(stmt);
                return true;
            }

            throw new ParserException("TODO " + lexer.info());
        }

        return false;
    }

    public SQLCreateTableStatement parseCreateTable() {
        SQLCreateTableParser parser = new HiveCreateTableParser(this.exprParser);
        return parser.parseCreateTable();
    }

    public SQLStatement parseCreateFunction() {
        accept(Token.CREATE);
        accept(Token.FUNCTION);

        HiveCreateFunctionStatement stmt = new HiveCreateFunctionStatement();
        stmt.setDbType(dbType);

        SQLName name = this.exprParser.name();
        stmt.setName(name);

        if (lexer.token() == Token.AS) {
            lexer.nextToken();
            SQLExpr className = this.exprParser.expr();
            stmt.setClassName(className);
        }

        if (lexer.identifierEquals(FnvHash.Constants.LOCATION)) {
            lexer.nextToken();
            SQLExpr location = this.exprParser.primary();
            stmt.setLocationn(location);
        }

        if (lexer.identifierEquals(FnvHash.Constants.SYMBOL)) {
            lexer.nextToken();
            accept(Token.EQ);
            SQLExpr symbol = this.exprParser.primary();
            stmt.setSymbol(symbol);
        }

        return stmt;
    }
}
